/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Silvan Nellen
   co-programmer: Benjamin Knecht
*/


#include "playable.h"

#include "weapons/weapon_manager.h"
#include "event_handler.h"
#include "player.h"
#include "state.h"

#include "power_ups/weapon_power_up.h"
#include "power_ups/param_power_up.h"


Playable::Playable()
{
  this->setClassID(CL_PLAYABLE, "Playable");
  PRINTF(4)("PLAYABLE INIT\n");

  this->toList(OM_GROUP_01);
  this->weaponMan = new WeaponManager(this);

  // the reference to the Current Player is NULL, because we dont have one at the beginning.
  this->currentPlayer = NULL;

  this->setSynchronized(true);
}



Playable::~Playable()
{
  delete this->weaponMan;

  if (this->currentPlayer)
  {
    PRINTF(2)("There is Still a Player subscribed to this Playable (%s::%s)\n", this->getClassName(), this->getName());

  }
}


void Playable::addWeapon(Weapon* weapon, int configID, int slotID)
{
  this->weaponMan->addWeapon(weapon, configID, slotID);

  this->weaponConfigChanged();
}


void Playable::removeWeapon(Weapon* weapon)
{
  this->weaponMan->removeWeapon(weapon);

    this->weaponConfigChanged();
}


void Playable::nextWeaponConfig()
{
  this->weaponMan->nextWeaponConfig();
    this->weaponConfigChanged();
}


void Playable::previousWeaponConfig()
{
  this->weaponMan->previousWeaponConfig();
    this->weaponConfigChanged();
}


void Playable::weaponConfigChanged()
{
  if (this->currentPlayer != NULL)
    this->currentPlayer->weaponConfigChanged();
}


/**
 * @brief helps us colliding Playables
 */
void Playable::collidesWith(WorldEntity* entity, const Vector& location)
{
  if (entity->isA(CL_PROJECTILE))
    this->removeEnergy(entity->getEnergy());

  // EXTREME HACK
  if (this->getEnergy() == 0.0f)
    this->deactivateNode();
}

/**
 * subscribe to all events the controllable needs
 * @param player the player that shall controll this Playable
 */
bool Playable::subscribePlayer(Player* player)
{
  if (this->currentPlayer != NULL)
  {
    PRINTF(1)("Already registered Player:%s to this Playable (%s:%s)\n", this->currentPlayer->getName(), this->getClassName(), this->getName());
    return false;
  }
  else
  {
    this->currentPlayer = player;
    /*EventHandler*/
    EventHandler* evh = EventHandler::getInstance();
    std::list<int>::iterator ev;
    for (ev = this->events.begin(); ev != events.end(); ev++)
      evh->subscribe(player, ES_GAME, (*ev));
    this->enter();
    return true;
  }
}

/**
 * unsubscribe from all events the controllable needs
 * @param player the Player, that controlled this Ship.
 */
bool Playable::unsubscribePlayer(Player* player)
{
  if (this->currentPlayer != player)
  {
    PRINTF(1)("not the right Player to unregister from this Playable (%s:%s)\n", this->currentPlayer->getName(), this->getClassName(), this->getName());
    return false;
  }

  else
  {
    /*EventHandler*/
    EventHandler* evh = EventHandler::getInstance();
    std::list<int>::iterator ev;
    for (ev = this->events.begin(); ev != events.end(); ev++)
      evh->unsubscribe( ES_GAME, (*ev));

    this->leave();
    this->currentPlayer = NULL;
    return true;
  }
}

bool Playable::pickup(PowerUp* powerUp)
{
  if(powerUp->isA(CL_WEAPON_POWER_UP)) {
    Weapon* weapon = dynamic_cast<WeaponPowerUp*>(powerUp)->getWeapon();
    WeaponManager* manager = this->getWeaponManager();
    manager->addWeapon(weapon, 2, -1);
    return true;
  }
  else if(powerUp->isA(CL_PARAM_POWER_UP)) {
    ParamPowerUp* ppu = dynamic_cast<ParamPowerUp*>(powerUp);
    switch(ppu->getType()) {
      case POWERUP_PARAM_HEALTH:
        this->addEnergy(ppu->getValue());
        return true;
      case POWERUP_PARAM_MAX_HEALTH:
        this->setMaxEnergy(this->getMaxEnergy() + ppu->getValue());
        return true;
    }
  }
  return false;
}

/**
 * add an event to the event list of events this Playable can capture
 * @param eventType the Type of event to add
 */
void Playable::registerEvent(int eventType)
{
  this->events.push_back(eventType);

  if (this->currentPlayer != NULL)
    EventHandler::getInstance()->subscribe(this->currentPlayer, ES_GAME, eventType);
}

/**
 * remove an event to the event list this Playable can capture.
 * @param event the event to unregister.
 */
void Playable::unregisterEvent(int eventType)
{
  this->events.remove(eventType);

  if (this->currentPlayer != NULL)
    EventHandler::getInstance()->unsubscribe(ES_GAME, eventType);
}


void  Playable::attachCamera()
{
  State::getCamera()->setParentSoft(this);
  State::getCameraTarget()->setParentSoft(this);

}


void  Playable::detachCamera()
{
}
