/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "water.h"
#include "factory.h"
#include "load_param.h"

#include "grid.h"
#include "material.h"

#include "resource_manager.h"
#include "shader.h"

#include "skybox.h"

#include "network_game_manager.h"

using namespace std;

CREATE_FACTORY(Water, CL_WATER);


Water::Water(const TiXmlElement* root)
{
  this->setClassID(CL_WATER, "Water");
  this->toList(OM_ENVIRON);

  this->resX = this->resY = 10;
  this->sizeX = this->sizeY = 1.0f;
  this->height = 0.5f;
  this->grid = NULL;

  this->velocities = NULL;
  this->viscosity = 5;
  this->cohesion = .0000000001;

  if (root != NULL)
    this->loadParams(root);

  this->rebuildGrid();
  this->waterMaterial = new Material();
  this->waterShader = (Shader*)ResourceManager::getInstance()->load("shaders/water.vert", SHADER, RP_GAME, "shaders/water.frag");

  this->grid->height(this->grid->columns()/2,this->grid->rows()/2) = 100;
}

Water::~Water()
{
  delete this->grid;
  delete this->waterMaterial;
}

void Water::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "size", this, Water, setSize)
  .describe("the size of the WaterSurface")
  .defaultValues(2, 1.0f, 1.0f);

  LoadParam(root, "resolution", this, Water, setResolution)
  .describe("sets the resolution of the water surface")
  .defaultValues(2, 10, 10);

  LoadParam(root, "height", this, Water, setHeight)
  .describe("the height of the Waves")
  .defaultValues(1, 0.5f);
}

void Water::rebuildGrid()
{
  if (this->velocities != NULL)
  {
    assert (this->grid != NULL);
    for (unsigned int i = 0; i < this->grid->rows(); i++)
      delete[] this->velocities[i];
    delete[] this->velocities;
  }

  //   WE DO NOT NEED THIS AS IT IS DONE IN WORLDENTITY->setModel();
  //   if (this->grid != NULL)
  //     this->grid = NULL;

  this->grid = new Grid(this->sizeX, this->sizeY, this->resX, this->resY);
  this->velocities = new float*[this->resX];
  for (unsigned int i = 0; i < this->grid->rows(); i++)
  {
    this->velocities[i] = new float[this->resY];
    for (unsigned int j = 0; j < this->resY; j++)
      this->velocities[i][j] = 0.0;
  }
  this->setModel(this->grid, 0);
}

void Water::setResolution(unsigned int resX, unsigned int resY)
{
  this->resX = resX;
  this->resY = resY;
}

void Water::setSize(float sizeX, float sizeY)
{
  this->sizeX = sizeX;
  this->sizeY = sizeY;
}

void Water::setHeight(float height)
{
  this->height = height;
}


void Water::draw() const
{
  if (this->grid != NULL)
  {
    //SkyBox::enableCubeMap();
    WorldEntity::draw();
    glBindTexture(GL_TEXTURE_2D, 15);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
    glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
    glEnable(GL_TEXTURE_GEN_S);
    glEnable(GL_TEXTURE_GEN_T);

    glEnable(GL_BLEND);
    glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    // this->waterShader->activateShader();
    //  this->waterMaterial->select();
    //Shader::deactivateShader();

    SkyBox::disableCubeMap();
  }
}

void Water::tick(float dt)
{
  if (unlikely(this->velocities == NULL))
    return;
  /*
      THE OLD USELESS ALGORITHM
    phase += dt *.1;
    for (unsigned int i = 0; i < this->grid->rows(); i++)
    {
      for (unsigned int j = 0; j < this->grid->columns(); j++)
      {
        this->grid->height(i,j) = this->height*sin(((float)i/(float)this->grid->rows() *phase)+
            this->height*cos((float)j/(float)this->grid->columns()) * phase * 2.0);
      }
    }
    this->grid->rebuildNormals(this->height);*/


  unsigned int i, j;
  float u;

  // wave/advection
  // calc movement
  for(j = 1; j < this->grid->rows() - 1; j++)
  {
    for(i = 1; i < this->grid->columns() - 1; i++)
    {
      u =  this->grid->height(i+1,j)+ this->grid->height(i-1, j) +
           this->grid->height(i, j+1) + this->grid->height(i, j-1) -
           4 * this->grid->height(i, j);
      this->velocities[i][j] += dt * this->viscosity * this->viscosity * u / this->height;
      this->grid->height(i, j) += dt * this->velocities[i][j];
    }
  }
  // boundraries
  for (j = 0; j < this->grid->rows(); j++)
  {
    this->grid->height(0,j) = this->grid->height(1,j);
    this->grid->height(this->grid->rows()-1,j) = this->grid->height(this->grid->rows()-2, j);
  }
  for (i = 0; i < this->grid->rows(); i++)
  {
    this->grid->height(i,0) = this->grid->height(i,1);
    this->grid->height(i,this->grid->columns()-1) = this->grid->height(i, this->grid->columns()-2);
  }
  /*  // advect
    for(j = 1; j < this->grid->rows() - 1; j++) {
      for(i = 1; i < this->grid->columns() - 1; i++) {
        this->grid->height(i, j) += dt * this->velocities[i][j];
      }
    }*/
  // bound
  //   unsigned int w = this->grid->columns - 1;
  //   for(i = 0; i < this->grid->columns; i++) {
  //     _map[i][0].u[1] = _map[i][1  ].u[1];
  //     _map[i][w].u[1] = _map[i][w-1].u[1];
  //     _map[0][i].u[1] = _map[1  ][i].u[1];
  //     _map[w][i].u[1] = _map[w-1][i].u[1];
  //   }

  // diffusion
  for(j = 1; j < this->grid->rows() - 1; j++)
  {
    for(i = 1; i < this->grid->columns() - 1 ; i++)
    {
      u = this->grid->height(i+1, j) + this->grid->height(i-1, j) +
          this->grid->height(i, j+1) + this->grid->height(i, j-1) -
          4* this->grid->height(i, j);
      this->grid->height(i,j) += dt * this->cohesion * u / this->height;
    }
  }

  // calc normals
  //   float l[3];
  //   float m[3];
  //   for(j = 1; j < this->grid->rows() -1; j++) {
  //     for(i = 1; i < this->grid->columns() - 1; i++) {
  //       l[0] = this->grid->vertexG(i, j-1).x - this->grid->vertexG(i, j+1).x;
  //       l[1] = this->grid->vertexG(i, j-1).y - this->grid->vertexG(i, j+1).y;
  //       l[2] = this->grid->vertexG(i, j-1).z - this->grid->vertexG(i, j+1).z;
  //       m[0] = this->grid->vertexG(i-1,j).x - this->grid->vertexG(i+1, j).x;
  //       m[1] = this->grid->vertexG(i-1,j).y - this->grid->vertexG(i+1, j).y;
  //       m[2] = this->grid->vertexG(i-1,j).z - this->grid->vertexG(i+1, j).z;
  //       this->grid->normalG(i, j).x = l[1] * m[2] - l[2] * m[1];
  //       this->grid->normalG(i, j).y = l[2] * m[0] - l[0] * m[2];
  //       this->grid->normalG(i, j).z = l[0] * m[1] - l[1] * m[0];
  //     }
  //   }
  this->grid->rebuildNormals(this->height);
}


/**
 * Writes data from network containing information about the state
 * @param data pointer to data
 * @param length length of data
 * @param sender hostID of sender
 */
int Water::writeBytes( const byte * data, int length, int sender )
{
  setRequestedSync( false );
  setIsOutOfSync( false );

  SYNCHELP_READ_BEGIN();

  SYNCHELP_READ_FKT( Water::writeState );

  return SYNCHELP_READ_N;
}


/**
 * data copied in data will bee sent to another host
 * @param data pointer to data
 * @param maxLength max length of data
 * @return the number of bytes writen
 */
int Water::readBytes( byte * data, int maxLength, int * reciever )
{
  SYNCHELP_WRITE_BEGIN();

  if ( isOutOfSync() && !requestedSync() && this->getHostID()!=this->getOwner() )
  {
    (NetworkGameManager::getInstance())->sync( this->getUniqueID(), this->getOwner() );
    setRequestedSync( true );
  }

  int rec = this->getRequestSync();
  if ( rec > 0 )
  {
    *reciever = rec;
    SYNCHELP_WRITE_FKT( Water::readState );
  }

  *reciever = 0;
  return SYNCHELP_WRITE_N;
}



/**
 * data copied in data will bee sent to another host
 * @param data pointer to data
 * @param maxLength max length of data
 * @return the number of bytes writen
 */
int Water::readState( byte * data, int maxLength )
{
  SYNCHELP_WRITE_BEGIN();

  SYNCHELP_WRITE_FKT( WorldEntity::readState );

  // sync the size
  SYNCHELP_WRITE_FLOAT( this->sizeX );
  SYNCHELP_WRITE_FLOAT( this->sizeY );

  //sync resolution
  SYNCHELP_WRITE_INT( this->resX );
  SYNCHELP_WRITE_INT( this->resY );

  //sync the height
  SYNCHELP_WRITE_FLOAT( this->height );

  return SYNCHELP_WRITE_N;
}


/**
 * Writes data from network containing information about the state
 * @param data pointer to data
 * @param length length of data
 * @param sender hostID of sender
 */
int Water::writeState( const byte * data, int length, int sender )
{
  SYNCHELP_READ_BEGIN();

  SYNCHELP_READ_FKT( WorldEntity::writeState );

  float f1, f2;
  int i1, i2;

  //read the size
  SYNCHELP_READ_FLOAT( f1 );
  SYNCHELP_READ_FLOAT( f2 );
  this->sizeX = f1;
  this->sizeY = f2;
  PRINTF(0)("Setting Water to size: %f x %f\n", f1, f2);

  //read the resolution
  SYNCHELP_READ_INT( i1 );
  SYNCHELP_READ_INT( i2 );
  this->resX = i1;
  this->resY = i2;
  PRINTF(0)("Setting Water to resolution: %i x %i\n", i1, i2);

  //read the height
  SYNCHELP_READ_FLOAT( f1 );
  this->height = f1;

  this->rebuildGrid();

  return SYNCHELP_READ_N;
}


