/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "dot_particles.h"

#include "load_param.h"
#include "factory.h"
#include "material.h"
#include "state.h"
#include "shell_command.h"

#include "parser/tinyxml/tinyxml.h"
#include <algorithm>


CREATE_FACTORY(DotParticles, CL_SPRITE_PARTICLES);

SHELL_COMMAND(texture, DotParticles, setMaterialTexture)
    ->defaultValues(1, "maps/evil-flower.png");

using namespace std;

/**
 *  standard constructor
 * @param maxCount the Count of particles in the System
 * @param type The Type of the DotParticles
*/
DotParticles::DotParticles (unsigned int maxCount)
  : ParticleSystem(maxCount)
{
  this->init();
}

/**
 * @brief creates a Particle System out of a XML-element
 * @param root: the XML-element to load from
 */
DotParticles::DotParticles(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);
}

/**
 *  standard deconstructor
*/
DotParticles::~DotParticles()
{
  // deleting all the living Particles
  while (this->particles)
  {
    Particle* tmpDelPart = this->particles;
    this->particles = this->particles->next;
    delete tmpDelPart;
  }

  // deleting all the dead particles
  while (this->deadList)
  {
    Particle* tmpDelPart = this->deadList;
    this->deadList = this->deadList->next;
    delete tmpDelPart;
  }
}

/**
 * @brief initializes the DotParticles with default values
*/
void DotParticles::init()
{
  this->setClassID(CL_SPRITE_PARTICLES, "DotParticles");

  this->material.setDiffuseMap("maps/radial-trans-noise.png");
}


/**
 * loads Parameters from a TiXmlElement
 * @param root the XML-element to load from.
 */
void DotParticles::loadParams(const TiXmlElement* root)
{
  ParticleSystem::loadParams(root);

  LoadParam(root, "texture", this, DotParticles, setMaterialTexture);
}

/**
 * @brief sets the Texutre that is placed onto the particles
 * @param textureFile the Texture to load onto these DotParticles
 */
void DotParticles::setMaterialTexture(const char* textureFile)
{
  this->material.setDiffuseMap(textureFile);
}

/**
 * @brief draws all the Particles of this System
 *
 * The Cases in this Function all do the same:
 * Drawing all the particles with the appropriate Type.
 * This is just the fastest Way to do this, but will most likely be changed in the future.
 */
void DotParticles::draw() const
{
  glPushAttrib(GL_ENABLE_BIT);


  GLboolean checkLight = false;
  glGetBooleanv(GL_LIGHTING, &checkLight);
//  if (checkLight == GL_TRUE)
//    glDisable(GL_LIGHTING);
  glMatrixMode(GL_MODELVIEW);
  glDepthMask(GL_FALSE);

  material.select();
  glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);

  Particle* drawPart = particles;
  //glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  glBegin(GL_POINTS);
  while (likely(drawPart != NULL))
  {
    glColor4fv(drawPart->color);
    glVertex3f(drawPart->position.x, drawPart->position.y, drawPart->position.z);

    drawPart = drawPart->next;
  }
  glEnd();
  glDepthMask(GL_TRUE);
  glPopAttrib();
}
