/*!
 * @file resource_manager.h
  *  The Resource Manager checks if a file/resource is loaded.

    If a file/resource was already loaded the resourceManager will
    return a void pointer to the desired resource.
    Otherwise it will instruct the coresponding resource-loader to load,
    and receive a pointer to it.

    it is possible to compile the resource Manager without some modules by
    just adding the compile flag -D....
    (NO_MODEL)
    (NO_AUDIO)
    (NO_TEXT)
    (NO_TEXTURES)
    (NO_SHADERS)
*/

#ifndef _RESOURCE_MANAGER_H
#define _RESOURCE_MANAGER_H

#include "base_object.h"


#include <list>

//! An eumerator for different fileTypes the resourceManager supports
typedef enum ResourceType
{
#ifndef NO_MODEL
  OBJ,                  //!< loading .obj file
  PRIM,                 //!< loading primitive model
  MD2,                  //!< loading md2-file
#endif /* NO_MODEL */
#ifndef NO_TEXT
  TTF,                  //!< loading a TrueTypeFont
#endif /* NO_TEXT */
#ifndef NO_AUDIO
  WAV,                  //!< loading wav
  MP3,                  //!< loading mp3
  OGG,                  //!< loading ogg
#endif /* NO_AUDIO */
#ifndef NO_TEXTURES
  IMAGE,                //!< loading an image
#endif /* NO_TEXTURES */
#ifndef NO_SHADERS
  SHADER,               //!< openGL-shader program
#endif /* NO_SHADERS */
};

//! An enumerator for different UNLOAD-types.
/**
   RP_NO:        will be unloaded on request
   RP_LEVEL:     will be unloaded at the end of a Level
   RP_CAMPAIGN:  will be unloaded at the end of a Campaign
   RP_GAME:      will be unloaded at the end of the whole Game (when closing orxonox)
*/
typedef enum ResourcePriority
{
  RP_NO        =   0,
  RP_LEVEL     =   1,
  RP_CAMPAIGN  =   2,
  RP_GAME      =   4
};

//! A Struct that keeps track about A resource its name its Type, and so on
struct Resource
{
  BaseObject*       pointer;           //!< Pointer to the Resource.
  unsigned int      count;             //!< How many times this Resource has been loaded.

  char*             name;              //!< Name of the Resource.
  ResourceType      type;              //!< ResourceType of this Resource.
  ResourcePriority  prio;              //!< The Priority of this resource. (This will only be increased)

  // more specific
  union {
#ifndef NO_MODEL
  float             modelSize;         //!< the size of the model (OBJ/PRIM)
  char*             secFileName;       //!< a seconf fileName
#endif /* NO_MODEL */
#ifndef NO_TEXT
  unsigned int      ttfSize;           //!< the size of the ttf-font (TTF)
#endif /* NO_TEXT */
#ifndef NO_TEXTURES
  GLenum            texTarget;
#endif /* NO_TEXTURES */
  };
};


//! The ResourceManager is a class, that decides if a file/resource should be loaded
/**
 * If a file/resource was already loaded the resourceManager will
 * return a pointer to the desired resource.
 * Otherwise it will instruct the corresponding resource-loader to load,
 * and receive the pointer to it.
 *
 * It does it by looking, if a desired file has already been loaded.
 * There is also the possibility to check for some variables
 */
class ResourceManager : public BaseObject
{
 public:
  virtual ~ResourceManager();
  /** @returns a Pointer to the only object of this Class */
  inline static ResourceManager* getInstance() { if (!singletonRef) singletonRef = new ResourceManager();  return singletonRef; };

  bool setDataDir(const char* dataDir);
  /** @returns the Name of the data directory */
  inline const char* getDataDir() const { return this->dataDir; };


  bool tryDataDir(const char* dataDir);
  bool verifyDataDir(const char* fileInside);
  bool addImageDir(const char* imageDir);

  void cache(const char* fileName, ResourceType type, ResourcePriority prio = RP_NO,
             void* param1 = NULL, void* param2 = NULL, void* param3 = NULL);

  BaseObject* load(const char* fileName, ResourcePriority prio = RP_NO,
               void* param1 = NULL, void* param2 = NULL, void* param3 = NULL);
  BaseObject* load(const char* fileName, ResourceType type, ResourcePriority prio = RP_NO,
               void* param1 = NULL, void* param2 = NULL, void* param3 = NULL);
  bool unload(void* pointer, ResourcePriority prio = RP_NO);
  bool unload(Resource* resource, ResourcePriority = RP_NO);
  bool unloadAllByPriority(ResourcePriority prio);

  Resource* locateResourceByInfo(const char* fileName, ResourceType type, void* param1, void* param2, void* param3) const;
  Resource* locateResourceByPointer(const void* pointer) const;

  void debug() const;


  // utility functions for handling files in and around the data-directory
  static bool isDir(const char* directory);
  static bool isFile(const char* fileName);
  static bool touchFile(const char* fileName);
  static bool deleteFile(const char* fileName);
  static char* homeDirCheck(const char* fileName);
  static char* getFullName(const char* fileName);
  static bool isInDataDir(const char* fileName);

  static const char* ResourceTypeToChar(ResourceType type);


 private:
  ResourceManager();
  Resource* loadResource(const char* fileName, ResourceType type, ResourcePriority prio,
                     void* param1, void* param2, void* param3);

 private:
  static ResourceManager*  singletonRef;       //!< singleton Reference

  char*                    dataDir;            //!< The Data Directory, where all relevant Data is stored.
  std::list<Resource*>     resourceList;       //!< The List of Resources, that has already been loaded.
  std::list<char*>         imageDirs;          //!< A list of directories in which images are stored.

};

#endif /* _RESOURCE_MANAGER_H */
