

/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Benjamin Grauer
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_BASE

#include "base_object.h"

#include "load_param.h"
#include "compiler.h"
#include "class_list.h"

#include "synchronizeable.h"

using namespace std;


/**
 * @brief sets the name from a LoadXML-Element
 * @param root the element to load from
 */
BaseObject::BaseObject()
{
  this->className = "BaseObject";
  this->classID = CL_BASE_OBJECT;

  this->objectName = NULL;
  this->classList = NULL;
  this->xmlElem = NULL;

//  ClassList::addToClassList(this, this->classID, "BaseObject");
}

/**
 * @brief standard deconstructor
*/
BaseObject::~BaseObject ()
{
  ClassList::removeFromClassList(this);

  //  delete []this->className;
  if (this->objectName)
    delete[] this->objectName;
  if (this->xmlElem != NULL)
    delete this->xmlElem;
}

/**
 * @brief loads parameters
 * @param root the element to load from
 */
void BaseObject::loadParams(const TiXmlElement* root)
{
  // all loadParams should sometime arrive here.
  assert (root != NULL);

  if (this->xmlElem != NULL)
    delete this->xmlElem;
  this->xmlElem = root->Clone();
  // name setup
  LoadParam(root, "name", this, BaseObject, setName)
      .describe("the Name of the Object.");
}

/**
 * @brief sets the class identifiers
 * @param id a number for the class from class_id.h enumeration
 * @param className the class name
*/
void BaseObject::setClassID(ClassID classID, const char* className)
{
  //printf("%s(0x%.8X)->%s(0x%.8X)\n", this->className, this->classID, className, classID);
  assert (!(this->classID & classID & !CL_MASK_SUBSUPER_CLASS_IDA ));

  this->classID |= (long)classID;
  this->className = className;

  this->classList = ClassList::addToClassList(this, classID, this->classID, className);
}


/**
 * @brief set the name of the Object
 */
void BaseObject::setName (const char* objectName)
{
  if (this->objectName)
    delete[] this->objectName;
  if (objectName != NULL)
  {
    this->objectName = new char[strlen(objectName)+1];
    strcpy(this->objectName, objectName);
  }
  else
    this->objectName = NULL;
}


/**
 * @brief queries for the ClassID of the Leaf Class (the last made class of this type
 * @returns the ClassID of the Leaf Class (e.g. the ID of the Class)
 *
 * the returned ID can be used to generate new Objects of the same type through
 * Factory::fabricate(Object->getLeafClassID());
 */
ClassID BaseObject::getLeafClassID() const
{
  assert (this->classList != NULL);
  return this->classList->getLeafClassID();
}



/**
 * @brief checks if the class is a classID
 * @param classID the Identifier to check for
 * @returns true if it is, false otherwise
*/
bool BaseObject::isA (ClassID classID) const
{
  // if classID is a derivable object from a SUPERCLASS
  if (classID & CL_MASK_SUPER_CLASS)
  {
    if( likely(this->classID & classID))
      return true;
  }
  // if classID is a SubSuperClass, and
  else if (classID & CL_MASK_SUBSUPER_CLASS)
  {
    if (likely(((this->classID & CL_MASK_SUBSUPER_CLASS_IDA) == (this->classID & CL_MASK_SUBSUPER_CLASS_IDA)) &&
        this->classID & classID & CL_MASK_SUBSUPER_CLASS_IDB))
      return true;
  }
  // if classID is a LOWLEVEL-class
  else
  {
    if( likely((this->classID & CL_MASK_LOWLEVEL_CLASS) == classID))
      return true;
  }
  return false;
}



/**
 * @brief checks if the class is a classID
 * @param classID the Identifier to check for
 * @returns true if it is, false otherwise
 */
bool BaseObject::isA (const char* className) const
{
  ClassID classID = ClassList::StringToID(className);
  if (classID != CL_NULL)
    return this->isA(classID);
}


/**
 * @brief compares the ObjectName with an external name
 * @param objectName: the name to check.
 * @returns true on match, false otherwise.
 */
bool BaseObject::operator==(const char* objectName)
{
  if (likely(this->objectName != NULL && objectName != NULL))
    return (strcmp(this->objectName, objectName)) ? false : true;
}


/**
 * @brief displays everything this class is
 * @TODO REIMPLEMENT WITH SENSE.
 */
void BaseObject::whatIs() const
{

}

/**
 * Writes data from network containing information about the state
 * @param data pointer to data
 * @param length length of data
 * @param sender hostID of sender
 */
int BaseObject::writeState( const byte * data, int length, int sender )
{
  SYNCHELP_READ_BEGIN();

  if ( objectName )
  {
    delete[] objectName;
    objectName = NULL;
  }
  SYNCHELP_READ_STRINGM( this->objectName );
  if ( this->objectName && !strcmp(this->objectName, "") )
  {
    delete[] this->objectName;
    this->objectName = NULL;
  }

  return SYNCHELP_READ_N;
}

/**
 * data copied in data will bee sent to another host
 * @param data pointer to data
 * @param maxLength max length of data
 * @return the number of bytes writen
 */
int BaseObject::readState( byte * data, int maxLength )
{
  SYNCHELP_WRITE_BEGIN();

  //PRINTF(0)("objectname = %s\n", this->objectName);
  SYNCHELP_WRITE_STRING( this->objectName );

  return SYNCHELP_WRITE_N;
}
