/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "model_particles.h"

#include "load_param.h"
#include "factory.h"
#include "material.h"
#include "state.h"
#include "shell_command.h"

#include "parser/tinyxml/tinyxml.h"
#include <algorithm>


CREATE_FACTORY(ModelParticles, CL_MODEL_PARTICLES);

SHELL_COMMAND(texture, ModelParticles, setMaterialTexture)
->defaultValues(1, "maps/evil-flower.png");

using namespace std;

/**
 *  standard constructor
 * @param maxCount the Count of particles in the System
 * @param type The Type of the ModelParticles
*/
ModelParticles::ModelParticles (unsigned int maxCount)
    : ParticleSystem(maxCount)
{
  this->init();
}

/**
 * @brief creates a Particle System out of a XML-element
 * @param root: the XML-element to load from
 */
ModelParticles::ModelParticles(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);
}

/**
 *  standard deconstructor
*/
ModelParticles::~ModelParticles()
{
  // deleting all the living Particles
  while (this->particles)
  {
    Particle* tmpDelPart = this->particles;
    this->particles = this->particles->next;
    delete tmpDelPart;
  }

  // deleting all the dead particles
  while (this->deadList)
  {
    Particle* tmpDelPart = this->deadList;
    this->deadList = this->deadList->next;
    delete tmpDelPart;
  }
}

/**
 * @brief initializes the ModelParticles with default values
*/
void ModelParticles::init()
{
  this->setClassID(CL_MODEL_PARTICLES, "ModelParticles");

  this->material.setDiffuseMap("maps/radial-trans-noise.png");
}


/**
 * loads Parameters from a TiXmlElement
 * @param root the XML-element to load from.
 */
void ModelParticles::loadParams(const TiXmlElement* root)
{
  ParticleSystem::loadParams(root);

  LoadParam(root, "texture", this, ModelParticles, setMaterialTexture);
}

/**
 * @brief sets the Texutre that is placed onto the particles
 * @param textureFile the Texture to load onto these ModelParticles
 */
void ModelParticles::setMaterialTexture(const char* textureFile)
{
  this->material.setDiffuseMap(textureFile);
}

/**
 * @brief draws all the Particles of this System
 *
 * The Cases in this Function all do the same:
 * Drawing all the particles with the appropriate Type.
 * This is just the fastest Way to do this, but will most likely be changed in the future.
 */
void ModelParticles::draw() const
{
  glPushAttrib(GL_ENABLE_BIT);
  Particle* drawPart = particles;

  GLboolean checkLight = false;
  glGetBooleanv(GL_LIGHTING, &checkLight);
  if (checkLight == GL_TRUE)
    glDisable(GL_LIGHTING);
  glMatrixMode(GL_MODELVIEW);

  this->material.select();
  glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  if (likely(this->getModel() != NULL))
    while (likely(drawPart != NULL))
    {
      glPushMatrix();
      glMatrixMode(GL_MODELVIEW);
      /* move */
      glTranslatef(drawPart->position.x, drawPart->position.y, drawPart->position.z);
      /* scale */
      glScalef(drawPart->radius, drawPart->radius, drawPart->radius);
      /* rotate */
      Vector tmpRot = drawPart->orientation.getSpacialAxis();
      glRotatef (drawPart->orientation.getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );

      this->getModel()->draw();

      glPopMatrix();
      drawPart = drawPart->next;
    }
  else
    PRINTF(2)("no model loaded onto ParticleSystem-%s\n", this->getName());
  glPopAttrib();
}
