/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christian Meyer
   co-programmer: ...
*/

/*!
 * @file quaternion.h
 * A basic 3D quaternion math framework
 *
 * Contains classes to handle vectors, lines, rotations and planes
*/

#ifndef __QUATERNION_H_
#define __QUATERNION_H_

#include <math.h>
#include "compiler.h"
//! PI the circle-constant
#define PI 3.14159265359f
#include "vector.h"

//! Quaternion
/**
   Class to handle 3-dimensional rotation efficiently
*/
class Quaternion
{
 public:
  /** creates a Default quaternion (multiplicational identity Quaternion)*/
  inline Quaternion () { w = 1; v = Vector(0,0,0); }
  /** creates a Quaternion looking into the direction v @param v: the direction @param f: the value */
  inline Quaternion (const Vector& v, float f) { this->w = f; this->v = v; }
  Quaternion (float m[4][4]);
  /** turns a rotation along an axis into a Quaternion @param angle: the amount of radians to rotate @param axis: the axis to rotate around */
  inline Quaternion (float angle, const Vector& axis) { w = cos(angle/2.0); v = axis * sin(angle/2.0); }
  Quaternion (const Vector& dir, const Vector& up);
  Quaternion (float roll, float pitch, float yaw);

  /** @param q: the Quaternion to compare with this one. @returns true if the Quaternions are the same, false otherwise */
  inline bool operator== (const Quaternion& q) const { return (unlikely(this->v==q.v&&this->w==q.w))?true:false; };
  /** @param f: a real value @return a Quaternion containing the quotient */
  inline Quaternion operator/ (const float& f) const { return (unlikely(f==0.0)) ? Quaternion() : Quaternion(this->v/f, this->w/f); };
  /** @param f: the value to divide by @returns the quaternion devided by f (this /= f) */
  inline const Quaternion& operator/= (const float& f) {*this = *this / f; return *this;}
  /** @param f: a real value @return a Quaternion containing the product */
  inline Quaternion operator* (const float& f) const { return Quaternion(this->v*f, this->w*f); };
  /** @param f: the value to multiply by @returns the quaternion multiplied by f (this *= f) */
  inline const Quaternion& operator*= (const float& f) {*this = *this * f; return *this;}
  /** @param q: another Quaternion to rotate this by @return a quaternion that represents the first one rotated by the second one (WARUNING: this operation is not commutative! e.g. (A*B) != (B*A)) */
  Quaternion operator* (const Quaternion& q) const { return Quaternion(Vector(this->w*q.v.x + this->v.x*q.w + this->v.y*q.v.z - this->v.z*q.v.y,
                                                                         this->w*q.v.y + this->v.y*q.w + this->v.z*q.v.x - this->v.x*q.v.z,
                                                                         this->w*q.v.z + this->v.z*q.w + this->v.x*q.v.y - this->v.y*q.v.x),
                                                                         this->w*q.w - this->v.x*q.v.x - this->v.y*q.v.y - this->v.z*q.v.z); };
  /** @param q: the Quaternion to multiply by @returns the quaternion multiplied by q (this *= q) */
  inline const Quaternion& operator*= (const Quaternion& q) {*this = *this * q; return *this; };
  /** @param q the Quaternion by which to devide @returns the division from this by q (this / q) */
  inline Quaternion operator/ (const Quaternion& q) const { return *this * q.inverse(); };
  /** @param q the Quaternion by which to devide @returns the division from this by q (this /= q) */
  inline const Quaternion& operator/= (const Quaternion& q) { *this = *this * q.inverse(); return *this; };
  /** @param q the Quaternion to add to this @returns the quaternion added with q (this + q) */
  inline Quaternion operator+ (const Quaternion& q) const { return Quaternion(q.v + v, q.w + w); };
  /** @param q the Quaternion to add to this @returns the quaternion added with q (this += q) */
  inline const Quaternion& operator+= (const Quaternion& q) { this->v += q.v; this->w += q.w; return *this; };
  /** @param q the Quaternion to substrace from this @returns the quaternion substracted by q (this - q) */
  inline Quaternion operator- (const Quaternion& q) const { return Quaternion(q.v - v, q.w - w); }
  /** @param q the Quaternion to substrace from this @returns the quaternion substracted by q (this -= q) */
  inline const Quaternion& operator-= (const Quaternion& q) { this->v -= q.v; this->w -= q.w; return *this; };
  /** copy constructor @param q: the Quaternion to set this to. @returns the Quaternion q (or this) */
  inline Quaternion operator= (const Quaternion& q) {this->v = q.v; this->w = q.w; return *this;}
  /** conjugates this Quaternion @returns the conjugate */
  inline Quaternion conjugate () const { return Quaternion(Vector(-v.x, -v.y, -v.z), this->w); };
  /** @returns the norm of The Quaternion */
  inline float norm () const { return sqrt(w*w + v.x*v.x + v.y*v.y + v.z*v.z); };
  /** @returns the inverted Quaterntion of this */
  inline Quaternion inverse () const { return conjugate() / (w*w + v.x*v.x + v.y*v.y + v.z*v.z); };
  /** @returns the dot Product of a Quaternion */
  inline float dot (const Quaternion& q) const { return v.x*q.v.x + v.y*q.v.y + v.z*q.v.z + w*q.w; };
  /** @retuns the Distance between two Quaternions */
  inline float distance(const Quaternion& q) const { return 2*acos(fabsf(this->dot(q))); };
  /** @param v: the Vector  @return a new Vector representing v rotated by the Quaternion */
  inline Vector apply (const Vector& v) const { return (*this * Quaternion(v, 0) * conjugate()).v; };
  void matrix (float m[4][4]) const;
  /** @returns the normalized Quaternion (|this|) */
  inline Quaternion getNormalized() const { float n = this->norm(); return Quaternion(this->v/n, this->w/n); };
  /** normalizes the current Quaternion */
  inline void normalize() { float n = this->norm(); this->v /= n; this->w/=n; };

  /** @returns the rotational axis of this Quaternion */
  inline Vector getSpacialAxis() const { return this->v / sin(acos(w));/*sqrt(v.x*v.x + v.y*v.y + v.z+v.z);*/ };
  /** @returns the rotational angle of this Quaternion around getSpacialAxis()  !! IN DEGREE !! */
  inline float getSpacialAxisAngle() const { return 360.0 / M_PI * acos( this->w ); };

  static Quaternion quatSlerp(const Quaternion& from, const Quaternion& to, float t);

  void debug();
  void debug2();


 public:
  Vector    v;        //!< Imaginary Vector
  float     w;        //!< Real part of the number
};


#endif /* __QUATERNION_H_ */

