/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "spark_particles.h"

#include "load_param.h"
#include "factory.h"
#include "material.h"
#include "state.h"
#include "shell_command.h"

#include "parser/tinyxml/tinyxml.h"
#include <algorithm>


CREATE_FACTORY(SparkParticles, CL_SPARK_PARTICLES);

using namespace std;

/**
 *  standard constructor
 * @param maxCount the Count of particles in the System
 * @param type The Type of the SparkParticles
*/
SparkParticles::SparkParticles (unsigned int maxCount)
    : ParticleSystem(maxCount)
{
  this->init();
}

/**
 * @brief creates a Particle System out of a XML-element
 * @param root: the XML-element to load from
 */
SparkParticles::SparkParticles(const TiXmlElement* root)
{
  this->init();

  if (root != NULL)
    this->loadParams(root);
}

/**
 *  standard deconstructor
*/
SparkParticles::~SparkParticles()
{
  // deleting all the living Particles
  while (this->particles)
  {
    Particle* tmpDelPart = this->particles;
    this->particles = this->particles->next;
    delete tmpDelPart;
  }

  // deleting all the dead particles
  while (this->deadList)
  {
    Particle* tmpDelPart = this->deadList;
    this->deadList = this->deadList->next;
    delete tmpDelPart;
  }
}

/**
 * @brief initializes the SparkParticles with default values
*/
void SparkParticles::init()
{
  this->setClassID(CL_SPARK_PARTICLES, "SparkParticles");

  this->material = NULL;
}


/**
 * loads Parameters from a TiXmlElement
 * @param root the XML-element to load from.
 */
void SparkParticles::loadParams(const TiXmlElement* root)
{
  ParticleSystem::loadParams(root);
}

/**
 * @brief draws all the Particles of this System
 *
 * The Cases in this Function all do the same:
 * Drawing all the particles with the appropriate Type.
 * This is just the fastest Way to do this, but will most likely be changed in the future.
 */
void SparkParticles::draw() const
{
  glPushAttrib(GL_ENABLE_BIT);

  Particle* drawPart = particles;

  glDepthMask(GL_FALSE);

  glDisable(GL_LIGHTING);
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_BLEND);

  glBegin(GL_LINES);
  while (likely(drawPart != NULL))
  {
    glColor4fv(drawPart->color);
    glVertex3f(drawPart->position.x, drawPart->position.y, drawPart->position.z);
    glVertex3f(drawPart->position.x - drawPart->velocity.x * drawPart->radius,
               drawPart->position.y - drawPart->velocity.y * drawPart->radius,
               drawPart->position.z - drawPart->velocity.z * drawPart->radius);
    drawPart = drawPart->next;
  }
  glEnd();

  glDepthMask(GL_TRUE);
  glPopAttrib();
}
