/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "sprite_particles.h"

#include "load_param.h"
#include "factory.h"
#include "material.h"
#include "state.h"
#include "shell_command.h"

#include "parser/tinyxml/tinyxml.h"
#include <algorithm>


CREATE_FACTORY(SpriteParticles, CL_SPRITE_PARTICLES);

SHELL_COMMAND(texture, SpriteParticles, setMaterialTexture)
    ->defaultValues(1, "maps/evil-flower.png");

using namespace std;

/**
 *  standard constructor
 * @param maxCount the Count of particles in the System
 * @param type The Type of the SpriteParticles
*/
SpriteParticles::SpriteParticles (unsigned int maxCount)
  : ParticleSystem(maxCount)
{
  this->init();
}

/**
 * @brief creates a Particle System out of a XML-element
 * @param root: the XML-element to load from
 */
SpriteParticles::SpriteParticles(const TiXmlElement* root)
{
  this->init();

  this->loadParams(root);
}

/**
 *  standard deconstructor
*/
SpriteParticles::~SpriteParticles()
{
  // deleting all the living Particles
  while (this->particles)
  {
    Particle* tmpDelPart = this->particles;
    this->particles = this->particles->next;
    delete tmpDelPart;
  }

  // deleting all the dead particles
  while (this->deadList)
  {
    Particle* tmpDelPart = this->deadList;
    this->deadList = this->deadList->next;
    delete tmpDelPart;
  }
}

/**
 * @brief initializes the SpriteParticles with default values
*/
void SpriteParticles::init()
{
  this->setClassID(CL_SPRITE_PARTICLES, "SpriteParticles");

  this->material = NULL;
}


/**
 * loads Parameters from a TiXmlElement
 * @param root the XML-element to load from.
 */
void SpriteParticles::loadParams(const TiXmlElement* root)
{
  SpriteParticles::loadParams(root);

  LoadParam(root, "texture", this, SpriteParticles, setMaterialTexture);
}

// setting properties
/**
 * @brief sets the material to an external material
 * @param material: the material to set this material to.
 *
 * !! important if the extern material gets deleted it MUST be unregistered here or segfault !!
*/
void SpriteParticles::setMaterial(Material* material)
{
  this->material = *material;
}

void SpriteParticles::setMaterialTexture(const char* textureFile)
{
  this->material.setDiffuseMap(textureFile);
}

/**
 * @brief draws all the Particles of this System
 *
 * The Cases in this Function all do the same:
 * Drawing all the particles with the appropriate Type.
 * This is just the fastest Way to do this, but will most likely be changed in the future.
 */
void SpriteParticles::draw() const
{
  glPushAttrib(GL_ENABLE_BIT);

  Particle* drawPart = particles;

  glDisable(GL_LIGHTING);
  glMatrixMode(GL_MODELVIEW);
  glDepthMask(GL_FALSE);

  material.select();
  glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);

  //glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_ENV_MODE, GL_MODULATE);


  while (likely(drawPart != NULL))
  {
    glColor4fv(drawPart->color);
    //! @todo implement a faster code for the look-at Camera algorithm.

    const PNode* camera = State::getCamera();  //!< @todo MUST be different
    Vector cameraPos = camera->getAbsCoor();
    Vector cameraTargetPos = State::getCameraTarget()->getAbsCoor();
    Vector view = cameraTargetPos - cameraPos;
    Vector up = Vector(0, 1, 0);
    up = camera->getAbsDir().apply(up);
    Vector h = up.cross(view);
    Vector v = h.cross(view);
    h.normalize();
    v.normalize();
    v *= .5 * drawPart->radius;
    h *= .5 * drawPart->radius;

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2i(1, 1);
    glVertex3f(drawPart->position.x - h.x - v.x,
               drawPart->position.y - h.y - v.y,
               drawPart->position.z - h.z - v.z);
    glTexCoord2i(0, 1);
    glVertex3f(drawPart->position.x - h.x + v.x,
               drawPart->position.y - h.y + v.y,
               drawPart->position.z - h.z + v.z);
    glTexCoord2i(1, 0);
    glVertex3f(drawPart->position.x + h.x - v.x,
               drawPart->position.y + h.y - v.y,
               drawPart->position.z + h.z - v.z);
    glTexCoord2i(0, 0);
    glVertex3f(drawPart->position.x + h.x + v.x,
               drawPart->position.y + h.y + v.y,
               drawPart->position.z + h.z + v.z);

    glEnd();

    drawPart = drawPart->next;
  }
  glDepthMask(GL_TRUE);
  glPopAttrib();
}
