/*!
 * @file particle_emitter.h
  *  Definition of a ParticleEmitter
*/

#ifndef _PARTICLE_EMITTER_H
#define _PARTICLE_EMITTER_H

#include "p_node.h"

// FORWARD DECLARATION
class ParticleSystem;
class TiXmlElement;

// Default values
#define PARTICLE_EMITTER_DEFAULT_SIZE              1.0
#define PARTICLE_EMITTER_DEFAULT_EMISSION_RATE     50
#define PARTICLE_EMITTER_DEFAULT_TYPE              EMITTER_DOT
#define PARTICLE_EMITTER_DEFAULT_INHERIT_SPEED     0.0
#define PARTICLE_EMITTER_DEFAULT_SPREAD            M_PI

//! The form of the Emitter to emit from
typedef enum EMITTER_TYPE
{
  EMITTER_DOT     = 1,
  EMITTER_PLANE   = 2,
  EMITTER_SPHERE  = 4,
  EMITTER_CUBE    = 8
};

//! A class to handle an Emitter.
class ParticleEmitter : public PNode
{
  friend class ParticleSystem;
public:
  ParticleEmitter(const Vector& direction, float angle = .5,
                  float emissionRate = 1.0, float velocity = 1.0);
  ParticleEmitter(const TiXmlElement* root);
  virtual ~ParticleEmitter();

  void init();
  virtual void loadParams(const TiXmlElement* root);

  /* controlling the emitter: interface */
  void start();
  void stop();
  void tick(float dt, ParticleSystem* system);

  void setSystem(ParticleSystem* system);
  ParticleSystem* getSystem() const { return this->system; };

  /* controlling the behavour: these can be used as Animation interfaces */
  void setType(EMITTER_TYPE type);
  void setType(const char* type);
  void setSize(float emitterSize);
  void setEmissionRate(float emissionRate);
  void setInheritSpeed(float value);
  void setSpread(float angle, float randomAngle = 0.0);
  void setEmissionVelocity(float velocity, float randomVelocity = 0.0);
  void setEmissionMomentum(float momentum, float randomMomentum = 0.0);

  void setDirection(float x, float y, float z) { this->direction = Vector(x,y,z); }
  ; //!< todo this should be done via PNODE

  /** @returns the type of the emitter */
  inline EMITTER_TYPE getType() const { return this->type; };
  /** @returns the Type as a const char * */
  const char* getTypeC() const;
  /** @returns the Size of the emitter */
  inline float getSize() const { return this->emitterSize; };
  /** @returns the emissionRate */
  inline float getEmissionRate() const { return this->emissionRate; };
  /** @returns the inherit-speed-factor */
  inline float getInheritSpeed() const { return this->inheritSpeed; };
  /** @returns the SpreadAngle of the emitter */
  inline float getSpread() const { return this->angle; };
  /** @returns the EmissionVelocity of the emitter */
  inline float getEmissionVelocity() const { return this->velocity; };
  /** @returns the EmissionMomentum of this emitter */
  inline float getEmissionMomentum() const { return this->momentum; };

  void debug() const;


private:
  ParticleSystem* system;            //!< The ParticleSystem this Emitter Emits into.

  EMITTER_TYPE    type;              //!< The type of emitter this is.
  float           emitterSize;       //!< The size of the emitter (not for EMITTER_DOT).
  float           inheritSpeed;      //!< How much speed the particle inherits from the Emitters speed.
  Vector          direction;         //!< emition direction.
  float           angle;             //!< max angle from the direction of the emitter
  float           randomAngle;       //!< random emission angle (angle +- angleRandom is the emitted angle.
  float           emissionRate;      //!< amount of particles per seconds emitted by emitter.
  float           velocity;          //!< the initial speed of a Particles.
  float           randomVelocity;    //!< the random variation from the initial Speed.
  float           momentum;          //!< The Initial spped of the Rotation.
  float           momentumRandom;    //!< The random variation of the Momentum.

  float           saveTime;          //!< The time that was missing by the last Tick (otherwise there would be no emission when framefate is too big).
};

#endif /* _PARTICLE_EMITTER_H */
