/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christian Meyer
   co-programmer: ...
*/

/*!
 * @file vector.h
 * A basic 3D math framework
 *
 * Contains classes to handle vectors, lines, rotations and planes
*/

#ifndef __VECTOR_H_
#define __VECTOR_H_

#include <math.h>
#include "compiler.h"
//! PI the circle-constant
#define PI 3.14159265359f


//! this is a small and performant 3D vector
typedef float sVec3D[3];


//! small and performant 2D vector
typedef float sVec2D[2];



//! 3D Vector
/**
        Class to handle 3D Vectors
*/
class Vector {
 public:
  Vector (float x, float y, float z) : x(x), y(y), z(z) {}  //!< assignment constructor
  Vector () : x(0), y(0), z(0) {}
  ~Vector () {}

  /** @param v: the Vecor to compare with this one @returns true, if the Vecors are the same, false otherwise */
  inline bool operator== (const Vector& v) const { return (this->x==v.x&&this->y==v.y&&this->z==v.z)?true:false; };
  /** @param index The index of the "array" @returns the x/y/z coordinate */
  inline float operator[] (float index) const {if( index == 0) return this->x; if( index == 1) return this->y; if( index == 2) return this->z; }
  /** @param v The vector to add @returns the addition between two vectors (this + v) */
  inline Vector operator+ (const Vector& v) const { return Vector(x + v.x, y + v.y, z + v.z); };
  /** @param v The vector to add @returns the addition between two vectors (this + v) */
  inline Vector operator+ (const sVec3D& v) const { return Vector(x + v[0], y + v[1], z + v[2]); };
  /** @param v The vector to add  @returns the addition between two vectors (this += v) */
  inline const Vector& operator+= (const Vector& v) { this->x += v.x; this->y += v.y; this->z += v.z; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline const Vector& operator+= (const sVec3D& v) { this->x += v[0]; this->y += v[1]; this->z += v[2]; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline Vector operator- (const Vector& v) const { return Vector(x - v.x, y - v.y, z - v.z); }
  /** @param v The vector to substract  @returns the substraction between two vectors (this - v) */
  inline Vector operator- (const sVec3D& v) const { return Vector(x - v[0], y - v[1], z - v[2]); }
  /** @param v The vector to substract  @returns the substraction between two vectors (this -= v) */
  inline const Vector& operator-= (const Vector& v) { this->x -= v.x; this->y -= v.y; this->z -= v.z; return *this; };
  /** @param v The vector to substract  @returns the substraction between two vectors (this -= v) */
  inline const Vector& operator-= (const sVec3D& v) { this->x -= v[0]; this->y -= v[1]; this->z -= v[2]; return *this; };
  /** @param v the second vector  @returns The dotProduct between two vector (this (dot) v) */
  inline float operator* (const Vector& v) const { return x * v.x + y * v.y + z * v.z; };
  /** @todo strange */
  inline const Vector& operator*= (const Vector& v) { this->x *= v.x; this->y *= v.y; this->z *= v.z; return *this; };
  /** @param f a factor to multiply the vector with @returns the vector multiplied by f (this * f) */
  inline Vector operator* (float f) const { return Vector(x * f, y * f, z * f); };
  /** @param f a factor to multiply the vector with @returns the vector multiplied by f (this *= f) */
  inline const Vector& operator*= (float f) { this->x *= f; this->y *= f; this->z *= f; return *this; };
  /** @param f a factor to divide the vector with @returns the vector divided by f (this / f) */
  inline Vector operator/ (float f) const { return (unlikely(f == 0.0))?Vector(0,0,0):Vector(this->x / f, this->y / f, this->z / f); };
  /** @param f a factor to divide the vector with @returns the vector divided by f (this /= f) */
  inline const Vector& operator/= (float f) {if (unlikely(f == 0.0)) {this->x=0;this->y=0;this->z=0;} else {this->x /= f; this->y /= f; this->z /= f;} return *this; };
  /**  copy constructor @todo (i do not know it this is faster) @param v the vector to assign to this vector. @returns the vector v */
  inline const Vector& operator= (const Vector& v) { this->x = v.x; this->y = v.y; this->z = v.z; return *this; };
  /** copy constructor* @param v the sVec3D to assign to this vector. @returns the vector v */
  inline const Vector& operator= (const sVec3D& v) { this->x = v[0]; this->y = v[1]; this->z = v[2]; }
  /** @param v: the other vector \return the dot product of the vectors */
  float dot (const Vector& v) const { return x*v.x+y*v.y+z*v.z; };
  /** @param v: the corss-product partner @returns the cross-product between this and v (this (x) v) */
  inline Vector cross (const Vector& v) const { return Vector(y * v.z - z * v.y, z * v.x - x * v.z, x * v.y - y * v.x ); }
  /** scales the this vector with v* @param v the vector to scale this with */
  void scale(const Vector& v) {   x *= v.x;  y *= v.y; z *= v.z; };
  /** @returns the length of the vector */
  inline float len() const { return sqrt (x*x+y*y+z*z); }
  /** normalizes the vector */
  inline void normalize() { float l = len(); if( unlikely(l == 0.0))return; this->x=this->x/l; this->y=this->y/l; this->z=this->z/l; };
  Vector getNormalized() const;
  Vector abs();

  void debug() const;

 public:
  float    x;     //!< The x Coordinate of the Vector.
  float    y;     //!< The y Coordinate of the Vector.
  float    z;     //!< The z Coordinate of the Vector.
};

/**
 *  calculate the angle between two vectors in radiances
 * @param v1: a vector
 * @param v2: another vector
 * @return the angle between the vectors in radians
*/
inline float angleDeg (const Vector& v1, const Vector& v2) { return acos( v1 * v2 / (v1.len() * v2.len())); };
/**
 *  calculate the angle between two vectors in degrees
 * @param v1: a vector
 * @param v2: another vector
 * @return the angle between the vectors in degrees
*/
inline float angleRad (const Vector& v1, const Vector& v2) { return acos( v1 * v2 / (v1.len() * v2.len())) * 180/M_PI; };

/** an easy way to create a Random Vector @param sideLength the length of the Vector (x not sqrt(x^2...)) */
#define VECTOR_RAND(sideLength)  (Vector((float)rand()/RAND_MAX -.5, (float)rand()/RAND_MAX -.5, (float)rand()/RAND_MAX -.5) * sideLength)


//! 3D rotation (OBSOLETE)
/**
  Class to handle 3-dimensional rotations.
  Can create a rotation from several inputs, currently stores rotation using a 3x3 Matrix
*/
class Rotation {
  public:

  float m[9]; //!< 3x3 Rotation Matrix

  Rotation ( const Vector& v);
  Rotation ( const Vector& axis, float angle);
  Rotation ( float pitch, float yaw, float roll);
  Rotation ();
  ~Rotation () {}

  Rotation operator* (const Rotation& r);

  void glmatrix (float* buffer);
};

//!< Apply a rotation to a vector
Vector rotateVector( const Vector& v, const Rotation& r);

//! 3D line
/**
  Class to store Lines in 3-dimensional space

  Supports line-to-line distance measurements and rotation
*/
class Line
{
  public:

  Vector r;   //!< Offset
  Vector a;   //!< Direction

  Line ( Vector r, Vector a) : r(r), a(a) {}  //!< assignment constructor
  Line () : r(Vector(0,0,0)), a(Vector (1,1,1)) {}
  ~Line () {}

  float distance (const Line& l) const;
  float distancePoint (const Vector& v) const;
  float distancePoint (const sVec3D& v) const;
  Vector* footpoints (const Line& l) const;
  float len () const;

  void rotate(const Rotation& rot);
};

//! 3D plane
/**
  Class to handle planes in 3-dimensional space

  Critical for polygon-based collision detection
*/
class Plane
{
  public:

  Vector n;   //!< Normal vector
  float k;    //!< Offset constant

  Plane (const Vector& a, const Vector& b, const Vector& c);
  Plane (const Vector& norm, const Vector& p);
  Plane (const Vector& norm, const sVec3D& p);
  Plane (const Vector& n, float k) : n(n), k(k) {} //!< assignment constructor
  Plane () : n(Vector(1,1,1)), k(0) {}
  ~Plane () {}

  Vector intersectLine (const Line& l) const;
  float distancePoint (const Vector& p) const;
  float distancePoint (const sVec3D& p) const;
  float locatePoint (const Vector& p) const;
};



//! A class that represents a rectangle, this is needed for SpatialSeparation
class Rectangle
{

  public:
    Rectangle() { this->center = Vector(); }
    Rectangle(const Vector &center, float len) { this->center = Vector(center.x, center.y, center.z); this->axis[0] = len; this->axis[1] = len; }
    virtual ~Rectangle() {}

    /** \brief sets the center of the rectangle to a defined vector @param center the new center */
   inline void setCenter(const Vector &center) { this->center = center;}
    /** \brief sets the center of the rectangle to a defined vector @param x coord of the center @param y coord of the center @param z coord of the center */
   inline void setCenter(float x, float y, float z) { this->center.x = x; this->center.y = y; this->center.z = z; }
   /** \brief returns the center of the rectangle to a defined vector @returns center the new center */
   inline const Vector& getCenter() const { return this->center; }

   /** \brief sets both axis of the rectangle to a defined vector @param unityLength the new center */
   inline void setAxis(float unityLength) { this->axis[0] = unityLength; this->axis[1] = unityLength; }
   /** \brief sets both axis of the rectangle to a defined vector @param v1 the length of the x axis @param v2 the length of the z axis*/
   inline void setAxis(float v1, float v2) { this->axis[0] = v1; this->axis[1] = v2; }
   /** \brief gets one axis length of the rectangle  @returns the length of the axis 0 */
   inline float getAxis() { return this-> axis[0]; }

  private:
    Vector          center;
    float           axis[2];
};


#endif /* __VECTOR_H_ */

