/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file load_param.h
 * A Class and macro-functions, that makes our lives easy to load-in parameters
 */

#ifndef _LOAD_PARAM_H
#define _LOAD_PARAM_H

#include "base_object.h"

#include "executor/executor.h"
#include "executor/executor_specials.h"

#include "helper_functions.h"

// Forward Declaration //
template<class T> class tList;
class LoadClassDescription;
class LoadParamDescription;
class MultiType;


/**
 * Loads a Parameter from ROOT named PARAMETER_NAME
 * onto OBJECT of CLASS, trough the FUNCTION
 * @param ROOT the TiXmlElement to load the Parameter from
 * @param PARAMETER_NAME the Name of the Parameter to load
 * @param OBJECT The BaseObject to load the new setting to.
 * @param CLASS What Class the BaseObejct is of (this is for identifying the Functuon)
 * @param FUNCTION The function of Class to Load (if you want to call &CLASS::FUNCTION write FUNCTION here).
 */
#define LoadParam(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         CLoadParam(ROOT, PARAMETER_NAME, OBJECT, ExecutorObjective<CLASS>(&CLASS::FUNCTION), false)

#define LoadParam_CYCLE(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         CLoadParam(ROOT, PARAMETER_NAME, OBJECT, ExecutorObjective<CLASS>(&CLASS::FUNCTION), true)

#define LoadParamXML(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         CLoadParam(ROOT, PARAMETER_NAME, OBJECT, ExecutorXML<CLASS>(&CLASS::FUNCTION, ROOT, PARAMETER_NAME), false)

#define LoadParamXML_CYCLE(ROOT, PARAMETER_NAME, OBJECT, CLASS, FUNCTION) \
         CLoadParam(ROOT, PARAMETER_NAME, OBJECT, ExecutorXML<CLASS>(&CLASS::FUNCTION, ROOT, PARAMETER_NAME), true)


/**
 * this Starts a Cycle in the Loading Process
 * be aware, that in the cycle the first parameter of load_param should because
 * called element, and that you must say true at the Fith parameter, or it will fail
 * also you will have to close the Cycle again with LOAD_PARAM_END_CYCLE
 *
 * @param ROOT The root XLM-element to search element under.
 * @param ELEMENT the element to search
 */
#define LOAD_PARAM_START_CYCLE(ROOT, ELEMENT) \
  const TiXmlElement* ELEMENT; \
  ELEMENT= ROOT->FirstChildElement(); \
  while( ELEMENT != NULL) \
{
/**
   * closes a LoadParam Loop
   * @see LOAD_PARAM_START_CYCLE
   * @param ELEMENT the Element to step through.
 */
#define LOAD_PARAM_END_CYCLE(ELEMENT) \
  ELEMENT = ELEMENT->NextSiblingElement(); \
}

/**************************
**** REAL DECLARATIONS ****
**************************/
//! abstract Base class for a Loadable parameter
class CLoadParam : public BaseObject
{
  public:
    CLoadParam(const TiXmlElement* root, const char* paramName, BaseObject* object, const Executor& executor, bool inLoadCycle = false);
    ~CLoadParam();

    CLoadParam& describe(const char* descriptionText);
    CLoadParam& defaultValues(unsigned int count, ...);
    CLoadParam& attribute(const char* attributeName, const Executor& executor);


  private:
    bool                     inLoadCycle;
    Executor*                executor;
    BaseObject*              object;
    const char*              paramName;

    LoadClassDescription*    classDesc;            //!< The LoadClassDescription of this CLoadParameter
    LoadParamDescription*    paramDesc;            //!< The LoadParameterDescription of this LoadParameter
    const TiXmlElement*      loadElem;             //!< The Element to load.
    const char*              loadString;           //!< The string loaded by this LoadParam
    const void*              pointerToParam;       //!< A Pointer to a Parameter.

    MultiType*               defaultValue;
};

// helper function

const char* grabParameter(const TiXmlElement* root, const char* parameterName);
const TiXmlElement* grabParameterElement(const TiXmlElement* root, const char* parameterName);

#endif /* _LOAD_PARAM_H */
