/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "water.h"
#include "factory.h"
#include "load_param.h"

#include "grid.h"
#include "material.h"

#include "resource_manager.h"
#include "shader.h"

#include "skybox.h"


using namespace std;

CREATE_FACTORY(Water, CL_WATER);


Water::Water(const TiXmlElement* root)
{
  this->setClassID(CL_WATER, "Water");
  this->toList(OM_ENVIRON);

  this->resX = this->resY = 10;
  this->sizeX = this->sizeY = 1.0f;
  this->height = 0.5f;
  this->grid = NULL;

  this->velocities = NULL;
  this->viscosity = 5;
  this->cohesion = .0000000001;

  if (root != NULL)
    this->loadParams(root);

  this->rebuildGrid();
  this->waterMaterial = new Material();
  this->waterShader = (Shader*)ResourceManager::getInstance()->load("shaders/water.vert", SHADER, RP_GAME, (void*)"shaders/water.frag");

  this->grid->height(this->grid->columns()/2,this->grid->rows()/2) = 100;
}

Water::~Water()
{
  delete this->grid;
  delete this->waterMaterial;
}

void Water::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "size", this, Water, setSize)
      .describe("the size of the WaterSurface")
      .defaultValues(2, 1.0f, 1.0f);

  LoadParam(root, "resolution", this, Water, setResolution)
      .describe("sets the resolution of the water surface")
      .defaultValues(2, 10, 10);

  LoadParam(root, "height", this, Water, setHeight)
      .describe("the height of the Waves")
      .defaultValues(1, 0.5f);
}

void Water::rebuildGrid()
{
  if (this->velocities != NULL)
  {
    assert (this->grid != NULL);
    for (unsigned int i = 0; i < this->grid->rows(); i++)
      delete[] this->velocities[i];
    delete[] this->velocities;
  }
  if (this->grid != NULL)
    delete this->grid;
  this->grid = new Grid(this->sizeX, this->sizeY, this->resX, this->resY);
  this->velocities = new float*[this->resX];
  for (unsigned int i = 0; i < this->grid->rows(); i++)
    this->velocities[i] = new float[this->resY];

  this->setModel(this->grid, 0);
}

void Water::setResolution(unsigned int resX, unsigned int resY)
{
  this->resX = resX;
  this->resY = resY;
}

void Water::setSize(float sizeX, float sizeY)
{
  this->sizeX = sizeX;
  this->sizeY = sizeY;
}

void Water::setHeight(float height)
{
  this->height = height;
}


void Water::draw() const
{
  SkyBox::enableCubeMap();

 // this->waterShader->activateShader();
//  this->waterMaterial->select();
  WorldEntity::draw();
  //Shader::deactivateShader();

  SkyBox::disableCubeMap();
}

void Water::tick(float dt)
{
/*
    THE OLD USELESS ALGORITHM
  phase += dt *.1;
  for (unsigned int i = 0; i < this->grid->rows(); i++)
  {
    for (unsigned int j = 0; j < this->grid->columns(); j++)
    {
      this->grid->height(i,j) = this->height*sin(((float)i/(float)this->grid->rows() *phase)+
          this->height*cos((float)j/(float)this->grid->columns()) * phase * 2.0);
    }
  }
  this->grid->rebuildNormals(this->height);*/


  unsigned int i, j;
  float u;

  // wave/advection
  // calc movement
  for(j = 1; j < this->grid->rows() - 1; j++) {
    for(i = 1; i < this->grid->columns() - 1; i++) {
      u =  this->grid->height(i+1,j)+ this->grid->height(i-1, j) +
          this->grid->height(i, j+1) + this->grid->height(i, j-1) -
          4 * this->grid->height(i, j);
      this->velocities[i][j] += dt * this->viscosity * this->viscosity * u / this->height;
      this->grid->height(i, j) += dt * this->velocities[i][j];
    }
  }
/*  // advect
  for(j = 1; j < this->grid->rows() - 1; j++) {
    for(i = 1; i < this->grid->columns() - 1; i++) {
      this->grid->height(i, j) += dt * this->velocities[i][j];
    }
  }*/
  // bound
//   unsigned int w = this->grid->columns - 1;
//   for(i = 0; i < this->grid->columns; i++) {
//     _map[i][0].u[1] = _map[i][1  ].u[1];
//     _map[i][w].u[1] = _map[i][w-1].u[1];
//     _map[0][i].u[1] = _map[1  ][i].u[1];
//     _map[w][i].u[1] = _map[w-1][i].u[1];
//   }

  // diffusion
  for(j = 1; j < this->grid->rows() - 1; j++) {
    for(i = 1; i < this->grid->columns() - 1 ; i++) {
      u = this->grid->height(i+1, j) + this->grid->height(i-1, j) +
          this->grid->height(i, j+1) + this->grid->height(i, j-1) -
          4* this->grid->height(i, j);
      this->grid->height(i,j) += dt * this->cohesion * u / this->height;
    }
  }

  // calc normals
//   float l[3];
//   float m[3];
//   for(j = 1; j < this->grid->rows() -1; j++) {
//     for(i = 1; i < this->grid->columns() - 1; i++) {
//       l[0] = this->grid->vertexG(i, j-1).x - this->grid->vertexG(i, j+1).x;
//       l[1] = this->grid->vertexG(i, j-1).y - this->grid->vertexG(i, j+1).y;
//       l[2] = this->grid->vertexG(i, j-1).z - this->grid->vertexG(i, j+1).z;
//       m[0] = this->grid->vertexG(i-1,j).x - this->grid->vertexG(i+1, j).x;
//       m[1] = this->grid->vertexG(i-1,j).y - this->grid->vertexG(i+1, j).y;
//       m[2] = this->grid->vertexG(i-1,j).z - this->grid->vertexG(i+1, j).z;
//       this->grid->normalG(i, j).x = l[1] * m[2] - l[2] * m[1];
//       this->grid->normalG(i, j).y = l[2] * m[0] - l[0] * m[2];
//       this->grid->normalG(i, j).z = l[0] * m[1] - l[1] * m[0];
//     }
//   }
  this->grid->rebuildNormals(this->height);
}
