/*!
 * @file particle_system.h

*/

#ifndef _PARTICLE_SYSTEM_H
#define _PARTICLE_SYSTEM_H

#include "world_entity.h"
#include "physics_interface.h"

#include "glincl.h"
#include "vector.h"

#include "quick_animation.h"

// Forward Declaration
class TiXmlElement;

#define PARTICLE_DOT_MASK              0x000001     //!< A Mask if the Particles should be displayed as DOTs
#define PARTICLE_SPARK_MASK            0x000010     //!< A Mask if the Particles should be displayed as SPARKs
#define PARTICLE_SPRITE_MASK           0x000100     //!< A Mask if the Particles should be displayed as SPRITESs
#define PARTICLE_MODEL_MASK            0x001000     //!< A Mask if the Particles should be displayed as MODELSs
#define PARTICLE_WORDL_ENTITY_MASK     0x010000     //!< A Mask if the Particles should be displayed as WORLD_ENTITIEs
#define PARTICLE_MULTI_MASK            0x100000     //!< A Mask if they are Multi-partilces

//! An enumerator for the different types of particles.
typedef enum PARTICLE_TYPE
{
  PARTICLE_DOT           =  PARTICLE_DOT_MASK,
  PARTICLE_SPARK         =  PARTICLE_SPARK_MASK,
  PARTICLE_SPRITE        =  PARTICLE_SPRITE_MASK,
  PARTICLE_MULTI_SPRITE  =  PARTICLE_SPRITE_MASK | PARTICLE_MULTI_MASK,
  PARTICLE_MODEL         =  PARTICLE_MODEL_MASK,
  PARTICLE_MULTI_MODE    =  PARTICLE_MODEL_MASK | PARTICLE_MULTI_MASK
};

#define PARTICLE_DEFAULT_MAX_COUNT    200               //!< A default count of particles in the system.
#define PARTICLE_DEFAULT_TYPE         PARTICLE_SPRITE   //!< A default type of the system.

// FORWARD DECLARATION
class Material;
class ParticleEmitter;
class Field;

//! A struct for one Particle
typedef struct Particle
{
  float         lifeTime;            //!< The time this particle has to live.
  float         lifeCycle;           //!< The fraction of time passed. (in percentage of its lifeTime)

  Vector        position;            //!< The current position of this particle.
  Vector        velocity;            //!< The current velocity of this Particle.
  Vector        extForce;            //!< The external Force that influences this Particle.
  Quaternion    orientation;         //!< The current orientation of this Particle.
  Quaternion    momentum;            //!< The current angular momentum (spin) of this Particle.
  float         mass;                //!< The mass of this Particle.
  float         massRand;            //!< A random mass
  float         radius;              //!< The current size of this Particle.
  float         radiusRand;          //!< a random Radius
  GLfloat       color [4];           //!< A Color for the particles.

  Particle*     next;                //!< pointer to the next particle in the List. (NULL if no preceding one)
};

//! A class to handle ParticleSystems
class ParticleSystem : public WorldEntity, public PhysicsInterface {

 public:
  ParticleSystem(unsigned int maxCount = PARTICLE_DEFAULT_MAX_COUNT,
                 PARTICLE_TYPE type = PARTICLE_DEFAULT_TYPE);
  ParticleSystem(const TiXmlElement* root);
  virtual ~ParticleSystem();

  void init();
  virtual void loadParams(const TiXmlElement* root);

  void setType(const char* particleType);
  void setType(PARTICLE_TYPE particleType, int count = 0);
  void setMaterial(Material* material);
  void setMaterialTexture(const char* textureFile);
  void setModel(const char* modelName = NULL);
  void setLifeSpan(float lifeSpan, float randomLifeSpan = 0.0);
  void setConserve(float conserve);
  void setMaxCount(int maxCount);

  /* Per-Particle-Attributes */
  void setRadius(float lifeCycleTime, float radius, float randRadius = 0.0);
  void setMass(float lifeCycleTime, float mass, float randMass = 0.0);
  void setColor(float lifeCycleTime, float red, float green, float blue, float alpha);

  /** @returns the Type of the particles */
  inline PARTICLE_TYPE getType() const { return this->particleType; };
  /** @returns the Material that lies on this particles */
  inline const Material* getMaterial() const { return this->material; };
  /** @returns the lifespan of the particles */
  inline float getLifeSpan() const { return this->lifeSpan; };
  /** @returns the starting-radius of the particles */
  inline float getStartRadius() { return this->radiusAnim.getValue(0.0); };
  /** @returns the end-radius of the particles */
  inline float getEndRadius() { return this->radiusAnim.getValue(1.0); };
  /** @returns the conserve-factor of the particles */
  inline float getConserve() const { return this->conserve; };
  /** @returns the initial mass of the particles */
  inline float getMass() const { return this->initialMass; };

  virtual unsigned int getFaceCount() const;


  virtual void applyField(Field* field);
  /** \brief this is an empty function, because the Physics are implemented in tick @param dt: useless here */
  virtual void tickPhys(float dt) {};

  void addParticle(const Vector& position, const Vector& velocity, const Quaternion& orientation, const Quaternion& momentum, unsigned int data = 0);

  virtual void tick(float dt);
  virtual void draw() const;

  void debug() const;

 private:
  float             conserve;            //!< How much energy gets conserved to the next Tick.
  float             lifeSpan;            //!< Initial lifetime of a Particle.
  float             randomLifeSpan;      //!< A random value for the Lifespan (around the initial lifetime)
  float             initialMass;         //!< The initial Mass of the Particle
  float             randomInitialMass;   //!< The random initial Mass of the Particle

  int               maxCount;            //!< The maximum count of Particles.
  int               count;               //!< The current count of Particles.
  PARTICLE_TYPE     particleType;        //!< A type for all the Particles
  Material*         material;            //!< A Material for all the Particles.
  Particle*         particles;           //!< A list of particles of this System.
  Particle*         deadList;            //!< A list of dead Particles in the System.

  GLuint*           glID;                //!< A List of different gl-List-ID's
  GLuint            dialectCount;        //!< How many different types of particles are there in the Particle System

  // per particle attributes
  QuickAnimation    radiusAnim;          //!< Animation of the radius
  QuickAnimation    randRadiusAnim;      //!< Animation of the random Value of the radius
  QuickAnimation    massAnim;            //!< Animation of the mass
  QuickAnimation    randMassAnim;        //!< Animation of the random Mass
  QuickAnimation    colorAnim[4];        //!< Animation of the 4 colors (r,g,b,a)
};

#endif /* _PARTICLE_SYSTEM_H */
