/*!
 * @file skybox.h
 *  Definition of the SkyBox, that handles the Display of an atmosphere for orxonox.
 *
 * A SkyBox is always centered at the current working Camera, and will only obey the cameras
 * movment but not its rotation.
*/

#ifndef _SKYBOX_H
#define _SKYBOX_H

/* INCLUDES */
#include "world_entity.h"

/* FORWARD DECLARATION */
class Material;
class Texture;

//! A Class to handle a SkyBox
class SkyBox : public WorldEntity
{
 public:
  SkyBox(const char* fileName = NULL);
  SkyBox(const TiXmlElement* root);

  virtual ~SkyBox();

  void init();
  void preInit();

  void loadParams(const TiXmlElement* root);

  void postInit();

  virtual void draw();

  void setSize(float size);
  /** assumes jpg as input-format */
  void setTexture(const char* name) { if (textureName) delete[] textureName; textureName = new char[strlen(name)+1]; strcpy(textureName, name); this->setTextureAndType (name, "jpg"); };

  void setTextureAndType(const char* name, const char* extension);
  void setTextures(const char* top, const char* bottom, const char* left,
                   const char* right, const char* front, const char* back);

  void loadCubeMapTextures(const char* top, const char* bottom, const char* left,
                           const char* right, const char* front, const char* back);

  static void enableCubeMap();
  static void disableCubeMap();

  virtual int       writeBytes(const byte* data, int length, int sender);
  virtual int       readBytes(byte* data, int maxLength, int * reciever);
  virtual void      writeDebug() const;
  virtual void      readDebug() const;

 private:
  void rebuild();

  Material*       material[6];     //!< Materials for the SkyBox. sorted by number (0-5) top, bottom, left, right, front, back
  Texture*        texture[6];      //!< Textures for the CubeMap.

  float           size;            //!< Size of the SkyBox. This should match the frustum maximum range.
  float           textureSize;     //!< this is the length of a texture (assumes a square texture)
  char*           textureName;     //!< Name of the Texture

};

#endif  /* _SKYBOX_H */



