/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "skybox.h"

#include "load_param.h"
#include "factory.h"
#include "static_model.h"

#include "material.h"
#include "texture.h"

#include "network_game_manager.h"
#include "converter.h"

using namespace std;

CREATE_FACTORY(SkyBox, CL_SKYBOX);

/**
 * Constructs a SkyBox and takes fileName as a map.
 * @param fileName the file to take as input for the SkyBox
*/
SkyBox::SkyBox(const char* fileName)
{
  this->preInit();
  if (fileName)
    this->setTextureAndType(fileName, ".jpg");
  this->postInit();
}

/**
 *  initializes a skybox from a XmlElement
*/
SkyBox::SkyBox(const TiXmlElement* root)
{
  this->preInit();

  this->loadParams(root);

  this->postInit();
}

void SkyBox::loadParams(const TiXmlElement* root)
{
  static_cast<WorldEntity*>(this)->loadParams(root);

  LoadParam(root, "Materialset", this, SkyBox, setTexture)
      .describe("Sets the material on the SkyBox. The string must be the path relative to the data-dir, and without a trailing .jpg");

  LoadParam(root, "Size", this, SkyBox, setSize)
      .describe("Sets the Size of the SkyBox (normally this should be 90% of the maximal viewing Distance).");
}

void SkyBox::preInit()
{
  this->setClassID(CL_SKYBOX, "SkyBox");
  this->toList(OM_ENVIRON_NOTICK);
  this->size = 100.0;
  this->textureSize = 1024.0f;

  for (int i = 0; i < 6; i++)
    {
      this->material[i] = new Material();
      this->material[i]->setIllum(3);
      this->material[i]->setDiffuse(0.0,0.0,0.0);
      this->material[i]->setSpecular(0.0,0.0,0.0);
      this->material[i]->setAmbient(2.0, 2.0, 2.0);

      this->texture[i] = NULL;
    }
  this->setParentMode(PNODE_MOVEMENT);

  this->textureName = NULL;
}

void SkyBox::postInit()
{
  this->rebuild();
}


/**
 *  default destructor
*/
SkyBox::~SkyBox()
{
  PRINTF(5)("Deleting SkyBox\n");
  this->setModel(NULL); //< so that WorldEntity does not try to delete it again.
  for (int i = 0; i < 6; i++)
    delete this->material[i];
}

/**
 *  sets A set of textures when just giving a Name and an extension:

   usage: give this function an argument like
   setTexture("skybox", "jpg");
   and it will convert this to
   setTextures("skybox_top.jpg", "skybox_bottom.jpg", "skybox_left.jpg",
               "skybox_right.jpg", "skybox_front.jpg", "skybox_back.jpg");
*/
void SkyBox::setTextureAndType(const char* name, const char* extension)
{
  char* top    = new char[strlen(name)+strlen(extension)+ 10];
  char* bottom = new char[strlen(name)+strlen(extension)+ 10];
  char* left   = new char[strlen(name)+strlen(extension)+ 10];
  char* right  = new char[strlen(name)+strlen(extension)+ 10];
  char* front  = new char[strlen(name)+strlen(extension)+ 10];
  char* back   = new char[strlen(name)+strlen(extension)+ 10];

  sprintf(top, "%s_top.%s", name, extension);
  sprintf(bottom, "%s_bottom.%s", name, extension);
  sprintf(left, "%s_left.%s", name, extension);
  sprintf(right, "%s_right.%s", name, extension);
  sprintf(front, "%s_front.%s", name, extension);
  sprintf(back, "%s_back.%s", name, extension);

  this->setTextures(top, bottom, left, right, front, back);

  // deleted alocated memory of this function
  delete []top;
  delete []bottom;
  delete []left;
  delete []right;
  delete []front;
  delete []back;
}

/**
 *  Defines which textures should be loaded onto the SkyBox.
 * @param top the top texture.
 * @param bottom the bottom texture.
 * @param left the left texture.
 * @param right the right texture.
 * @param front the front texture.
 * @param back the back texture.
*/
void SkyBox::setTextures(const char* top, const char* bottom, const char* left,
                          const char* right, const char* front, const char* back)
{
  this->material[0]->setDiffuseMap(top);
  this->material[1]->setDiffuseMap(bottom);
  this->material[2]->setDiffuseMap(left);
  this->material[3]->setDiffuseMap(right);
  this->material[4]->setDiffuseMap(front);
  this->material[5]->setDiffuseMap(back);
  this->loadCubeMapTextures(top, bottom, left, right, front, back);
  //this->enableCubeMap();
}

void SkyBox::loadCubeMapTextures(const char* top, const char* bottom, const char* left,
                                  const char* right, const char* front, const char* back)
{
  this->texture[0] = new Texture (top, GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB);
  this->texture[1] = new Texture (bottom, GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB);
  this->texture[2] = new Texture (left, GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB);
  this->texture[3] = new Texture (right, GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB);
  this->texture[4] = new Texture (front, GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB);
  this->texture[5] = new Texture (back, GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB);
}

void SkyBox::enableCubeMap()
{
  glEnable(GL_TEXTURE_CUBE_MAP_ARB);
  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_ARB);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_ARB);
  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_ARB);
  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
  glEnable(GL_TEXTURE_GEN_R);
}

void SkyBox::disableCubeMap()
{
  glDisable(GL_TEXTURE_CUBE_MAP_ARB);
  glDisable(GL_TEXTURE_GEN_S);
  glDisable(GL_TEXTURE_GEN_T);
  glDisable(GL_TEXTURE_GEN_R);
}



/**
 * @param size The new size of the SkyBox

 * do not forget to rebuild the SkyBox after this.
*/
void SkyBox::setSize(float size)
{
  this->size = size;
}



void SkyBox::draw()
{
  glPushAttrib(GL_LIGHTING_BIT);
  glDisable(GL_LIGHTING);
  WorldEntity::draw();
//   glEnable(GL_LIGHTING);
  glPopAttrib();

}


/**
 *  rebuilds the SkyBox

   this must be done, when changing the Size of the Skybox (runtime-efficency)
*/
void SkyBox::rebuild()
{
  StaticModel* model = new StaticModel();

  model->addVertex (-0.5*size, -0.5*size, 0.5*size);
  model->addVertex (0.5*size, -0.5*size, 0.5*size);
  model->addVertex (-0.5*size, 0.5*size, 0.5*size);
  model->addVertex (0.5*size, 0.5*size, 0.5*size);
  model->addVertex (-0.5*size, 0.5*size, -0.5*size);
  model->addVertex (0.5*size, 0.5*size, -0.5*size);
  model->addVertex (-0.5*size, -0.5*size, -0.5*size);
  model->addVertex (0.5*size, -0.5*size, -0.5*size);

//   model->addVertexTexture (0.0, 1.0);
//   model->addVertexTexture (1.0, 1.0);
//   model->addVertexTexture (1.0, 0.0);
//   model->addVertexTexture (0.0, 0.0);

  model->addVertexTexture (1.0/this->textureSize, (this->textureSize - 1.0)/this->textureSize);
  model->addVertexTexture ((this->textureSize - 1.0)/this->textureSize, (this->textureSize - 1.0)/this->textureSize);
  model->addVertexTexture ((this->textureSize - 1.0)/this->textureSize, 1.0/this->textureSize);
  model->addVertexTexture (1.0/this->textureSize, 1.0/this->textureSize);


  model->addVertexNormal (0.0, 0.0, 1.0);
  model->addVertexNormal (0.0, 1.0, 0.0);
  model->addVertexNormal (0.0, 0.0, -1.0);
  model->addVertexNormal (0.0, -1.0, 0.0);
  model->addVertexNormal (1.0, 0.0, 0.0);
  model->addVertexNormal (-1.0, 0.0, 0.0);

  model->setMaterial(material[0]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 2,0,3, 3,1,3, 5,2,3, 4,3,3); // top
  model->setMaterial(material[1]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 6,0,1, 7,1,1, 1,2,1, 0,3,1); // bottom
  model->setMaterial(material[2]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 4,2,2, 5,3,2, 7,0,2, 6,1,2); // left
  model->setMaterial(material[3]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 0,0,0, 1,1,0, 3,2,0, 2,3,0); // right
  model->setMaterial(material[4]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 1,0,5, 7,1,5, 5,2,5, 3,3,5); // front
  model->setMaterial(material[5]);
  model->addFace (4, VERTEX_TEXCOORD_NORMAL, 6,0,4, 0,1,4, 2,2,4, 4,3,4); // back

  model->finalize();

  this->setModel(model);
}

int SkyBox::writeBytes( const byte * data, int length, int sender )
{
  setRequestedSync( false );
  setIsOutOfSync( false );

  SYNCHELP_READ_BEGIN();

  SYNCHELP_READ_FKT( WorldEntity::writeState );

  SYNCHELP_READ_FLOAT( size );
  if ( textureName )
  {
    delete[] textureName;
    textureName = NULL;
  }
  SYNCHELP_READ_STRINGM( textureName );

  this->setSize( size );
  this->setTextureAndType( textureName, "jpg" );
  this->rebuild();

  return SYNCHELP_READ_N;
}



int SkyBox::readBytes( byte * data, int maxLength, int * reciever )
{
  if ( isOutOfSync() && !requestedSync() && this->getHostID()!=this->getOwner() )
  {
    (NetworkGameManager::getInstance())->sync( this->getUniqueID(), this->getOwner() );
    setRequestedSync( true );
  }

  int rec = this->getRequestSync();
  if ( rec > 0 )
  {
    *reciever = rec;

    SYNCHELP_WRITE_BEGIN();

    SYNCHELP_WRITE_FKT( WorldEntity::readState );

    SYNCHELP_WRITE_FLOAT(this->size);
    SYNCHELP_WRITE_STRING(this->textureName);

    return SYNCHELP_WRITE_N;
  }

  *reciever = 0;
  return 0;
}

void SkyBox::writeDebug( ) const
{
}

void SkyBox::readDebug( ) const
{
}
