/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "helicopter.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "factory.h"
#include "key_mapper.h"
#include "event_handler.h"
#include "state.h"

#include "graphics_engine.h"

using namespace std;

CREATE_FACTORY(Helicopter, CL_HELICOPTER);

/**
 *  creates the controlable Spaceship
 */
Helicopter::Helicopter()
{
  this->init();
}

/**
 *  destructs the spaceship, deletes alocated memory
 */
Helicopter::~Helicopter ()
{
}

/**
 * loads a Spaceships information from a specified file.
 * @param fileName the name of the File to load the spaceship from (absolute path)
 */
Helicopter::Helicopter(const char* fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for Helicopter.\n", fileName);
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 *  creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
Helicopter::Helicopter(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_CANNON));

  cannon->setName("BFG");

  this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1 ,1);
  this->addWeapon(cannon, 0, 6);

  //this->addWeapon(turret, 3, 0);

  this->getWeaponManager()->changeWeaponConfig(1);
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);
}


/**
 * initializes a Helicopter
 */
void Helicopter::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_HELICOPTER, "Helicopter");

  PRINTF(4)("SPACESHIP INIT\n");

  this->loadModel("models/ships/helicopter_#.obj", 1.0);

  EventHandler::getInstance()->grabEvents(true);

  bUp = bDown = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = false;
  bFire = false;
  xMouse = yMouse = 0;
  mouseSensitivity = 0.05;
  //rotorspeed = 1;
  //tailrotorspeed = 0;

  //cycle = 0.0;


  //travelSpeed = 15.0;
  this->velocity = Vector(0.0,0.0,0.0);
  this->velocityDir = Vector(1.0,0.0,0.0);

//   GLGuiButton* button = new GLGuiPushButton();
//   button->show();
//   button->setLabel("orxonox");
//   button->setBindNode(this);

  //add events to the eventlist
  registerEvent(SDLK_w);
  registerEvent(SDLK_s);
  registerEvent(SDLK_a);
  registerEvent(SDLK_d);
  registerEvent(SDLK_q);
  registerEvent(SDLK_e);
  registerEvent(SDLK_c);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  registerEvent(SDLK_PAGEUP);
  registerEvent(SDLK_PAGEDOWN);
  registerEvent(EV_MOUSE_MOTION);

  this->getWeaponManager()->setSlotCount(7);

  this->getWeaponManager()->setSlotPosition(0, Vector(-2.6, .1, -3.0));
  this->getWeaponManager()->setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(1, Vector(-2.6, .1, 3.0));
  this->getWeaponManager()->setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(2, Vector(-1.5, .5, -.5));
  this->getWeaponManager()->setSlotDirection(2, Quaternion(-M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(3, Vector(-1.5, .5, .5));
  this->getWeaponManager()->setSlotDirection(3, Quaternion(M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(4, Vector(-1.5, -.5, .5));
  this->getWeaponManager()->setSlotDirection(4, Quaternion(-M_PI_4*.5+M_PI, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(5, Vector(-1.5, -.5, -.5));
  this->getWeaponManager()->setSlotDirection(5, Quaternion(+M_PI_4*.5-M_PI, Vector(1,0,0)));
//
   this->getWeaponManager()->setSlotPosition(6, Vector(-1, 0.0, 0));
   this->getWeaponManager()->setSlotCapability(6, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
   //
//   this->getWeaponManager()->setSlotPosition(8, Vector(-2.5, -0.3, -2.0));
//   this->getWeaponManager()->setSlotDirection(8, Quaternion(-M_PI, Vector(1,0,0)));
//
//   this->getWeaponManager()->setSlotPosition(9, Vector(-2.5, -0.3, 2.0));
//   this->getWeaponManager()->setSlotDirection(9, Quaternion(+M_PI, Vector(1,0,0)));:

  this->getWeaponManager()->getFixedTarget()->setParent(this);
  this->getWeaponManager()->getFixedTarget()->setRelCoor(100000,0,0);

  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);

}

/**
 * loads the Settings of a Helicopter from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void Helicopter::loadParams(const TiXmlElement* root)
{
  static_cast<WorldEntity*>(this)->loadParams(root);
}




void Helicopter::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( true);
  this->attachCamera();


}

void Helicopter::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);
  this->detachCamera();

}


/**
 *  effect that occurs after the Helicopter is spawned
*/
void Helicopter::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the spaceship left the game
*/
void Helicopter::leftWorld ()
{}

/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void Helicopter::collidesWith(WorldEntity* entity, const Vector& location)
{
  if (entity->isA(CL_TURRET_POWER_UP))
  {
    this->ADDWEAPON();
    }
//  PRINTF(3)("collision %s vs %s @ (%f,%f,%f)\n", this->getClassName(), entity->getClassName(), location.x, location.y, location.z);
}



/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void Helicopter::tick (float time)
{
  /*
  tailrotorspeed += xMouse/20;
  if (tailrotorspeed >= 0.07) tailrotorspeed = 0.07;
  else if (tailrotorspeed <= -0.07) tailrotorspeed = -0.07;

  if (tailrotorspeed > 0.0008) tailrotorspeed -= 0.001;
  else if (tailrotorspeed < -0.0008) tailrotorspeed += 0.001;
  if (tailrotorspeed <= 0.001 && tailrotorspeed >= -0.001) tailrotorspeed = 0;
  */
  
  // spaceship controlled movement
  this->calculateVelocity(time);

  Vector move = (velocity)*time;

  // this is the air friction (necessary for a smooth control)
  if(velocity.len() != 0) velocity -= velocity*0.1;
  
  //travelSpeed = velocity.len();

  //physics: Gravity
  /*this->shiftCoor(Vector(0,-1,0));

  this->shiftCoor(getAbsDirY()*rotorspeed);
  */
  
  /*
  //hoover effect
  cycle += time;
  this->shiftCoor(Vector(0,1,0)*cos(this->cycle*2.0)*0.02);
  */

  //readjust
 // if (this->getAbsDirZ().y > 0.1) this->shiftDir(Quaternion(time*0.3, Vector(1,0,0)));
  //else if (this->getAbsDirZ().y < -0.1) this->shiftDir(Quaternion(-time*0.3, Vector(1,0,0)));

  //SDL_WarpMouse(GraphicsEngine::getInstance()->getResolutionX()/2, GraphicsEngine::getInstance()->getResolutionY()/2);

  this->shiftCoor (move);
  //this->shiftDir(Quaternion(-M_PI/4*tailrotorspeed, Vector(0,1,0)));

  this->getWeaponManager()->tick(time);
  // weapon system manipulation
  this->weaponAction();
}

/**
 *  calculate the velocity
 * @param time the timeslice since the last frame
*/
void Helicopter::calculateVelocity (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  float rotValX = 0.0;
  float rotValZ = 0.0;
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */

  if( this->bUp )
   {
     //this->shiftCoor(this->getAbsDirX());
     //accel -= this->getAbsDirY();
     
     accel += Vector((this->getAbsDirX()).x,0,(this->getAbsDirX()).z);
     if((this->getAbsDirX()).y >= -0.1) rotValZ -= time;
   }
   else
   {
       if(this->getAbsDirX().y < -.02) this->shiftDir(Quaternion(time, Vector(0,0,1)));
   }

  if( this->bDown )
   {
     //this->shiftCoor((this->getAbsDirX())*-1);
     //accel -= this->getAbsDirY();
     
     accel -= Vector((this->getAbsDirX()).x,0,(this->getAbsDirX()).z);
     rotValZ += time;
   }
   else
   {
         if(this->getAbsDirX().y > 0.02) this->shiftDir(Quaternion(-time, Vector(0,0,1)));
   }

  if( this->bLeft /* > -this->getRelCoor().z*2*/)
  {
    //this->shiftDir(Quaternion(time, Vector(0,1,0)));
    //accel -= this->getAbsDirY();
    //velocityDir.normalize();
    
    accel -= Vector((this->getAbsDirZ()).x,0,(this->getAbsDirZ()).z);
    rotValX -= time;
  }
  else
   {
         if(this->getAbsDirZ().y > 0.02) this->shiftDir(Quaternion(time, Vector(1,0,0)));
   }

  if( this->bRight /* > this->getRelCoor().z*2*/)
  {
    //this->shiftDir(Quaternion(-time, Vector(0,1,0)));
    //accel += this->getAbsDirY();
    //velocityDir.normalize();
    
    accel += Vector((this->getAbsDirZ()).x,0,(this->getAbsDirZ()).z);
    rotValX += time;
  }
  else
   {
         if(this->getAbsDirZ().y < -0.02) this->shiftDir(Quaternion(-time, Vector(1,0,0)));
   }

  if( this->bRollL /* > -this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(-time, Vector(1,0,0)));
    //accel -= rightDirection;
    //velocityDir.normalize();
    //rot +=Vector(1,0,0);
    //rotVal -= .4;
  }
  if( this->bRollR /* > this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(time, Vector(1,0,0)));

    //accel += rightDirection;
    //velocityDir.normalize();
    //rot += Vector(1,0,0);
    //rotVal += .4;
  }
  if (this->bAscend )
  {
    //this->shiftDir(Quaternion(time, Vector(0,0,1)));

    accel += this->getAbsDirY();
    //rotorspeed += 0.05;
    //if (rotorspeed >= 2) rotorspeed = 2;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal += .4;
  }
  else
  {
    //if(rotorspeed >= 1.05) rotorspeed -= 0.05;
  }

  if (this->bDescend )
  {
    //this->shiftDir(Quaternion(-time, Vector(0,0,1)));

    accel -= this->getAbsDirY();
    //rotorspeed -= 0.05;
    //if (rotorspeed <= 0) rotorspeed = 0;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal -= .4;
  }
  else
  {
    //if(rotorspeed <= 0.05) rotorspeed += 0.05;
  }

  velocity += accel*3;
  if((this->getAbsDirX()).y <= 0.3 && (this->getAbsDirX()).y >= -0.3) this->shiftDir(Quaternion(rotValZ, Vector(0,0,1)));
  if((this->getAbsDirZ()).y <= 0.3 && (this->getAbsDirZ()).y >= -0.3) this->shiftDir(Quaternion(rotValX, Vector(1,0,0)));
}


void Helicopter::draw() const
{
  WorldEntity::draw();

  this->getWeaponManager()->draw();
}


/**
 * weapon manipulation by the player
*/
void Helicopter::weaponAction()
{
  if( this->bFire)
    {
      this->getWeaponManager()->fire();
    }
}

/**
 * @todo switch statement ??
 */
void Helicopter::process(const Event &event)
{


  if( event.type == SDLK_a)
      this->bLeft = event.bPressed;
  else if( event.type == SDLK_d)
      this->bRight = event.bPressed;
  else if( event.type == KeyMapper::PEV_FIRE1)
      this->bFire = event.bPressed;
  else if( event.type == KeyMapper::PEV_NEXT_WEAPON && event.bPressed)
    this->getWeaponManager()->nextWeaponConfig();//if( !event.bPressed) this->bWeaponChange = !this->bWeaponChange;
  else if ( event.type == KeyMapper::PEV_PREVIOUS_WEAPON && event.bPressed)
    this->getWeaponManager()->previousWeaponConfig();
  else if( event.type == SDLK_e)
    this->bAscend = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == SDLK_c)
    this->bDescend = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == SDLK_w)
    this->bUp = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == SDLK_s)
    this->bDown = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse = event.xRel*mouseSensitivity;
    this->yMouse = event.yRel*mouseSensitivity;

    this->shiftDir(Quaternion(-M_PI/4*xMouse*mouseSensitivity, Vector(0,1,0))/*Quaternion(-M_PI/4*yMouse*mouseSensitivity, Vector(0,0,1))*/);
  }
}

#include "weapons/aiming_turret.h"
// FIXME THIS MIGHT BE CONSIDERED EITHER A FEATURE, OR A BUG
void Helicopter::ADDWEAPON()
{
  Weapon* turret = NULL;

  if ((float)rand()/RAND_MAX < .1)
  {
    //if (this->getWeaponManager()->hasFreeSlot(2, WTYPE_TURRET))
    {
      turret = new Turret();
      this->addWeapon(turret, 2);
      this->getWeaponManager()->changeWeaponConfig(2);
    }
  }
  else
  {
    //if (this->getWeaponManager()->hasFreeSlot(3))
    {
      turret = new AimingTurret();
      this->addWeapon(turret, 3);

      this->getWeaponManager()->changeWeaponConfig(3);
    }
  }

  if(turret != NULL)
  {
    turret->setName("Turret");
    turret->setStateDuration(WS_SHOOTING, (float)rand()/RAND_MAX*.5+.1);
  }
}
