/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "vertex_array_model.h"

#include "stdlibincl.h"
#include <stdarg.h>

#include "tc.h"

using namespace std;

/////////////
/// MODEL ///
/////////////
/**
 * @brief Creates a 3D-VertexArrayModel.
 *
 * assigns it a Name and a Type
 */
VertexArrayModel::VertexArrayModel()
{
  this->setClassID(CL_MODEL, "VertexArrayModel");

  this->newStripe();
}

/**
 * @brief special copy constructor for converting Models to VertexArray-Stripes
 * @param model the Model to produce a VertexArray model from.
 *
 * Code that uses Brad Granthams
 * excelent TC-code for generating stripes out of a mix of ModelCoordinates.
 */
VertexArrayModel::VertexArrayModel(const Model& model)
{
  this->setClassID(CL_MODEL, "VertexArrayModel");

  // importing the data to the new Model.
  this->newStripe();

  for (unsigned int i = 0; i < model.getVertexCount()*3; i+=3)
    this->addVertex(model.getVertexArray()[i], model.getVertexArray()[i+1], model.getVertexArray()[i+2]);
  for (unsigned int i = 0; i < model.getVertexCount()*3; i+=3)
    this->addColor((float)i / (float)model.getVertexCount(), 0, 0);

  for (unsigned int i = 0; i < model.getNormalsCount()*3; i+=3)
     this->addNormal(model.getNormalsArray()[i], model.getNormalsArray()[i+1], model.getNormalsArray()[i+2]);
  for (unsigned int i = 0; i < model.getTexCoordCount(); i+=2)
     this->addTexCoor(model.getTexCoordArray()[i], model.getTexCoordArray()[i+1]);


  // The acTC object generating this Model. //
  ACTCData *tc;
  tc = actcNew();
  if(tc == NULL) {
    /* memory allocation failed */
    /* print error here and exit or whatever */
  }

  // inputing the data of model to the tc
  actcBeginInput(tc);
  for(unsigned int i = 0; i < model.getTriangleCount(); i++)
  {
      actcAddTriangle(tc,
                      model.getTriangles()[i].indexToVertices[0],
                      model.getTriangles()[i].indexToVertices[1],
                      model.getTriangles()[i].indexToVertices[2]);
  }
  actcEndInput(tc);



  int prim;
  uint v1, v2, v3;

  actcBeginOutput(tc);
  while((prim = actcStartNextPrim(tc, &v1, &v2) != ACTC_DATABASE_EMPTY))
  {
    this->newStripe();

    this->addIndice(v1);
    this->addIndice(v2);
    /* start a primitive of type "prim" with v1 and v2 */
    while(actcGetNextVert(tc, &v3) != ACTC_PRIM_COMPLETE)
    {
      /* continue primitive using v3 */
      this->addIndice(v3);
    }
  }
  actcEndOutput(tc);

  this->finalize();
}


/**
 * @brief deletes a VertexArrayModel.
 *
 * Looks if any from model allocated space is still in use, and if so deleted it.
 */
VertexArrayModel::~VertexArrayModel()
{
  PRINTF(4)("Deleting VertexArrayModel ");
  if (this->getName())
  {
    PRINT(4)("%s\n", this->getName());
  }
  else
  {
    PRINT(4)("\n");
  }
}


/**
 * @brief Draws the VertexArrayModels of all Groups.
 *
 * It does this by just calling the Lists that must have been created earlier.
 */
void VertexArrayModel::draw() const
{
  PRINTF(4)("drawing 3D-VertexArrayModel %s\n", this->getName());
  glEnableClientState(GL_VERTEX_ARRAY );
  glEnableClientState(GL_TEXTURE_COORD_ARRAY );
  glEnableClientState(GL_NORMAL_ARRAY );
  glEnableClientState(GL_COLOR_ARRAY );


  glVertexPointer(3, GL_FLOAT, 0, &this->vertices[0]);
  glNormalPointer(GL_FLOAT, 0, &this->normals[0]);
  glTexCoordPointer(2, GL_FLOAT, 0, &this->texCoords[0]);
  glColorPointer(3, GL_FLOAT, 0, &this->colors[0]);

  for (GLuint i = 1; i < this->stripes.size(); ++i)
    {
      glDrawElements( GL_TRIANGLE_STRIP,
                      this->stripes[i] - this->stripes[i-1],
                      GL_UNSIGNED_INT,
                      &this->indices[this->stripes[i-1]] );
    }
}


//////////
// MESH //
//////////
/**
 * @brief generates a new Stripe in this Model
 */
void VertexArrayModel::newStripe()
{
  // no stripes of size 0
  if (this->stripes.empty() || this->indices.size() != this->stripes.back())
    this->stripes.push_back(this->indices.size());
}


/**
 * @brief parses a vertex-String
 * @param x the X-coordinate of the Vertex to add.
 * @param y the Y-coordinate of the Vertex to add.
 * @param z the Z-coordinate of the Vertex to add.
 */
void VertexArrayModel::addVertex(float x, float y, float z)
{
  this->vertices.push_back(x);
  this->vertices.push_back(y);
  this->vertices.push_back(z);
  this->pModelInfo.numVertices++;
}


/**
 * @brief adds a VertexNormal.
 * @param x The x coordinate of the Normal.
 * @param y The y coordinate of the Normal.
 * @param z The z coordinate of the Normal.
 *
 * If a vertexNormal line is found this function will inject it into the vertexNormal-Array
 */
void VertexArrayModel::addNormal(float x, float y, float z)
{
  this->normals.push_back(x);
  this->normals.push_back(y);
  this->normals.push_back(z);
  this->pModelInfo.numNormals++;
}


/**
 * @brief adds a Texture Coordinate
 * @param u The u coordinate of the TextureCoordinate.
 * @param v The y coordinate of the TextureCoordinate.
 *
 *  If a TextureCoordinate line is found this function will inject it into the TextureCoordinate-Array
 */
void VertexArrayModel::addTexCoor(float u, float v)
{
  this->texCoords.push_back(u);
  this->texCoords.push_back(v);
  this->pModelInfo.numTexCoor++;
}

/**
 * @brief adds a new Color
 * @param r the Red Component of the VertexColor to add.
 * @param g the Green Component of the VertexColor to add.
 * @param b the Blue of the VertexColor to add.
 */
void VertexArrayModel::addColor(float r, float g, float b)
{
  this->colors.push_back(r);
  this->colors.push_back(g);
  this->colors.push_back(b);
  // FIXME
}


/**
 *  adds a new Face
 * @param faceElemCount the number of Vertices to add to the Face.
 * @param type The information Passed with each Vertex
*/
void VertexArrayModel::addIndice(GLuint indice)
{
  this->indices.push_back(indice);
}


/**
 * @brief Finalizes an Object. This can be done outside of the Class.
 */
void VertexArrayModel::finalize()
{
  // finalize the Arrays
  this->newStripe();
}




/////////////
// TESTING //
/////////////
/**
* @brief Includes a default model
*
* This will inject a Cube, because this is the most basic model.
*/
void VertexArrayModel::planeModel(float sizeX, float sizeY, unsigned int resolutionX, unsigned int resolutionY)
{
  GLuint i, j;
  for (i = 0; i < resolutionY; i++)
    {
      for (j = 0; j < resolutionX; j++)
        {
          this->addVertex((float)i - (float)sizeY/2.0, 0.0, (float)j - (float)sizeX/2.0);
          this->addNormal(0.0, 1, 0.0);
          this->addTexCoor((float)i/(float)resolutionY, (float)j/(float)resolutionY);
          this->addColor((float)i/20.0, 0.0, (float)j/20.0);
        }
    }

  for (i = 0; i < resolutionY-1; i++)
  {
    for (j = 0; j < resolutionX; j++)
    {
      this->addIndice( resolutionY*i + j );
      this->addIndice( resolutionY*(i+1) + j );
    }
    this->newStripe();
  }
}

#include <cmath>

/**
 * @brief builds a Triangle Stripped sphere
 * @param radius: radius
 * @param loops: the count of loops
 * @param segmentsPerLoop how many Segments per loop
 */
void VertexArrayModel::spiralSphere(const float radius, const unsigned int loops, const unsigned int segmentsPerLoop)
{
  for (unsigned int loopSegmentNumber = 0; loopSegmentNumber < segmentsPerLoop; ++loopSegmentNumber)
  {
    float theta = 0;
    float phi = loopSegmentNumber * 2 * PI / segmentsPerLoop;
    float sinTheta = std::sin(theta);
    float sinPhi = std::sin(phi);
    float cosTheta = std::cos(theta);
    float cosPhi = std::cos(phi);
    this->addVertex(radius * cosPhi * sinTheta, radius * sinPhi * sinTheta, radius * cosTheta);
    this->addNormal(radius * cosPhi * sinTheta, radius * sinPhi * sinTheta, radius * cosTheta);
    this->addTexCoor(0,0); /// FIXME
    this->addColor(.125,.436,.246); ///FIXME
  }
  for (unsigned int loopNumber = 0; loopNumber <= loops; ++loopNumber)
  {
    for (unsigned int loopSegmentNumber = 0; loopSegmentNumber < segmentsPerLoop; ++loopSegmentNumber)
    {
      float theta = (loopNumber * PI / loops) + ((PI * loopSegmentNumber) / (segmentsPerLoop * loops));
      if (loopNumber == loops)
      {
        theta = PI;
      }
      float phi = loopSegmentNumber * 2 * PI / segmentsPerLoop;
      float sinTheta = std::sin(theta);
      float sinPhi = std::sin(phi);
      float cosTheta = std::cos(theta);
      float cosPhi = std::cos(phi);
      this->addVertex(radius * cosPhi * sinTheta, radius * sinPhi * sinTheta, radius * cosTheta);
      this->addNormal(radius * cosPhi * sinTheta, radius * sinPhi * sinTheta, radius * cosTheta);
      this->addTexCoor(0,0); //FIXME
      this->addColor(.125,.436,.246);

    }
  }
  for (unsigned int loopSegmentNumber = 0; loopSegmentNumber < segmentsPerLoop; ++loopSegmentNumber)
  {
    this->addIndice(loopSegmentNumber);
    this->addIndice(segmentsPerLoop + loopSegmentNumber);
  }
  for (unsigned int loopNumber = 0; loopNumber < loops; ++loopNumber)
  {
    for (unsigned int loopSegmentNumber = 0; loopSegmentNumber < segmentsPerLoop; ++loopSegmentNumber)
    {
      this->addIndice( ((loopNumber + 1) * segmentsPerLoop) + loopSegmentNumber);
      this->addIndice( ((loopNumber + 2) * segmentsPerLoop) + loopSegmentNumber);
    }
  }
}


/**
 * @brief print out some nice debug information about this VertexArrayModel.
 */
void VertexArrayModel::debug() const
{
  PRINT(0)("VertexArrayModel (%s): debug\n", this->getName());
  PRINT(0)("Stripes: %d; Indices: %d; Vertices: %d; Normals %d; TextCoords %d; Colors %d\n",
            this->stripes.size(),
            this->indices.size(),
            this->vertices.size()/3,
            this->normals.size()/3,
            this->texCoords.size()/2,
            this->colors.size() )/3;
  for (GLuint i = 1; i < this->stripes.size(); ++i)
  {
    PRINT(0)("Stripe-%d (s:%d:e:%d):: ", i, this->stripes[i-1], this->stripes[i]);
    for (GLuint j = this->stripes[i-1] ; j < this->stripes[i]; j++)
    {
      PRINT(0)("->%d", this->indices[j]);
    }
    PRINT(0)("\n");
  }
}
