/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "vertex_array_model.h"

#include "stdlibincl.h"
#include <stdarg.h>

#include "tc.h"

using namespace std;

/////////////
/// MODEL ///
/////////////
/**
 * @brief Creates a 3D-VertexArrayModel.
 *
 * assigns it a Name and a Type
 */
VertexArrayModel::VertexArrayModel()
{
  this->setClassID(CL_MODEL, "VertexArrayModel");

  this->bFinalized = false;
  this->newStripe();
}

/**
 * @brief special copy constructor for converting Models to VertexArray-Stripes
 * @param model the Model to produce a VertexArray model from.
 *
 * Code that uses Brad Granthams
 * excelent TC-code for generating stripes out of a mix of ModelCoordinates.
 */
VertexArrayModel::VertexArrayModel(const Model& model)
{
  this->setClassID(CL_MODEL, "VertexArrayModel");
  this->bFinalized = false;

  // The acTC object generating this Model. //
  ACTCData *tc;
  tc = actcNew();
  if(tc == NULL) {
    /* memory allocation failed */
    /* print error here and exit or whatever */
  }

  // inputing the data of model to the tc
  actcBeginInput(tc);
  for(unsigned int i = 0; i < model.getTriangleCount(); i++)
      actcAddTriangle(tc, model.getTriangles()[i].indexToVertices[0], model.getTriangles()[i].indexToVertices[1], model.getTriangles()[i].indexToVertices[2]);
  actcEndInput(tc);

  // importing the data to the new Model.
  this->newStripe();

  for (unsigned int i = 0; i < model.getVertexCount(); i+=3)
    this->addVertex(model.getVertexArray()[i], model.getVertexArray()[i+1], model.getVertexArray()[i+2]);
  for (unsigned int i = 0; i < model.getNormalsCount(); i+=3)
    this->addNormal(model.getNormalsArray()[i], model.getNormalsArray()[i+1], model.getNormalsArray()[i+2]);
  for (unsigned int i = 0; i < model.getTexCoordCount(); i+=2)
    this->addTexCoor(model.getTexCoordArray()[i], model.getTexCoordArray()[i+1]);


  int prim;
  unsigned int v1, v2, v3;

  actcBeginOutput(tc);
  while((prim = actcStartNextPrim(tc, &v1, &v2) != ACTC_DATABASE_EMPTY))
  {
    this->addIndice(v1);
    this->addIndice(v2);
    printf("%d\n", v1);
    printf("%d\n", v2);
    /* start a primitive of type "prim" with v1 and v2 */
    while(actcGetNextVert(tc, &v3) != ACTC_PRIM_COMPLETE)
    {
      /* continue primitive using v3 */
      this->addIndice(v3);
      printf("%d\n", v3);
    }
    this->newStripe();
  }
  actcEndOutput(tc);

  this->finalize();
}


/**
 * @brief deletes a VertexArrayModel.
 *
 * Looks if any from model allocated space is still in use, and if so deleted it.
 */
VertexArrayModel::~VertexArrayModel()
{
  PRINTF(4)("Deleting VertexArrayModel ");
  if (this->getName())
  {
    PRINT(4)("%s\n", this->getName());
  }
  else
  {
    PRINT(4)("\n");
  }
}


/**
 * @brief Draws the VertexArrayModels of all Groups.
 *
 * It does this by just calling the Lists that must have been created earlier.
 */
void VertexArrayModel::draw() const
{
  PRINTF(4)("drawing the 3D-VertexArrayModels\n");

  glEnableClientState(GL_VERTEX_ARRAY |
		      GL_TEXTURE_COORD_ARRAY |
		      GL_NORMAL_ARRAY);
  //  glEnableClientState(GL_INDEX_ARRAY);

  glVertexPointer(3, GL_FLOAT, 0, this->vertices.getArray());
  glNormalPointer(GL_FLOAT, 0, this->normals.getArray());
  glTexCoordPointer(2, GL_FLOAT, 0, this->texCoords.getArray());

  for (GLuint i = 1; i < this->stripes.size(); ++i)
    {
      glDrawRangeElements(GL_TRIANGLE_STRIP,
			  this->stripes[i-1],
			  this->stripes[i],
			  this->indices.getCount(),
			  GL_UNSIGNED_BYTE,
			  this->indices.getArray());
    }
}


//////////
// MESH //
//////////
/**
 * @brief generates a new Stripe in this Model
 */
void VertexArrayModel::newStripe()
{
  this->stripes.push_back(this->vertices.getCount()-1);
}


/**
 * @brief parses a vertex-String
 * @param x the X-coordinate of the Vertex to add.
 * @param y the Y-coordinate of the Vertex to add.
 * @param z the Z-coordinate of the Vertex to add.
 */
void VertexArrayModel::addVertex(float x, float y, float z)
{
  this->vertices.addEntry(x, y, z);
  this->pModelInfo.numVertices++;
}


/**
 * @brief adds a VertexNormal.
 * @param x The x coordinate of the Normal.
 * @param y The y coordinate of the Normal.
 * @param z The z coordinate of the Normal.
 *
 * If a vertexNormal line is found this function will inject it into the vertexNormal-Array
 */
void VertexArrayModel::addNormal(float x, float y, float z)
{
  this->normals.addEntry(x, y, z);
  this->pModelInfo.numNormals++;
}


/**
 *  adds a Texture Coordinate
 * @param u The u coordinate of the TextureCoordinate.
 * @param v The y coordinate of the TextureCoordinate.
 *
 *  If a TextureCoordinate line is found this function will inject it into the TextureCoordinate-Array
 */
void VertexArrayModel::addTexCoor(float u, float v)
{
  this->texCoords.addEntry(u);
  this->texCoords.addEntry(v);
  this->pModelInfo.numTexCoor++;
}


/**
 *  adds a new Face
 * @param faceElemCount the number of Vertices to add to the Face.
 * @param type The information Passed with each Vertex
*/
void VertexArrayModel::addIndice(GLubyte indice)
{
  this->indices.addEntry(indice);
}


/**
 * @brief Finalizes an Object. This can be done outside of the Class.
 */
void VertexArrayModel::finalize()
{
  // finalize the Arrays
  this->vertices.finalizeArray();
  this->texCoords.finalizeArray();
  this->normals.finalizeArray();
  this->indices.finalizeArray();

  this->newStripe();

  /*
    glEnableClientState(GL_VERTEX_ARRAY |
    GL_TEXTURE_COORD_ARRAY |
    GL_NORMAL_ARRAY);
  */

  this->bFinalized = true;
}



/////////////
// TESTING //
/////////////
/**
 * @brief Includes a default model
 *
 * This will inject a Cube, because this is the most basic model.
 */
void VertexArrayModel::planeModel()
{
  unsigned int i, j;
  for (i = 0; i < 20; i++)
    {
      for (j = 0; j < 20; j++)
	{
	  this->addVertex(i* 50, .5, (j)*50);
	  this->addNormal(0, 1, 0);
	  this->addTexCoor((float)i/20.0, (float)j/20.0);

	}
    }
  for (i = 0; i < 20; i++)
    {
      this->addIndice(i);
      this->addIndice(i+20);
    }
}
