/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Knecht
   co-programmer: Silvan Nellen

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "executor/executor.h"
#include "space_ship.h"

#include "objModel.h"
#include "resource_manager.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "factory.h"
#include "key_mapper.h"
#include "event_handler.h"

#include "graphics_engine.h"

using namespace std;

CREATE_FACTORY(SpaceShip, CL_SPACE_SHIP);

/**
 *  creates the controlable Spaceship
 */
SpaceShip::SpaceShip()
{
  this->init();
}

/**
 *  destructs the spaceship, deletes alocated memory
 */
SpaceShip::~SpaceShip ()
{
}

/**
 * loads a Spaceships information from a specified file.
 * @param fileName the name of the File to load the spaceship from (absolute path)
 */
SpaceShip::SpaceShip(const char* fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for spaceship.\n", fileName);
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 *  creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
SpaceShip::SpaceShip(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_CANNON));

  cannon->setName("BFG");

  this->getWeaponManager()->addWeapon(wpLeft, 1, 0);
  this->getWeaponManager()->addWeapon(wpRight,1 ,1);
  this->getWeaponManager()->addWeapon(cannon, 0, 6);

  //this->getWeaponManager()->addWeapon(turret, 3, 0);

  this->getWeaponManager()->changeWeaponConfig(1);
}


/**
 * initializes a Spaceship
 */
void SpaceShip::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_SPACE_SHIP, "SpaceShip");

  PRINTF(4)("SPACESHIP INIT\n");

//   EventHandler::getInstance()->grabEvents(true);

  bUp = bDown = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = false;
  bFire = false;
  xMouse = yMouse = 0;
  mouseSensitivity = 0.001;
  airViscosity = 1.0;
  cycle = 0.0;


  travelSpeed = 15.0;
  this->velocity = Vector(0.0,0.0,0.0);
  this->mouseDir = this->getAbsDir();

//   GLGuiButton* button = new GLGuiPushButton();
//   button->show();
//   button->setLabel("orxonox");
//   button->setBindNode(this);

  //add events to the eventlist
  registerEvent(SDLK_w);
  registerEvent(SDLK_s);
  registerEvent(SDLK_a);
  registerEvent(SDLK_d);
  registerEvent(SDLK_q);
  registerEvent(SDLK_e);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  registerEvent(SDLK_PAGEUP);
  registerEvent(SDLK_PAGEDOWN);
  registerEvent(EV_MOUSE_MOTION);

  this->getWeaponManager()->setSlotCount(7);

  this->getWeaponManager()->setSlotPosition(0, Vector(-2.6, .1, -3.0));
  this->getWeaponManager()->setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(1, Vector(-2.6, .1, 3.0));
  this->getWeaponManager()->setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(2, Vector(-1.5, .5, -.5));
  this->getWeaponManager()->setSlotDirection(2, Quaternion(-M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(3, Vector(-1.5, .5, .5));
  this->getWeaponManager()->setSlotDirection(3, Quaternion(M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(4, Vector(-1.5, -.5, .5));
  this->getWeaponManager()->setSlotDirection(4, Quaternion(-M_PI_4*.5+M_PI, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(5, Vector(-1.5, -.5, -.5));
  this->getWeaponManager()->setSlotDirection(5, Quaternion(+M_PI_4*.5-M_PI, Vector(1,0,0)));
//
   this->getWeaponManager()->setSlotPosition(6, Vector(-1, 0.0, 0));
   this->getWeaponManager()->setSlotCapability(6, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
   //
//   this->getWeaponManager()->setSlotPosition(8, Vector(-2.5, -0.3, -2.0));
//   this->getWeaponManager()->setSlotDirection(8, Quaternion(-M_PI, Vector(1,0,0)));
//
//   this->getWeaponManager()->setSlotPosition(9, Vector(-2.5, -0.3, 2.0));
//   this->getWeaponManager()->setSlotDirection(9, Quaternion(+M_PI, Vector(1,0,0)));:

  this->getWeaponManager()->getFixedTarget()->setParent(this);
  this->getWeaponManager()->getFixedTarget()->setRelCoor(100000,0,0);

  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);
}

/**
 * loads the Settings of a SpaceShip from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void SpaceShip::loadParams(const TiXmlElement* root)
{
  static_cast<WorldEntity*>(this)->loadParams(root);
}


void SpaceShip::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( true);


}

void SpaceShip::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);


}

/**
 * adds a weapon to the weapon list of the spaceship
 * @param weapon to add
*/
void SpaceShip::addWeapon(Weapon* weapon)
{
  this->getWeaponManager()->addWeapon(weapon);
}


/**
 *  removes a weapon from the spaceship
 * @param weapon to remove
*/
void SpaceShip::removeWeapon(Weapon* weapon)
{
  this->getWeaponManager()->removeWeapon(weapon);
}

/**
 *  effect that occurs after the SpaceShip is spawned
*/
void SpaceShip::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the spaceship left the game
*/
void SpaceShip::leftWorld ()
{}

WorldEntity* ref = NULL;
/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void SpaceShip::collidesWith(WorldEntity* entity, const Vector& location)
{
  if (entity->isA(CL_TURRET_POWER_UP) && entity != ref)
  {
    this->ADDWEAPON();
    ref = entity;
    }
//  PRINTF(3)("collision %s vs %s @ (%f,%f,%f)\n", this->getClassName(), entity->getClassName(), location.x, location.y, location.z);
}

/**
 *  draws the spaceship after transforming it.
*/
void SpaceShip::draw () const
{
  this->drawLODsafe();

  this->getWeaponManager()->draw();

  //this->debug(0);
}

/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void SpaceShip::tick (float time)
{

  // spaceship controlled movement
  this->calculateVelocity(time);

  Vector move = (velocity)*time;

  //orient the velocity in the direction of the spaceship.
  travelSpeed = velocity.len();
  velocity += ((this->getAbsDirX())*travelSpeed-velocity)*airViscosity;
  velocity = (velocity.getNormalized())*travelSpeed;

  //orient the spaceship in direction of the mouse
   rotQuat = Quaternion::quatSlerp( this->getAbsDir(),mouseDir,fabsf(time)*3);
   if (this->getAbsDir().distance(rotQuat) > 0.001)
    this->setAbsDir( rotQuat);
   //this->setAbsDirSoft(mouseDir,5);

  // this is the air friction (necessary for a smooth control)
  if(velocity.len() != 0) velocity -= velocity*0.01;

  //hoover effect
  cycle += time;
  this->shiftCoor(Vector(0,1,0)*cos(this->cycle*2.0)*0.02);

  //readjust
 // if (this->getAbsDirZ().y > 0.1) this->shiftDir(Quaternion(time*0.3, Vector(1,0,0)));
  //else if (this->getAbsDirZ().y < -0.1) this->shiftDir(Quaternion(-time*0.3, Vector(1,0,0)));

  //SDL_WarpMouse(GraphicsEngine::getInstance()->getResolutionX()/2, GraphicsEngine::getInstance()->getResolutionY()/2);

  this->shiftCoor (move);

  this->getWeaponManager()->tick(time);
  // weapon system manipulation
  this->weaponAction();
}

/**
 *  calculate the velocity
 * @param time the timeslice since the last frame
*/
void SpaceShip::calculateVelocity (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  /*
  Vector rot(0.0, 0.0, 0.0); // wird bentigt fr Helicopter
  */
  //float rotVal = 0.0;
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */

  Plane plane(Vector(0,1,0), Vector(0,0,0));

  if( this->bUp )
   {
     //this->shiftCoor(this->getAbsDirX());
      accel += (this->getAbsDirX())*2;

      /* Heli-Steuerung
         accel += (this->getAbsDirX()*2;
         if(
      */
   }

  if( this->bDown )
   {
     //this->shiftCoor((this->getAbsDirX())*-1);
     accel -= (this->getAbsDirX())*2;
   }

  if( this->bLeft/* > -this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(time, Vector(0,1,0)));
//    accel -= rightDirection;
    //velocityDir.normalize();
    //rot +=Vector(1,0,0);
    //rotVal -= .4;
  }
  if( this->bRight /* > this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(-time, Vector(0,1,0)));

    //    accel += rightDirection;
    //velocityDir.normalize();
    //rot += Vector(1,0,0);
    //rotVal += .4;
  }


  if( this->bRollL /* > -this->getRelCoor().z*2*/)
  {
    mouseDir *= Quaternion(-time, Vector(1,0,0));
//    accel -= rightDirection;
    //velocityDir.normalize();
    //rot +=Vector(1,0,0);
    //rotVal -= .4;
  }
  if( this->bRollR /* > this->getRelCoor().z*2*/)
  {
    mouseDir *= Quaternion(time, Vector(1,0,0));

    //    accel += rightDirection;
    //velocityDir.normalize();
    //rot += Vector(1,0,0);
    //rotVal += .4;
  }
  if (this->bAscend )
  {
    this->shiftDir(Quaternion(time, Vector(0,0,1)));

//    accel += upDirection;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal += .4;
  }
  if (this->bDescend )
  {
    this->shiftDir(Quaternion(-time, Vector(0,0,1)));

    //    accel -= upDirection;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal -= .4;
  }

  velocity += accel;
  //rot.normalize();
  //this->setRelDirSoft(Quaternion(rotVal, rot), 5);
}

/**
 * weapon manipulation by the player
*/
void SpaceShip::weaponAction()
{
  if( this->bFire)
    {
      this->getWeaponManager()->fire();
    }
}

/**
 * @todo switch statement ??
 */
void SpaceShip::process(const Event &event)
{


  if( event.type == SDLK_a)
      this->bRollL = event.bPressed;
  else if( event.type == SDLK_d)
      this->bRollR = event.bPressed;
  else if( event.type == KeyMapper::PEV_FIRE1)
      this->bFire = event.bPressed;
  else if( event.type == KeyMapper::PEV_NEXT_WEAPON && event.bPressed)
    this->getWeaponManager()->nextWeaponConfig();//if( !event.bPressed) this->bWeaponChange = !this->bWeaponChange;
  else if ( event.type == KeyMapper::PEV_PREVIOUS_WEAPON && event.bPressed)
    this->getWeaponManager()->previousWeaponConfig();

  else if( event.type == SDLK_w)
    this->bUp = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == SDLK_s)
    this->bDown = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse = event.xRel;
    this->yMouse = event.yRel;
    mouseDir *= (Quaternion(-M_PI/4*xMouse*mouseSensitivity, Vector(0,1,0))*Quaternion(-M_PI/4*yMouse*mouseSensitivity, Vector(0,0,1)));
    if( xMouse*xMouse + yMouse*yMouse < 0.9)
     this->setAbsDir(mouseDir);
  }
}

#include "weapons/aiming_turret.h"
// FIXME THIS MIGHT BE CONSIDERED EITHER A FEATURE, OR A BUG
void SpaceShip::ADDWEAPON()
{
  Weapon* turret = NULL;

  if ((float)rand()/RAND_MAX < .1)
  {
    //if (this->getWeaponManager()->hasFreeSlot(2, WTYPE_TURRET))
    {
      turret = new Turret();
      this->getWeaponManager()->addWeapon(turret, 2);
      this->getWeaponManager()->changeWeaponConfig(2);
    }
  }
  else
  {
    //if (this->getWeaponManager()->hasFreeSlot(3))
    {
      turret = new AimingTurret();
      this->getWeaponManager()->addWeapon(turret, 3);

      this->getWeaponManager()->changeWeaponConfig(3);
    }
  }

  if(turret != NULL)
  {
    turret->setName("Turret");
    turret->setStateDuration(WS_SHOOTING, (float)rand()/RAND_MAX*.5+.1);
  }
}
