/*!
 * @file entity_manager.h
 *  Manages creation and destruction of entities
 */

#ifndef _NETWORK_GAME_MANGER
#define _NETWORK_GAME_MANAGER

/* include this file, it contains some default definitions */
#include "netdefs.h"

/* include base_object.h since all classes are derived from this one */
#include "synchronizeable.h"



/**
 * protocol definition
 *
 *  CREATE_ENTITY:       CLASS_ID, UNIQUE_ID, OWNER
 *  REMOVE_ENTITY:       UNIQUE_ID
 *
 *  CREATE_ENTITY_LIST:  NUMBER, [CLASS_ID, UNIQUE_ID, OWNER][0..NUMBER]
 *  REMOVE_ENTITY_LIST:  NUMBER, [UNIQUE_ID][0..NUMBER]
 *
 *  REQUEST_CREATE:      CLASS_ID
 *  REQUEST_REMOVE:      UNIQUE_ID
 *
 *  //REQUEST_CREATE_LIST: NUMBER, [CLASS_ID][0..NUMBER]
 *  //REQUEST_CREATE_LIST: NUMBER, [UNIQUE_ID][0..NUMBER]
 *
 *  REQUEST_SYNC:        UNIQUE_ID
 *  //REQUEST_SYNC_LIST:   NUMBER, [UNIQUE_ID][0..NUMBER]
 *
 *  YOU_ARE_ENTITY:      UNIQUE_ID
 *
 */

typedef enum NetworkGameManagerProtocol{
  CREATE_ENTITY = 0,
  REMOVE_ENTITY,
  CREATE_ENTITY_LIST,
  REMOVE_ENTITY_LIST,
  REQUEST_CREATE,
  REQUEST_REMOVE,
  REQUEST_SYNC,
  YOU_ARE_ENTITY
};

struct clientBuffer
{
  int length;
  int maxLength;
  byte * buffer;
};

/*!
 * a class that can create and remove entities
 */
class NetworkGameManager: public Synchronizeable
{
  public:
    NetworkGameManager();
    ~NetworkGameManager();

    virtual void writeBytes(const byte* data, int length, int sender);
    virtual int readBytes(byte* data, int maxLength, int * reciever);
    virtual void writeDebug() const;
    virtual void readDebug() const;

    void createEntity( ClassID classID, int owner = 0 );
    void removeEntity( int uniqueID );
    void sendYouAre( int uniqueID, int userID );

    void sync(int uniqueID);

    void sendEntityList(int userID);

  private:
    void requestCreateEntity(ClassID classID);
    void executeCreateEntity(ClassID classID, int uniqueID = 0, int owner = 0);

    void requestRemoveEntity(int uniqueID);
    void executeRemoveEntity(int uniqueID);

    void doCreateEntity(ClassID classID, int uniqueID, int owner);
    void doRemoveEntity(int uniqueID);
    void doRequestSync(int uniqueID, int userID);
    void doYouAre( int uniqueID );

    bool canCreateEntity(ClassID classID);

    void resizeBufferVector(int n);

    inline bool writeToClientBuffer( clientBuffer &cb, byte*data, int length );
    inline bool readFromClientBuffer( clientBuffer &cb, byte*data, int length );

    //helper functions for writeBytes
    inline bool handleRequestCreate( int& i, const byte* data, int length, int sender );
    inline bool handleRequestRemove( int& i, const byte* data, int length, int sender );
    inline bool handleCreateEntity( int& i, const byte* data, int length, int sender );
    inline bool handleRemoveEntity( int& i, const byte* data, int length, int sender );
    inline bool handleCreateEntityList( int& i, const byte* data, int length, int sender );
    inline bool handleRemoveEntityList( int& i, const byte* data, int length, int sender );
    inline bool handleYouAreEntity( int& i, const byte* data, int length, int sender );
    inline bool handleRequestSync( int& i, const byte* data, int length, int sender );

  private:
    std::vector<clientBuffer>     outBuffer;
    clientBuffer                  allOutBuffer;

    int                           newUniqueID;
};

#endif /*_ENTITY_MANGER*/
