/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Wuest
   co-programmer: ...
*/


/* this is for debug output. It just says, that all calls to PRINT() belong to the DEBUG_MODULE_NETWORK module
   For more information refere to https://www.orxonox.net/cgi-bin/trac.cgi/wiki/DebugOutput
*/
#define DEBUG_MODULE_NETWORK

#include "factory.h"
#include "network_stream.h"
#include "converter.h"

/* include your own header */
#include "network_game_manager.h"


/* using namespace std is default, this needs to be here */
using namespace std;

/*!
 * Standard constructor
 */
NetworkGameManager::NetworkGameManager()
{
  /* set the class id for the base object */
  this->setClassID(CL_ENTITY_MANAGER, "EntityManager");

  allOutBuffer.length = 0;

  allOutBuffer.maxLength = 10*1024;

  allOutBuffer.buffer = new byte[10*1024];

  newUniqueID = MAX_CONNECTIONS + 2;
}

/*!
 * Standard destructor
 */
NetworkGameManager::~NetworkGameManager()
{
  for ( int i = 0; i<outBuffer.size(); i++)
  {
    if ( outBuffer[i].buffer )
      delete outBuffer[i].buffer;
  }

  if ( allOutBuffer.buffer )
    delete allOutBuffer.buffer;
}


void NetworkGameManager::writeBytes(const byte* data, int length, int sender)
{
  int i = 0;
  byte b;

  while ( i<length )
  {
    b = data[i++];

    if ( isServer() )
    {
      if ( b == REQUEST_CREATE )
      {
        if ( !handleRequestCreate( i, data, length, sender ) )
          return;
        continue;
      }
      if ( b == REQUEST_REMOVE )
      {
        if ( !handleRequestRemove( i, data, length, sender ) )
          return;
        continue;
      }
    }

    if ( !isServer() )
    {
      if ( b == CREATE_ENTITY )
      {
        if ( !handleCreateEntity( i, data, length, sender ) )
          return;
        continue;
      }
      if ( b == REMOVE_ENTITY )
      {
        if ( !handleRemoveEntity( i, data, length, sender ) )
          return;
        continue;
      }
      if ( b == CREATE_ENTITY_LIST )
      {
        if ( !handleCreateEntityList( i, data, length, sender ) )
          return;
        continue;
      }
      if ( b == REMOVE_ENTITY_LIST )
      {
        if ( !handleRemoveEntityList( i, data, length, sender ) )
          return;
        continue;
      }
      if ( b == YOU_ARE_ENTITY )
      {
        if ( !handleYouAreEntity( i, data, length, sender ) )
          return;
        continue;
      }
    }

    if ( b == REQUEST_SYNC )
    {
      if ( !handleRequestSync( i, data, length, sender ) )
        return;
      continue;
    }

    //if we get her something with data is wrong
    PRINTF(1)("Data is not in the right format! i=%d\n", i);
    return;
  }
}

int NetworkGameManager::readBytes(byte* data, int maxLength, int * reciever)
{
  for ( int i = 0; i<outBuffer.size(); i++ )
  {
    if ( outBuffer[i].length>0 )
    {
      int nbytes = outBuffer[i].length;

      if ( nbytes > maxLength )
      {
        PRINTF(1)("OutBuffer.length (%d) > (%d) networkStreamBuffer.maxLength\n", nbytes, maxLength);
        return 0;
      }

      memcpy(data, outBuffer[i].buffer, nbytes);
      return nbytes;
    }
  }

  int nbytes = allOutBuffer.length;

  if ( nbytes <=0 )
    return 0;

  if ( nbytes > maxLength )
  {
    PRINTF(1)("OutBuffer.length (%d) > (%d) networkStreamBuffer.length\n", nbytes, maxLength);
    return 0;
  }

  memcpy( data, allOutBuffer.buffer, nbytes );
  return nbytes;
}

void NetworkGameManager::writeDebug() const
{
}

void NetworkGameManager::readDebug() const
{
}


/*!
 * Checks whether this is connected to a server or a client
 * and afterwards creates the needed entity
 * @param classID: The ID of the class of which an entity should be created
 */
void NetworkGameManager::createEntity( ClassID classID, int owner )
{
  if ( this->isServer() )
  {
    if ( newUniqueID < 0 )
    {
      PRINTF(1)("Cannot create entity! There are no more uniqueIDs left!\n");
      return;
    }

    this->executeCreateEntity( classID, newUniqueID++, owner );
  }
  else
  {
    this->requestCreateEntity( classID );
  }
}

/*!
 * Checks whether this is connected to a server or a client
 * and afterwards removes the specified entity
 * @param uniqueID: The ID of the entity object which should be removed
 */
void NetworkGameManager::removeEntity(int uniqueID)
{
  if ( this->isServer() )
  {
    this->executeRemoveEntity( uniqueID );
  }
  else
  {
    this->requestRemoveEntity( uniqueID );
  }
}



/*!
 * Creates the needed entity on the server if possible
 * @param classID: The ID of the class of which an entity should be created
 */
void NetworkGameManager::requestCreateEntity(ClassID classID)
{
  byte* bp;
  byte b = REQUEST_CREATE;
  bp = Converter::intToByteArray( classID );

  if ( !writeToClientBuffer( allOutBuffer, &b, 1 ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    return;
  }

  if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete bp;
    return;
  }
  delete bp;
}

/*!
 * Removes the specified entity on the server
 * @param uniqueID: The ID of the entity object which should be removed
 */
void NetworkGameManager::requestRemoveEntity(int uniqueID)
{
  byte* bp;
  byte b = REQUEST_REMOVE;
  bp = Converter::intToByteArray( uniqueID );

  if ( !writeToClientBuffer( allOutBuffer, &b, 1 ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    return;
  }

  if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete bp;
    return;
  }
  delete bp;
}

/*!
 * Creates the needed entity if possible
 * This function is called if this is a server
 * @param classID: The ID of the class of which an entity should be created
 */
void NetworkGameManager::executeCreateEntity(ClassID classID, int uniqueID, int owner)
{
  byte* bp;
  byte b = CREATE_ENTITY;

  if ( !writeToClientBuffer( allOutBuffer, &b, 1 ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    return;
  }

  bp = Converter::intToByteArray( uniqueID );
  if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete bp;
    return;
  }
  delete bp;

  bp = Converter::intToByteArray( uniqueID );
  if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete bp;
    return;
  }
  delete bp;

  bp = Converter::intToByteArray( owner );
  if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete bp;
    return;
  }
  delete bp;

  doCreateEntity( classID, uniqueID, owner );
}

/*!
 * Removes the specified entity
 * This function is called if this is a server
 * @param uniqueID: The ID of the entity object which should be removed
 */
void NetworkGameManager::executeRemoveEntity(int uniqueID)
{
  byte* bp;
  byte b = REMOVE_ENTITY;

  if ( !writeToClientBuffer( allOutBuffer, &b, 1 ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    return;
  }

  bp = Converter::intToByteArray( uniqueID );
  if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete bp;
    return;
  }
  delete bp;

  doRemoveEntity(uniqueID);
}

/*!
 * Checks whether it is possible to create an entity of a given class
 * @return: true if the entity can be created, false otherwise
 */
bool NetworkGameManager::canCreateEntity(ClassID classID)
{
  return true;
}

/*!
 * Sends the Entities to the new connected client
 * @param userID: The ID of the user
 */
void NetworkGameManager::sendEntityList( int userID )
{
  if ( !isServer() )
    return;

  if ( userID > outBuffer.size() )
    resizeBufferVector( userID );

  SynchronizeableList::const_iterator it, e;

  it = this->networkStream->getSyncBegin();
  e = this->networkStream->getSyncEnd();

  byte b = CREATE_ENTITY_LIST;
  byte* ib;

  if ( !writeToClientBuffer( outBuffer[userID], &b, 1 ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    return;
  }

  ib = Converter::intToByteArray( networkStream->getSyncCount() );
  if ( !writeToClientBuffer( outBuffer[userID], ib, INTSIZE ) )
  {
    PRINTF(1)("Could not write to clientBuffer\n");
    delete ib;
    return;
  }
  delete ib;

  while ( it != e )
  {
    ib = Converter::intToByteArray( (*it)->getClassID() );
    if ( !writeToClientBuffer( outBuffer[userID], ib, INTSIZE ) )
    {
      PRINTF(1)("Could not write to clientBuffer\n");
      delete ib;
      return;
    }
    delete ib;

    ib = Converter::intToByteArray( (*it)->getUniqueID() );
    if ( !writeToClientBuffer( outBuffer[userID], ib, INTSIZE ) )
    {
      PRINTF(1)("Could not write to clientBuffer\n");
      delete ib;
      return;
    }
    delete ib;

    ib = Converter::intToByteArray( (*it)->getOwner() );
    if ( !writeToClientBuffer( outBuffer[userID], ib, INTSIZE ) )
    {
      PRINTF(1)("Could not write to clientBuffer\n");
      delete ib;
      return;
    }
    delete ib;

    it++;
  }
}

/**
 * Creates a buffer for user n
 * @param n The ID of the user
 */
void NetworkGameManager::resizeBufferVector( int n )
{
  for ( int i = outBuffer.size(); i<=n; i++)
  {
    clientBuffer outBuf;

    outBuf.length = 0;

    outBuf.maxLength = 5*1024;

    outBuf.buffer = new byte[5*1014];

    outBuffer.push_back(outBuf);
  }
}

/**
 * Creates the entity on this host
 * @param classID: ClassID of the entity to create
 * @param uniqueID: Unique ID to assign to the synchronizeable
 * @param owner: owner of this synchronizealbe
 */
void NetworkGameManager::doCreateEntity( ClassID classID, int uniqueID, int owner )
{
  BaseObject * b = Factory::fabricate( classID );

  if ( !b )
  {
    PRINTF(1)("Could not fabricate Object with classID %d\n", classID);
    return;
  }

  if ( b->isA(CL_SYNCHRONIZEABLE) )
  {
    Synchronizeable * s = dynamic_cast<Synchronizeable*>(b);
    s->setUniqueID( uniqueID );
    s->setOwner( owner );
    this->networkStream->connectSynchronizeable( *s );
  }
  else
  {
    PRINTF(1)("Class with ID %d is not a synchronizeable!", (int)classID);
    delete b;
  }
}

/**
 * Removes a entity on this host
 * @param uniqueID: unique ID assigned with the entity to remove
 */
void NetworkGameManager::doRemoveEntity( int uniqueID )
{
  SynchronizeableList::const_iterator it,e;
  it = this->networkStream->getSyncBegin();
  e = this->networkStream->getSyncEnd();

  while ( it != e )
  {
    if ( (*it)->getUniqueID() == uniqueID )
    {
      delete *it;
      break;
    }
    it++;
  }
}

/**
 * Tell the synchronizeable that a user's synchronizeable is out of sync
 * @param uniqueID: unique ID assigned with the entity which is out of sync
 * @param userID: user ID who's synchronizeable is out of sync
 */
void NetworkGameManager::doRequestSync( int uniqueID, int userID )
{
  SynchronizeableList::const_iterator it,e;
  it = this->networkStream->getSyncBegin();
  e = this->networkStream->getSyncEnd();

  while ( it != e )
  {
    if ( (*it)->getUniqueID() == uniqueID )
    {
      (*it)->requestSync( userID );
      break;
    }
    it++;
  }
}

/**
 * Copies length bytes to the clientBuffer with error checking
 * @param clientBuffer: the clientBuffer to write to
 * @param data: buffer to the data
 * @param length: length of data
 * @return false on error true else
 */
bool NetworkGameManager::writeToClientBuffer( clientBuffer &cb, byte * data, int length )
{
  if ( length > cb.maxLength-cb.length )
  {
    PRINTF(1)("No space left in clientBuffer\n");
    return false;
  }

  memcpy( cb.buffer+cb.length, data, length );
  return true;
}

/**
 * Reads data from clientBuffer with error checking
 * @param clientBuffer: the clientBuffer to read from
 * @param data: pointer to the buffer
 * @param length:
 * @return
 */
bool NetworkGameManager::readFromClientBuffer( clientBuffer &cb, byte * data, int length )
{
  if ( cb.length < length )
  {
    PRINTF(0)("There is not enough data in clientBuffer\n");
    return 0;
  }

  memcpy( data, cb.buffer+cb.length-length, length );
  return true;
}

/**
 * Tells this client that he has to control this entity
 * @param uniqueID: the entity's uniqeID
 */
void NetworkGameManager::doYouAre( int uniqueID )
{
  //TODO: what has to be done
}

/**
 * Tells a remote client that he has to control this entity
 * @param uniqueID: the entity's uniqeID
 * @param userID: the users ID
 */
void NetworkGameManager::sendYouAre( int uniqueID, int userID )
{
  if ( !isServer() )
    return;

  byte* bp;
  byte b = YOU_ARE_ENTITY;

  if ( userID != 0 )
  {
    if ( !writeToClientBuffer( allOutBuffer, &b, 1 ) )
    {
      PRINTF(1)("Could not write to clientBuffer\n");
      return;
    }

    bp = Converter::intToByteArray( uniqueID );
    if ( !writeToClientBuffer( allOutBuffer, bp, INTSIZE ) )
    {
      PRINTF(1)("Could not write to clientBuffer\n");
      delete bp;
      return;
    }
    delete bp;
  }
  else
  {
    doYouAre(uniqueID);
  }
}

bool NetworkGameManager::handleRequestCreate( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read classID from buffer! Not enough data left!\n");
    return false;
  }
  int classID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  createEntity( (ClassID)classID );

  return true;
}

bool NetworkGameManager::handleRequestRemove( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
    return false;
  }
  int uniqueID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  removeEntity( uniqueID );

  return true;
}

bool NetworkGameManager::handleCreateEntity( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read classID from buffer! Not enough data left!\n");
    return false;
  }
  int classID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
    return false;
  }
  int uniqueID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read owner from buffer! Not enough data left!\n");
    return false;
  }
  int owner = Converter::byteArrayToInt( data );
  i += INTSIZE;

  doCreateEntity( (ClassID)classID, uniqueID, owner );

  return true;
}

bool NetworkGameManager::handleRemoveEntity( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
    return false;
  }
  int uniqueID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  doRemoveEntity( uniqueID );

  return true;
}

bool NetworkGameManager::handleCreateEntityList( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read n from buffer! Not enough data left!\n");
    return false;
  }
  int n = Converter::byteArrayToInt( data );
  i += INTSIZE;

  int classID, uniqueID, owner;

  for ( int j = 0; j<n; j++ )
  {

    if ( INTSIZE > length-i )
    {
      PRINTF(1)("Cannot read classID from buffer! Not enough data left!\n");
      return false;
    }
    classID = Converter::byteArrayToInt( data );
    i += INTSIZE;

    if ( INTSIZE > length-i )
    {
      PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
      return false;
    }
    uniqueID = Converter::byteArrayToInt( data );
    i += INTSIZE;

    if ( INTSIZE > length-i )
    {
      PRINTF(1)("Cannot read owner from buffer! Not enough data left!\n");
      return false;
    }
    owner = Converter::byteArrayToInt( data );
    i += INTSIZE;

    doCreateEntity( (ClassID)classID, uniqueID, owner );

  }
  return true;
}

bool NetworkGameManager::handleRemoveEntityList( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read n from buffer! Not enough data left!\n");
    return false;
  }
  int n = Converter::byteArrayToInt( data );
  i += INTSIZE;

  int uniqueID;

  for ( int j = 0; j<n; j++ )
  {

    if ( INTSIZE > length-i )
    {
      PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
      return false;
    }
    uniqueID = Converter::byteArrayToInt( data );
    i += INTSIZE;

    doRemoveEntity( uniqueID );
  }

  return true;
}

bool NetworkGameManager::handleYouAreEntity( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
    return false;
  }
  int uniqueID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  doYouAre( uniqueID );

  return true;
}

bool NetworkGameManager::handleRequestSync( int & i, const byte * data, int length, int sender )
{
  if ( INTSIZE > length-i )
  {
    PRINTF(1)("Cannot read uniqueID from buffer! Not enough data left!\n");
    return false;
  }
  int uniqueID = Converter::byteArrayToInt( data );
  i += INTSIZE;

  doRequestSync( uniqueID, sender );

  return true;
}

