/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "object_manager.h"
#include "class_list.h"

#include "world_entity.h"

#include "shell_command.h"
using namespace std;
SHELL_COMMAND(debug, ObjectManager, debug)
    ->defaultValues(2, NULL, 0);

/**
 * standard constructor
 */
ObjectManager::ObjectManager ()
{
   this->setClassID(CL_OBJECT_MANAGER, "ObjectManager");
   this->setName("ObjectManager");

   pNodeList = NULL;
}

/**
 *  the singleton reference to this class
 */
ObjectManager* ObjectManager::singletonRef = NULL;

/**
   @brief standard deconstructor
 */
ObjectManager::~ObjectManager ()
{
  ObjectManager::singletonRef = NULL;
}


/**
 * @brief moves an Entity from an old list to a new One
 * @param entity the entity to move.
 * @param omList the new List to move the entity to.
 *
 * this will erase the entity from the old list
 */
void ObjectManager::toList (WorldEntity* entity, OM_LIST omList)
{
  if (likely(entity->getOMListNumber() != OM_INIT))
    this->objectLists[entity->getOMListNumber()].erase(entity->getEntityIterator());

  if (likely(omList != OM_INIT))
  {
    this->objectLists[omList].push_back(entity);
    entity->getEntityIterator() = --this->objectLists[omList].end();
    entity->getOMListNumber() = omList;
  }
}


/**
 * @see ObjectManager::toList(WorldEntity* OM_LIST)
 * @param entity the entity to move.
 * @param omList the new List to move the entity to.
 *
 * this function also does a transformation from omList as char* to OM_LIST.
 */
void ObjectManager::toList (WorldEntity* entity, const char* omList)
{
  this->toList(entity, ObjectManager::StringToOMList(omList));
}



/**
 * returns a new List with a list of WorldEntities of distance Radius from center
 */
std::list<WorldEntity*>* ObjectManager::distanceFromObject(const PNode& center, float radius, ClassID classID)
{
  const std::list<BaseObject*>* objectList = ClassList::getList(classID);
  if (objectList != NULL)
  {
    std::list<WorldEntity*>* newList = new std::list<WorldEntity*>;

    list<BaseObject*>::const_iterator node;
    for (node = objectList->begin(); node != objectList->end(); node++)
      if ((dynamic_cast<PNode*>(*node)->getAbsCoor() - center.getAbsCoor()).len() < radius)
        newList->push_back(dynamic_cast<WorldEntity*>(*node));
    return newList;
  }
  return NULL;
}


/**
 * @brief print out nice debug information about Elements in the list OM_LIST
 * @param omList the List to debug.
 */
void ObjectManager::debug(OM_LIST omList) const
{
  if (omList != OM_INIT)
  {
    PRINT(0)(" +ObjectManager-LIST: '%s'------\n", ObjectManager::OMListToString((OM_LIST) omList));
    std::list<WorldEntity*>::const_iterator entity;
    for (entity = this->objectLists[omList].begin(); entity != this->objectLists[omList].end(); entity++)
    {
      PRINT(0)(" | %s::%s\n",(*entity)->getClassName(), (*entity)->getName());
    }
  }
  else
    PRINTF(2)("Invalid query. for OM_INIT-LIST\n");
}


/**
 * @brief prints out very nice debug information
 * @param listName the Name of the list to get Debug information from
 */
void ObjectManager::debug(const char* listName)
{
  PRINT(0)("=ObjectManager-DEBUG=============\n");
  if (listName == NULL || listName[0] == '\0')
    for (unsigned int i = 0; i < OM_SIZE; ++i)
      debug((OM_LIST) i);
  else
    debug(ObjectManager::StringToOMList(listName));
  PRINT(0)("=========================== OM ==\n");
}



/**
 * @brief transforms an omList into a String (usefull for debugging).
 * @param omList the OM_LIST to be transformed into a String.
 * @returns the String transformed from omList.
 */
const char* ObjectManager::OMListToString(OM_LIST omList)
{
  return ObjectManager::objectManagerListNames[omList];
}



/**
 * @brief transforms a String into an OM_LIST (usefull for debugging/Loading).
 * @param listName the OM_LIST-name to be transformed into an OM_LIST.
 * @returns the OM_LIST transformed from listName. or the default, if not found or NULL.
 */
OM_LIST ObjectManager::StringToOMList(const char* listName)
{
  if (unlikely(listName == NULL)) return OM_DEFAULT_LIST;

  for(unsigned int i = 0; i < OM_SIZE; ++i) {
    if(!strcmp(listName, ObjectManager::objectManagerListNames[i])) {
      return (OM_LIST)i;
    }
  }
  return OM_DEFAULT_LIST;
}



const char* ObjectManager::objectManagerListNames[] = {
    "null",
    "dead",
    "environ-notick",
    "environ",
    "common",

    "group00",
    "group00-proj",
    "group01",
    "group01-proj",
    "group02",
    "group02-proj",
    "group03",
    "group03-proj",
    "group04",
    "group04-proj",
    "group05",
    "group05-proj",
    "group06",
    "group06-proj",
    "group07",
    "group07-proj",
    "group08",
    "group08-proj",
    "group09",
    "group09-proj",
    "group10",
    "group10-proj",
    "group11",
    "group11-proj",
    "group12",
    "group12-proj",
    "group13",
    "group13-proj",
    "group14",
    "group14-proj",
    "group15",
    "group15-proj"
};
