/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
   co-programmer: Benjamin Grauer
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD

#include "network_world.h"

#include "shell_command.h"

#include "state.h"

#include "p_node.h"
#include "null_parent.h"
#include "pilot_node.h"
#include "world_entity.h"
#include "player.h"
#include "camera.h"
#include "environment.h"
#include "skysphere.h"
#include "skybox.h"
#include "satellite.h"
#include "test_entity.h"
#include "terrain.h"
#include "light.h"
#include "load_param.h"
#include "shell.h"

#include "garbage_collector.h"
#include "fast_factory.h"
#include "animation_player.h"
#include "particle_engine.h"
#include "graphics_engine.h"
#include "physics_engine.h"
#include "fields.h"

#include "md2Model.h"

#include "glmenu_imagescreen.h"
#include "list.h"
#include "game_loader.h"

#include "animation3d.h"

#include "substring.h"

#include "factory.h"

#include "weapons/projectile.h"
#include "event_handler.h"
#include "sound_engine.h"
#include "ogg_player.h"

#include "class_list.h"

#include "cd_engine.h"
#include "npcs/npc_test1.h"
#include "shader.h"

#include "playable.h"
#include "network_manager.h"
#include "playable.h"


SHELL_COMMAND(speed, NetworkWorld, setSpeed);
SHELL_COMMAND(togglePNodeVisibility, NetworkWorld, togglePNodeVisibility);
SHELL_COMMAND(toggleBVVisibility, NetworkWorld, toggleBVVisibility);

using namespace std;

//! This creates a Factory to fabricate a NetworkWorld
CREATE_FACTORY(NetworkWorld, CL_WORLD);

NetworkWorld::NetworkWorld(const TiXmlElement* root)
{
  this->constuctorInit("", -1);
  this->path = NULL;

  this->loadParams(root);
}

/**
  *  create a new NetworkWorld

    This creates a new empty world!
*/
NetworkWorld::NetworkWorld (const char* name)
{
  this->path = NULL;
  this->constuctorInit(name, -1);
}

/**
 *  creates a new NetworkWorld...
 * @param worldID with this ID
*/
NetworkWorld::NetworkWorld (int worldID)
{
  this->path = NULL;
  this->constuctorInit(NULL, worldID);
}

/**
 *  remove the NetworkWorld from memory

    delete everything explicitly, that isn't contained in the parenting tree!
    things contained in the tree are deleted automaticaly
 */
NetworkWorld::~NetworkWorld ()
{
  delete this->shell;
  PRINTF(3)("NetworkWorld::~NetworkWorld() - deleting current world\n");


  // here everything that is alocated by the NetworkWorld is deleted
  delete this->entities;
  State::setWorldEntityList(NULL);

  delete this->localPlayer;

  // delete all the initialized Engines.
  FastFactory::flushAll(true);
  delete LightManager::getInstance();
  delete ParticleEngine::getInstance();
  delete AnimationPlayer::getInstance();
  delete PhysicsEngine::getInstance();

  // external engines initialized by the orxonox-class get deleted
  SoundEngine::getInstance()->flushAllBuffers();
  SoundEngine::getInstance()->flushAllSources();


  // erease everything that is left.
  delete NullParent::getInstance();

  Shader::suspendShader();

  // unload the resources !!
  ResourceManager::getInstance()->unloadAllByPriority(RP_LEVEL);

  delete[] this->path;
}

/**
 * initializes the world.
 * @param name the name of the world
 * @param worldID the ID of this world
 *
 * set all stuff here that is world generic and does not use to much memory
 * because the real init() function StoryEntity::init() will be called
 * shortly before start of the game.
 * since all worlds are initiated/referenced before they will be started.
 * NO LEVEL LOADING HERE - NEVER!
*/
void NetworkWorld::constuctorInit(const char* name, int worldID)
{
  this->setClassID(CL_WORLD, "NetworkWorld");

  this->setName(name);
  this->debugNetworkWorldNr = worldID;
  this->gameTime = 0.0f;
  this->setSpeed(1.0);
  this->music = NULL;
  this->shell = NULL;
  this->entities = NULL;
  this->localPlayer = NULL;
  this->localCamera = NULL;

  this->showPNodes = false;
  this->showBV = false;
}

/**
 * loads the parameters of a NetworkWorld from an XML-element
 * @param root the XML-element to load from
 */
void NetworkWorld::loadParams(const TiXmlElement* root)
{
  PRINTF(4)("Creating a NetworkWorld\n");

  LoadParam(root, "identifier", this, NetworkWorld, setStoryID)
    .describe("Sets the StoryID of this world");

  LoadParam(root, "nextid", this, NetworkWorld, setNextStoryID)
    .describe("Sets the ID of the next world");

  LoadParam(root, "path", this, NetworkWorld, setPath)
    .describe("The Filename of this NetworkWorld (relative from the data-dir)");
}

/**
 * this is executed just before load
 *
 * since the load function sometimes needs data, that has been initialized
 * before the load and after the proceeding storyentity has finished
*/
ErrorMessage NetworkWorld::preLoad()
{
  State::setWorldEntityList(this->entities = new tList<WorldEntity>());
  this->cycle = 0;

  /* init the world interface */
  this->shell = new Shell();

  LightManager::getInstance();
  NullParent::getInstance ();

  AnimationPlayer::getInstance(); // initializes the animationPlayer
  ParticleEngine::getInstance();
  PhysicsEngine::getInstance();

  this->localCamera = new Camera();
  this->localCamera->setName ("NetworkWorld-Camera");

  State::setCamera(this->localCamera, this->localCamera->getTarget());

  GraphicsEngine::getInstance()->displayFPS(true);

  CDEngine::getInstance()->setEntityList( this->entities);
}


/**
 *  loads the NetworkWorld by initializing all resources, and set their default values.
*/
ErrorMessage NetworkWorld::load()
{
  PRINTF(3)("> Loading world: '%s'\n", getPath());
  TiXmlElement* element;
  GameLoader* loader = GameLoader::getInstance();

  if( getPath() == NULL)
    {
      PRINTF(1)("NetworkWorld has no path specified for loading");
      this->loadDebugNetworkWorld(this->getStoryID());
      return (ErrorMessage){213,"Path not specified","NetworkWorld::load()"};
    }

  TiXmlDocument* XMLDoc = new TiXmlDocument( getPath());
  // load the campaign document
  if( !XMLDoc->LoadFile())
  {
    // report an error
    PRINTF(1)("loading XML File: %s @ %s:l%d:c%d\n", XMLDoc->ErrorDesc(), this->getPath(), XMLDoc->ErrorRow(), XMLDoc->ErrorCol());
    delete XMLDoc;
    return (ErrorMessage){213,"XML File parsing error","NetworkWorld::load()"};
  }

  // check basic validity
  TiXmlElement* root = XMLDoc->RootElement();
  assert( root != NULL);

  if( root == NULL || root->Value() == NULL || strcmp( root->Value(), "WorldDataFile"))
    {
      // report an error
      PRINTF(1)("Specified XML File is not an orxonox world data file (WorldDataFile element missing)\n");
      delete XMLDoc;
      return (ErrorMessage){213,"Path not a WorldDataFile","NetworkWorld::load()"};
    }


  // load the parameters
  // name
  const char* string = grabParameter( root, "name");
  if( string == NULL)
    {
      PRINTF(2)("World is missing a proper 'name'\n");
      this->setName("Unknown");
    }
  else
    {
      this->setName(string);
    }


  ////////////////
  // LOADSCREEN //
  ////////////////
  element = root->FirstChildElement("LoadScreen");
  if (element == NULL)
    {
      PRINTF(2)("no LoadScreen specified, loading default\n");

      glmis->setBackgroundImage("pictures/load_screen.jpg");
      this->glmis->setMaximum(8);
      this->glmis->draw();
    }
  else
    {
      this->glmis->loadParams(element);
      this->glmis->draw();
    }
  this->glmis->draw();



  ////////////////////////////
  // Loading Spawning Point //
  ////////////////////////////
  element = root->FirstChildElement("SpawningPoints");
  if( element == NULL)
  {
    PRINTF(1)("NetworkWorld is missing 'SpawningPoints'\n");
  }
  else
  {
    element = element->FirstChildElement();
      // load Players/Objects/Whatever
    PRINTF(4)("Loading Spawning Points\n");
    while( element != NULL)
    {
      BaseObject* created = Factory::fabricate(element);
      if( created != NULL )
      {
        if(created->isA(CL_SPAWNING_POINT))
          this->spawn(dynamic_cast<WorldEntity*>(created));
        printf("Created a Spawning Point %s: %s\n", created->getClassName(), created->getName());
      }


      element = element->NextSiblingElement();
      glmis->step(); //! @todo temporary
    }
    PRINTF(4)("Done loading NetworkWorldEntities\n");
  }


  ////////////////////////
  // find WorldEntities //
  ////////////////////////
  if( NetworkManager::getInstance()->isGameServer())
  {}

  element = root->FirstChildElement("WorldEntities");
  if( element == NULL)
  {
    PRINTF(1)("NetworkWorld is missing 'WorldEntities'\n");
  }
  else
  {
    element = element->FirstChildElement();
      // load Players/Objects/Whatever
    PRINTF(4)("Loading NetworkWorldEntities\n");
    while( element != NULL)
    {
      if( NetworkManager::getInstance()->isGameServer() || !strcmp( element->Value(), "SkyBox") || !strcmp( element->Value(), "Terrain")
          || !strcmp( element->Value(), "SpaceShip"))
      {
        BaseObject* created = Factory::fabricate(element);
        if( created != NULL )
        {
          if(created->isA(CL_WORLD_ENTITY))
            this->spawn(dynamic_cast<WorldEntity*>(created));
          printf("Created a %s: %s\n", created->getClassName(), created->getName());
        }

          // if we load a 'Player' we use it as localPlayer


          //todo do this more elegant
        if( element->Value() != NULL && !strcmp( element->Value(), "SkyBox"))
          sky = dynamic_cast<SkyBox*>(created);
        if( element->Value() != NULL && !strcmp( element->Value(), "Terrain"))
        {
          terrain = dynamic_cast<Terrain*>(created);
          CDEngine::getInstance()->setTerrain(terrain);
        }

      }
      element = element->NextSiblingElement();
      glmis->step(); //! @todo temporary
      PRINTF(4)("Done loading NetworkWorldEntities\n");
    }
  }


    //////////////////////////////
    // LOADING ADDITIONAL STUFF //
    //////////////////////////////

    LoadParamXML(root, "LightManager", LightManager::getInstance(), LightManager, loadParams);

   LoadParamXML(root, "ParticleEngine", ParticleEngine::getInstance(), ParticleEngine, loadParams);
//   LoadParamXML(root, "PhysicsEngine", PhysicsEngine::getInstance(), PhysicsEngine, loadParams);

  // free the XML data

  delete XMLDoc;
  /* GENERIC LOADING PROCESS FINISHED */


  // Create a Player
  this->localPlayer = new Player();

  Playable* playable;
  const list<BaseObject*>* playableList = ClassList::getList(CL_PLAYABLE);
  if (playableList != NULL)
  {
    playable = dynamic_cast<Playable*>(playableList->front());
    this->localPlayer->setControllable(playable);
  }

  // bind camera
  playable->addChild (this->localCamera);

//   //localCamera->setParent(TrackNode::getInstance());
//  tn->addChild(this->localCamera);
  localCamera->setClipRegion(1, 10000.0);
  localCamera->lookAt(playable);
//  this->localPlayer->setParentMode(PNODE_ALL);
  if (sky != NULL)
  {
    this->sky->setParent(this->localCamera);
    this->sky->setParentMode(PNODE_MOVEMENT);
  }

  // initialize debug coord system
  objectList = glGenLists(1);
  glNewList (objectList, GL_COMPILE);

  glEndList();

  SoundEngine::getInstance()->setListener(this->localCamera);



  ////////////
  // STATIC //
  ////////////


//   TestEntity* testEntity = new TestEntity();
//   testEntity->setRelCoor(Vector(570, 10, -15));
//   testEntity->setRelDir(Quaternion(M_PI, Vector(0, 1, 0)));
//   this->spawn(testEntity);

  for(int i = 0; i < 100; i++)
  {
    WorldEntity* tmp = new NPCTest1();
    char npcChar[10];
    sprintf (npcChar, "NPC_%d", i);
        tmp->setName(npcChar);
    tmp->setAbsCoor(((float)rand()/RAND_MAX) * 5000, 50/*+ (float)rand()/RAND_MAX*20*/, ((float)rand()/RAND_MAX -.5) *30);
    this->spawn(tmp);
  }

  this->music = NULL;//(OggPlayer*)ResourceManager::getInstance()->load("sound/00-luke_grey_-_hypermode.ogg", OGG, RP_LEVEL);
  //music->playback();
}



/**
 * creates a debug world: only for experimental stuff
*/
void NetworkWorld::loadDebugNetworkWorld(int worldID)
{
  /*monitor progress*/
  this->glmis->step();
  // stuff beyond this point remains to be loaded properly

  // LIGHT initialisation
  LightManager::getInstance()->setAmbientColor(.1,.1,.1);
//  LightManager::getInstance()->addLight();
  LightManager::getInstance()->debug();

  switch(this->debugNetworkWorldNr)
    {
      /*
        this loads the hard-coded debug world. this only for simplicity and will be
        removed by a reald world-loader, which interprets a world-file.
        if you want to add an own debug world, just add a case DEBUG_WORLD_[nr] and
        make whatever you want...
      */
    case DEBUG_WORLD_0:
      {
        LightManager::getInstance()->getLight()->setAbsCoor(-5.0, 10.0, -40.0);
        /*monitor progress*/
        this->glmis->step();

        // bind camera
        this->localCamera = new Camera();
        this->localCamera->setName ("camera");
        /*monitor progress*/
        this->glmis->step();


        // Create SkySphere
        this->sky = new Skysphere("pictures/sky-replace.jpg");
        this->sky->setName("SkySphere");
        this->spawn(this->sky);
        this->localCamera->addChild(this->sky);
        this->sky->setParentMode(PNODE_MOVEMENT);
        /*monitor progress*/
        this->glmis->step();


        terrain = new Terrain("worlds/newGround.obj");
        terrain->setRelCoor(Vector(0,-10,0));
        this->spawn(terrain);
        /*monitor progress*/
        this->glmis->step();

        this->glmis->step();
        break;
      }
    case DEBUG_WORLD_1:
      {

        break;
      }
    case DEBUG_WORLD_2:
      {

        break;
      }
    default:
      break;
    }
}

/**
 *  initializes a new NetworkWorld shortly before start
 *
 * this is the function, that will be loaded shortly before the world is
 * started
*/
ErrorMessage NetworkWorld::init()
{
  this->bPause = false;

  /* update the object position before game start - so there are no wrong coordinates used in the first processing */
  NullParent::getInstance()->updateNode (0.001f);
  NullParent::getInstance()->updateNode (0.001f);

}


/**
 *  starts the NetworkWorld
*/
ErrorMessage NetworkWorld::start()
{
  PRINTF(3)("NetworkWorld::start() - starting current NetworkWorld: nr %i\n", this->debugNetworkWorldNr);
  this->bQuitOrxonox = false;
  this->bQuitCurrentGame = false;
  this->mainLoop();
}

/**
 *  stops the world.

   This happens, when the player decides to end the Level.
*/
ErrorMessage NetworkWorld::stop()
{
  PRINTF(3)("NetworkWorld::stop() - got stop signal\n");
  this->bQuitCurrentGame = true;
}

/**
 *  pauses the Game
*/
ErrorMessage NetworkWorld::pause()
{
  this->isPaused = true;
}

/**
 *  ends the pause Phase
*/
ErrorMessage NetworkWorld::resume()
{
  this->isPaused = false;
}

/**
 *  destroys the NetworkWorld
*/
ErrorMessage NetworkWorld::destroy()
{

}

/**
 *  shows the loading screen
*/
void NetworkWorld::displayLoadScreen ()
{
  PRINTF(3)("NetworkWorld::displayLoadScreen - start\n");

  //GLMenuImageScreen*
  this->glmis = new GLMenuImageScreen();
  this->glmis->setMaximum(8);

  PRINTF(3)("NetworkWorld::displayLoadScreen - end\n");
}

/**
 *  removes the loadscreen, and changes over to the game

   @todo take out the delay
*/
void NetworkWorld::releaseLoadScreen ()
{
  PRINTF(3)("NetworkWorld::releaseLoadScreen - start\n");
  this->glmis->setValue(this->glmis->getMaximum());
  PRINTF(3)("NetworkWorld::releaseLoadScreen - end\n");
  delete this->glmis;
}


/**
 *  gets the list of entities from the world
 * @returns entity list
*/
tList<WorldEntity>* NetworkWorld::getEntities()
{
  return this->entities;
}


/**
 *  this returns the current game time
 * @returns elapsed game time
*/
double NetworkWorld::getGameTime()
{
  return this->gameTime;
}


/**
 *  function to put your own debug stuff into it. it can display informations about
   the current class/procedure
*/
void NetworkWorld::debug()
{
  PRINTF(0)("Printing out the List of alive NetworkWorldEntities:\n");
  tIterator<WorldEntity>* iterator = this->entities->getIterator();
  WorldEntity* entity = iterator->firstElement();
  while( entity != NULL)
  {
    PRINTF(0)("%s::%s\n", entity->getClassName(), entity->getName());
    entity = iterator->nextElement();
  }
  delete iterator;
}


/**
  \brief main loop of the world: executing all world relevant function

  in this loop we synchronize (if networked), handle input events, give the heart-beat to
  all other member-entities of the world (tick to player, enemies etc.), checking for
  collisions drawing everything to the screen.
*/
void NetworkWorld::mainLoop()
{
  this->lastFrame = SDL_GetTicks ();
  PRINTF(3)("NetworkWorld::mainLoop() - Entering main loop\n");

  while( !this->bQuitOrxonox && !this->bQuitCurrentGame) /* @todo implement pause */
    {
      ++this->cycle;
      PRINTF(4)("NetworkWorld::mainloop() - number of entities: %i\n", this->entities->getSize());
      // Network
      this->synchronize ();
      // Process input
      this->handleInput ();
      if( this->bQuitCurrentGame || this->bQuitOrxonox)
          break;
      // Process time
      this->tick ();
      // Process collision
      this->collide ();
      // Update the state
      this->update ();
      // Draw
      this->display ();
    }

  PRINTF(3)("NetworkWorld::mainLoop() - Exiting the main loop\n");
}


/**
 *  synchronize local data with remote data
*/
void NetworkWorld::synchronize ()
{
  // Get remote input
  // Update synchronizables
  NetworkManager::getInstance()->synchronize();
}


/**
 *  run all input processing

   the command node is the central input event dispatcher. the node uses the even-queue from
   sdl and has its own event-passing-queue.
*/
void NetworkWorld::handleInput ()
{
  EventHandler::getInstance()->process();

  // remoteinput
}


/**
 *  advance the timeline

   this calculates the time used to process one frame (with all input handling, drawing, etc)
   the time is mesured in ms and passed to all world-entities and other classes that need
   a heart-beat.
*/
void NetworkWorld::tick ()
{
  Uint32 currentFrame = SDL_GetTicks();
  if(!this->bPause)
    {
      this->dt = currentFrame - this->lastFrame;

      if( this->dt > 10)
        {
          float fps = 1000/dt;

          // temporary, only for showing how fast the text-engine is
          char tmpChar[20];
          sprintf(tmpChar, "fps: %4.0f", fps);
        }
      else
        {
          /* the frame-rate is limited to 100 frames per second, all other things are for
             nothing.
          */
          PRINTF(3)("fps = 1000 - frame rate is adjusted\n");
          SDL_Delay(10-dt);
          this->dt = 10;
        }

      this->dtS = (float)this->dt / 1000.0 * this->speed;
      this->gameTime += this->dtS;

      tIterator<WorldEntity>* iterator = this->entities->getIterator();
      WorldEntity* entity = iterator->firstElement();
      while( entity != NULL)
        {
          entity->tick (this->dtS);
          entity = iterator->nextElement();
        }
      delete iterator;

      /* update tick the rest */
      this->localCamera->tick(this->dtS);
      // tick the engines
      AnimationPlayer::getInstance()->tick(this->dtS);
//      if (this->cycle > 5)
        PhysicsEngine::getInstance()->tick(this->dtS);

      ParticleEngine::getInstance()->tick(this->dtS);
      GarbageCollector::getInstance()->tick(this->dtS);


      /** actualy the Graphics Engine should tick the world not the other way around...
         but since we like the things not too complicated we got it this way around
         until there is need or time to do it the other way around.
         @todo: GraphicsEngine ticks world: separation of processes and data...

        bensch: in my opinion the GraphicsEngine could draw the world, but not tick it,
         beceause graphics have nothing(or at least not much) to do with Motion.
      */
      GraphicsEngine::getInstance()->tick(this->dtS);
    }
  this->lastFrame = currentFrame;
}


/**
 *  this function gives the world a consistant state

   after ticking (updating the world state) this will give a constistant
   state to the whole system.
*/
void NetworkWorld::update()
{
  GarbageCollector::getInstance()->update();
  GraphicsEngine::getInstance()->update(this->dtS);
  NullParent::getInstance()->updateNode (this->dtS);

  SoundEngine::getInstance()->update();
  //music->update();
}


void NetworkWorld::collide()
{
  CDEngine::getInstance()->checkCollisions();
}

/**
 *  render the current frame

   clear all buffers and draw the world
*/
void NetworkWorld::display ()
{
  // clear buffer
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // set camera
  this->localCamera->apply ();
  // draw world
  this->draw();
  // draw HUD
  /** @todo draw HUD */
  // flip buffers
  GraphicsEngine::swapBuffers();
  //SDL_Surface* screen = Orxonox::getInstance()->getScreen ();
  //SDL_Flip (screen);
}


/**
 *  runs through all entities calling their draw() methods
 */
void NetworkWorld::draw ()
{
  /* draw entities */
  WorldEntity* entity;
  glLoadIdentity();
  tIterator<WorldEntity>* iterator = this->entities->getIterator();
  entity = iterator->firstElement();
  while( entity != NULL )
  {
    if( entity->isVisible() ) entity->draw();
    if( unlikely( this->showBV)) entity->drawBVTree(3, 226);  // to draw the bounding boxes of the objects at level 2 for debug purp
    entity = iterator->nextElement();
  }
  delete iterator;

  glCallList (objectList);

  ParticleEngine::getInstance()->draw();

  if (unlikely(this->showPNodes))
    NullParent::getInstance()->debugDraw(0);

  GraphicsEngine::getInstance()->draw();
  //TextEngine::getInstance()->draw();
}

/**
 *  add and spawn a new entity to this world
 * @param entity to be added
*/
void NetworkWorld::spawn(WorldEntity* entity)
{
  this->entities->add (entity);
  entity->postSpawn ();
}


/**
 *  add and spawn a new entity to this world
 * @param entity to be added
 * @param absCoor At what coordinates to add this entity.
 * @param absDir In which direction should it look.
*/
void NetworkWorld::spawn(WorldEntity* entity, Vector* absCoor, Quaternion* absDir)
{
  this->entities->add (entity);

  entity->setAbsCoor (*absCoor);
  entity->setAbsDir (*absDir);

  entity->postSpawn ();
}


/**
 *  add and spawn a new entity to this world
 * @param entity to be added
 * @param entity to be added to (PNode)
 * @param At what relative  coordinates to add this entity.
 * @param In which relative direction should it look.
*/
void NetworkWorld::spawn(WorldEntity* entity, PNode* parentNode,
                  Vector* relCoor, Quaternion* relDir)
{
  if( parentNode != NULL)
    {
      parentNode->addChild (entity);

      entity->setRelCoor (*relCoor);
      entity->setRelDir (*relDir);

      this->entities->add (entity);

      entity->postSpawn ();
    }
}

void NetworkWorld::setPath( const char* name)
{
  if (this->path)
    delete this->path;
  if (ResourceManager::isFile(name))
  {
    this->path = new char[strlen(name)+1];
    strcpy(this->path, name);
  }
  else
    {
      this->path = new char[strlen(ResourceManager::getInstance()->getDataDir()) + strlen(name) +1];
      sprintf(this->path, "%s%s", ResourceManager::getInstance()->getDataDir(), name);
    }
}

const char* NetworkWorld::getPath( void)
{
  return path;
}
