/*!
 * @file class_list.h
  *  Definition of the Class List, that handles a Class-Specific-Control structure

 */

#ifndef _CLASS_LIST_H
#define _CLASS_LIST_H

#include "class_id.h"
#include <list>
#ifndef NULL
#define NULL     0    //!< NULL
#endif

// FORWARD DECLARATION
class BaseObject;

//! A class that handles Pointers to Objects of all type.
/**
 * here all the Pointers to all the Object of orxonox are stored, that implement BaseObject
 * for now.
 * You can get Any Object's Reference to BaseObject with dynamic_cast<T>(ClassList::getObject(name, CL_T_NAME));
 *  where: T: is the Class to cast to,
 *   name: the name of the Object (not className)
 *   CL_T_NAME: the class Identifier, (if CL_NULL or nothing it will take longer, because all BaseObject's are searched through)
 *
 * There is also the exists-function, that just checks, if a Reference is still in existence.
 *
 * @see ClassID, BaseObject, dynamic_cast
 */
class ClassList {

  public:
    ClassList(const long& classID, const char* className);
    virtual ~ClassList();

    // STATIC FUNCTIONS
    static void                           addToClassList(BaseObject* objectPointer, ClassID classID, const char* className);
    static void                           removeFromClassList(BaseObject* objectPointer);

    static const std::list<BaseObject*>*  getList(ClassID classID = CL_NULL);// { return (ClassList* fl = ClassList::getClassList(classID) != NULL)? &(fl->objectList) : NULL; };
    static const std::list<BaseObject*>*  getList(const char* className); // { return (ClassList* fl = ClassList::getClassList(className) != NULL)? &(fl->objectList) : NULL;  };
    static const std::list<const char*>*  getClassNames();
    static BaseObject*                    getObject(const char* name, ClassID classID = CL_NULL);
    static bool                           exists(const BaseObject* object, ClassID classID = CL_NULL);

    void                                  sendBack(std::list<BaseObject*>::const_iterator it);

    static void                           whatIs(const BaseObject* object);

    static const char*                    IDToString(ClassID classID = CL_NULL);
    static long                           StringToID(const char* className);
    static void                           debug(unsigned int debugLevel = 0, long classID = CL_NULL);
    static void                           debugS(const char* className = NULL, unsigned int debugLevel = 0);

    inline bool                           operator==(ClassID classID) { return (this->classID == classID); };
    bool                                  operator==(const char* className);

  private:

    static ClassList*                     getClassList(ClassID classID);
    static ClassList*                     getClassList(const char* className);

  private:

    long                            classID;                //!< ClassID stored in this ClassList \see ClassID
    const char*                     className;              //!< Name of the Class Stored here

    std::list<BaseObject*>          objectList;             //!< A list of Objects belonging to this Class

    // STATIC MEMBERS
    static std::list<ClassList>*    classList;              //!< The first Class in the List
    static std::list<const char*>   classNames;             //!< a List of all Names of all classes, that have registered so far.
};

#endif /* _CLASS_LIST_H */
