/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "vertex_array_model.h"

#include "stdlibincl.h"
#include <stdarg.h>

using namespace std;

/////////////
/// MODEL ///
/////////////
/**
 * @brief Creates a 3D-VertexArrayModel.
 *
 * assigns it a Name and a Type
 */
VertexArrayModel::VertexArrayModel()
{
  this->setClassID(CL_MODEL, "VertexArrayModel");

  this->finalized = false;
}


/**
 * @brief deletes an VertexArrayModel.
 *
 * Looks if any from model allocated space is still in use, and if so deleted it.
 */
VertexArrayModel::~VertexArrayModel()
{
  PRINTF(4)("Deleting VertexArrayModel ");
  if (this->getName())
  {
    PRINT(4)("%s\n", this->getName());
  }
  else
  {
    PRINT(4)("\n");
  }
}


/**
 * @brief Finalizes an Object. This can be done outside of the Class.
 */
void VertexArrayModel::finalize()
{
  // this creates the display List.
  this->importToVertexArray();

  this->finalized = true;
}


//////////
// DRAW //
//////////
/**
 *  Draws the VertexArrayModels of all Groups.
   It does this by just calling the Lists that must have been created earlier.
*/
void VertexArrayModel::draw () const
{
  PRINTF(4)("drawing the 3D-VertexArrayModels\n");

  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  //  glEnableClientState(GL_INDEX_ARRAY);

  glVertexPointer(3, GL_FLOAT, 0, this->vertices.getArray());
  glNormalPointer(GL_FLOAT, 0, this->normals.getArray());
  glTexCoordPointer(2, GL_FLOAT, 0, this->texCoords.getArray());  

  printf("%d\n", this->indices.getCount());
  glDrawElements(GL_TRIANGLE_STRIP,
		 this->indices.getCount(),
		 GL_UNSIGNED_BYTE,
		 this->indices.getArray());



  /*  const GLfloat* pVertices = NULL;
      const GLfloat* pNorm = NULL;
      
      glBegin(GL_TRIANGLES);
      for( int i = 0; i < this->triangleCount; ++i)
      {
      //printf("int i = %i\n", i);
      pNorm = &this->normals->getArray()[this->triangles[i].indexToNormals[0]];
      pVertices = &this->vertices->getArray()[this->triangles[i].indexToVertices[0]];
      glNormal3f(pNorm[0], pNorm[1], pNorm[2]);
      glVertex3f(pVertices[0], pVertices[1], pVertices[2]);
      
      pNorm = &this->normals->getArray()[this->triangles[i].indexToNormals[1]];
      pVertices = &this->vertices->getArray()[this->triangles[i].indexToVertices[1]];
      glNormal3f(pNorm[0], pNorm[1], pNorm[2]);
      glVertex3f(pVertices[0], pVertices[1], pVertices[2]);
      
      pNorm = &this->normals->getArray()[this->triangles[i].indexToNormals[2]];
      pVertices = &this->vertices->getArray()[this->triangles[i].indexToVertices[2]];
      glNormal3f(pNorm[0], pNorm[1], pNorm[2]);
      glVertex3f(pVertices[0], pVertices[1], pVertices[2]);
      
      }
      glEnd();
  */
}


//////////
// MESH //
//////////
/**
 * @brief parses a vertex-String
 * @param x the X-coordinate of the Vertex to add.
 * @param y the Y-coordinate of the Vertex to add.
 * @param z the Z-coordinate of the Vertex to add.
 */
void VertexArrayModel::addVertex(float x, float y, float z)
{
  this->vertices.addEntry(x, y, z);
  this->pModelInfo.numVertices++;
}


/**
 * @brief adds a VertexNormal.
 * @param x The x coordinate of the Normal.
 * @param y The y coordinate of the Normal.
 * @param z The z coordinate of the Normal.
 *
 * If a vertexNormal line is found this function will inject it into the vertexNormal-Array
 */
void VertexArrayModel::addNormal(float x, float y, float z)
{
  this->normals.addEntry(x, y, z);
  this->pModelInfo.numNormals++;
}


/**
 *  adds a Texture Coordinate
 * @param u The u coordinate of the TextureCoordinate.
 * @param v The y coordinate of the TextureCoordinate.
 *
 *  If a TextureCoordinate line is found this function will inject it into the TextureCoordinate-Array
 */
void VertexArrayModel::addTexCoor(float u, float v)
{
  this->texCoords.addEntry(u);
  this->texCoords.addEntry(v);
  this->pModelInfo.numTexCoor++;
}


/**
 *  adds a new Face
 * @param faceElemCount the number of Vertices to add to the Face.
 * @param type The information Passed with each Vertex
*/
void VertexArrayModel::addIndice(GLubyte indice)
{
  this->indices.addEntry(indice);

}


////////////
// openGL //
////////////
/**
 *  reads and includes the Faces/Materials into the openGL state Machine
*/
void VertexArrayModel::importToVertexArray()
{
  // finalize the Arrays
  this->vertices.finalizeArray();
  this->texCoords.finalizeArray();
  this->normals.finalizeArray();
  this->indices.finalizeArray();

  //  glEnableClientState(GL_VERTEX_ARRAY);
  //  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  //  glEnableClientState(GL_NORMAL_ARRAY);
  //  glEnableClientState(GL_INDEX_ARRAY);
}



/////////////
// TESTING //
/////////////
/**
 *  Includes a default model
 *
 * This will inject a Cube, because this is the most basic model.
 */
void VertexArrayModel::cubeModel()
{
  unsigned int i, j;
  for (i = 0; i < 20; i++)
    {
      for (j = 0; j < 20; j++)
	{
	  this->addVertex(i* 50, .5, (j)*50);
	  this->addNormal(0, 1, 0);
	  this->addTexCoor((float)i/20.0, (float)j/20.0);

	}
    }
  for (i = 0; i < 20; i++)
    {
      this->addIndice(i);
      this->addIndice(i+20);
    }

  /*
  this->addVertex (-0.5, -0.5, 0.5);
  this->addVertex (0.5, -0.5, 0.5);
  this->addVertex (-0.5, 0.5, 0.5);
  this->addVertex (0.5, 0.5, 0.5);
  this->addVertex (-0.5, 0.5, -0.5);
  this->addVertex (0.5, 0.5, -0.5);
  this->addVertex (-0.5, -0.5, -0.5);
  this->addVertex (0.5, -0.5, -0.5);

  this->addTexCoor (0.0, 0.0);
  this->addTexCoor (1.0, 0.0);
  this->addTexCoor (0.0, 1.0);
  this->addTexCoor (1.0, 1.0);
  this->addTexCoor (0.0, 2.0);
  this->addTexCoor (1.0, 2.0);
  this->addTexCoor (0.0, 3.0);
  this->addTexCoor (1.0, 3.0);
  this->addTexCoor (0.0, 4.0);
  this->addTexCoor (1.0, 4.0);
  this->addTexCoor (2.0, 0.0);
  this->addTexCoor (2.0, 1.0);
  this->addTexCoor (-1.0, 0.0);
  this->addTexCoor (-1.0, 1.0);

  this->addNormal (0.0, 0.0, 1.0);
  this->addNormal (0.0, 0.0, 1.0);
  this->addNormal (0.0, 0.0, 1.0);
  this->addNormal (0.0, 0.0, 1.0);
  this->addNormal (0.0, 1.0, 0.0);
  this->addNormal (0.0, 1.0, 0.0);
  this->addNormal (0.0, 1.0, 0.0);
  this->addNormal (0.0, 1.0, 0.0);
  this->addNormal (0.0, 0.0, -1.0);
  this->addNormal (0.0, 0.0, -1.0);
  this->addNormal (0.0, 0.0, -1.0);
  this->addNormal (0.0, 0.0, -1.0);
  this->addNormal (0.0, -1.0, 0.0);
  this->addNormal (0.0, -1.0, 0.0);
  this->addNormal (0.0, -1.0, 0.0);
  this->addNormal (0.0, -1.0, 0.0);
  this->addNormal (1.0, 0.0, 0.0);
  this->addNormal (1.0, 0.0, 0.0);
  this->addNormal (1.0, 0.0, 0.0);
  this->addNormal (1.0, 0.0, 0.0);
  this->addNormal (-1.0, 0.0, 0.0);
  this->addNormal (-1.0, 0.0, 0.0);
  this->addNormal (-1.0, 0.0, 0.0);
  this->addNormal (-1.0, 0.0, 0.0);
  */
}
