/*!
 * @file ini_parser.h
 * A small ini file parser
 *
 * Can be used to find a defined [Section] in an ini file and get the VarName = Value entries
 */

#ifndef _INI_PARSER_H
#define _INI_PARSER_H

#define PARSELINELENGHT     512       //!< how many chars to read at once
#ifndef NULL
 #define NULL 0x0                     //!< NULL
#endif

#include <list>

//! ini-file parser
/**
 * This class can be used to load an initializer file and parse it's contents for variablename=value pairs.
 */
class IniParser
{
  private:
    ////////////////////////////////////
    //! a struct for Entries in the Parser's File's Sections
    struct IniEntry
    {
      char*               comment;  //!< A Comment that is appendet to the Top of this Entry.
      char*               name;     //!< name of a given Entry
      char*               value;    //!< value of a given Entry
    };

    //! a struct for Sections in the Parser's file
    struct IniSection
    {
      char*               comment;  //!< A Comment that is appendet to the Top of this Section.
      char*               name;     //!< name of a given section
      std::list<IniEntry> entries;  //!< a list of entries for this section
    };
    ////////////////////////////////////

  public:
    IniParser (const char* filename = NULL);
    ~IniParser ();

    /** @returns true if the file is opened, false otherwise*/
    bool isOpen() const { return (this->fileName != NULL)? true : false; };
    /** @returns the fileName we have opened. */
    const char* getFileName() const { return this->fileName; };

    bool readFile(const char* fileName);
    bool writeFile(const char* fileName) const;

    void setFileComment(const char* fileComment);
    const char* getFileComment() const { return this->comment; };

    bool addSection(const char* sectionName);
    bool getSection(const char* sectionName);
    void setSectionComment(const char* comment, const char* sectionName);
    const char* getSectionComment(const char* sectionNane) const;

    // iterate through sections with these Functions
    void firstSection();
    const char* nextSection();


    bool addVar(const char* entryName, const char* value, const char* sectionName = NULL);
    const char* getVar(const char* entryName, const char* sectionName, const char* defaultValue = "") const;
    const char* setEntryComment(const char* comment, const char* entryName, const char* sectionName);
    const char* getEntryComment(const char* entryName, const char* sectionName) const;

    // iterate Through Variables with these Functions.
    void firstVar();
    bool nextVar();


    // retrieving functions when iterating.
    const char* getCurrentSection() const;
    const char* getCurrentName() const;
    const char* getCurrentValue() const;


    // maintenance.
    void debug() const;


  private:
    void deleteSections();
    void setFileName(const char* fileName);

    std::list<IniSection>::const_iterator getSectionIT(const char* sectionName) const;
    std::list<IniEntry>::const_iterator getEntryIT(const char* entryName, const char* sectionName = NULL) const;

  private:
    char*                            fileName;        //!< The name of the File that was parsed.
    char*                            comment;         //!< A Comment for the header of this File.
    std::list<IniSection>            sections;        //!< a list of all stored Sections of the Parser
    std::list<IniSection>::iterator  currentSection;  //!< the current selected Section
    std::list<IniEntry>::iterator    currentEntry;    //!< the current selected entry (in currentSection)

    std::list<char*>                 commentList;     //!< A list of Comments. (this is for temporary saving of Comments, that are inserted in front of Sections/Entries.)
};

#endif /* _INI_PARSER_H */
