/*!
 * @file ini_parser.h
 * A small ini file parser
 *
 * Can be used to find a defined [Section] in an ini file and get the VarName = Value entries
 */

#ifndef _INI_PARSER_H
#define _INI_PARSER_H

#define PARSELINELENGHT     512       //!< how many chars to read at once
#ifndef NULL
 #define NULL 0x0                     //!< NULL
#endif

#include <list>

//! ini-file parser
/**
 * This class can be used to load an initializer file and parse it's contents for variablename=value pairs.
 */
class IniParser
{
  private:
    ////////////////////////////////////
    //! a struct for Entries in the Parser's File's Sections
    struct IniEntry
    {
      char*              name;     //!< name of a given Entry
      char*              value;    //!< value of a given Entry
    };
    //! a struct for Sections in the Parser's file
    struct IniSection
    {
      char*               name;    //!< name of a given section
      std::list<IniEntry> entries; //!< a list of entries for this section
    };
    ////////////////////////////////////

  public:
    IniParser (const char* filename = NULL);
    ~IniParser ();

    bool readFile(const char* fileName);
    bool writeFile(const char* fileName) const;

    bool addSection(const char* sectionName);
    bool getSection(const char* sectionName);

    /** @returns true if the file is opened, false otherwise*/
    bool isOpen() const { return (this->fileName != NULL)? true : false; };

    void firstSection();
    const char* nextSection();

    /** @returns the fileName we have opened. */
    const char* getFileName() const { return this->fileName; };

    bool addVar(const char* entryName, const char* value, const char* sectionName = NULL);
    const char* getVar(const char* entryName, const char* sectionName, const char* defaultValue = "") const;

    void firstVar();
    bool nextVar();

    const char* getCurrentSection() const;
    const char* getCurrentName() const;
    const char* getCurrentValue() const;

    void debug() const;


  private:
    void deleteSections();
    void setFileName(const char* fileName);

  private:
    char*                            fileName;        //!< The name of the File that was parsed.
    std::list<IniSection>            sections;        //!< a list of all stored Sections of the Parser
    std::list<IniSection>::iterator  currentSection;  //!< the current selected Section
    std::list<IniEntry>::iterator    currentEntry;    //!< the current selected entry (in currentSection)
};

#endif /* _INI_PARSER_H */
