/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "font.h"
#include "text.h"

#ifdef HAVE_SDL_IMAGE_H
#include <SDL_image.h>
#else
#include <SDL/SDL_image.h>
#endif
#include "default_font.xpm"

#include "debug.h"
#include "stdlibincl.h"
#include "compiler.h"

using namespace std;

/**
 * constructs a Font out of a TTF-FIle
 * @param fontFile the File to load the font from
 * @param fontSize the Size of the Font in Pixels
 */
Font::Font(const char* fontFile, unsigned int renderSize)
{
  this->init();

  this->renderSize = renderSize;
  this->setStyle("c");

  if (fontFile != NULL)
    this->loadFontFromTTF(fontFile);
}

/**
 * constructs a Font out of an ImageFile
 * @param imageFile the ImageFile to load the Font From.
 */
Font::Font(const char* imageFile)
{
  this->init();
  this->setName(imageFile);
  //  this->setSize(fontSize);
  SDL_Surface* image = NULL;
  if (imageFile != NULL)
    image = IMG_Load(imageFile);
  else
    return;
  if (image != NULL)
  {
    this->loadFontFromSDL_Surface(image);
    SDL_FreeSurface(image);
  }
  else
    PRINTF(1)("loading from surface %s failed: %s\n", imageFile, IMG_GetError());
}

/**
 * constructs a Font
 * @param xpmArray the xpm-ARRAY to load the font from
 */
Font::Font(char** xpmArray)
{
  this->init();
  this->setName("XPM-array-font");
  //  this->setSize(fontSize);
  SDL_Surface* image = NULL;
  if (xpmArray != NULL)
    image = IMG_ReadXPMFromArray(xpmArray);
  if (image != NULL)
  {
    this->loadFontFromSDL_Surface(image);
    SDL_FreeSurface(image);
  }
  else
    PRINTF(1)("loading from surface failed: %s\n", IMG_GetError());
}


/**
 * destructs a font
 * this releases the memory a font uses to be opened.
 * deletes the glLists, and the TTF-handler, if present.
 */
Font::~Font()
{
  // deleting all Glyphs
  if (this->glyphArray != NULL)
  {
    for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
    {
      if (this->glyphArray[i] != NULL)
      {
        delete this->glyphArray[i];
      }
    }
    delete[] this->glyphArray;
  }

  //! @todo check if we really do not need to delete the fastTextureID here.
//   if (this->fastTextureID != 0)
//     if(glIsTexture(this->fastTextureID))
//       glDeleteTextures(1, &this->fastTextureID);

  // erease this font out of the memory.
  if (likely(this->fontTTF != NULL))
    TTF_CloseFont(this->fontTTF);
}

/**
 * initializes a Font (with default values)
 */
void Font::init()
{
  this->setClassID(CL_FONT, "Font");
  // setting default values.
  this->fontTTF = NULL;
  this->glyphArray = NULL;
}


/**
 * sets The Font.
 * @param fontFile The file containing the font.
 * @returns true if loaded, false if something went wrong, or if a font was loaded before.
 */
bool Font::loadFontFromTTF(const char* fontFile)
{
  // checking for existent Font.
  if (this->fontTTF != NULL)
  {
    TTF_CloseFont(this->fontTTF);
    this->fontTTF = NULL;
  }


  this->setName(fontFile);
  this->fontTTF = TTF_OpenFont(this->getName(), this->renderSize);

  if(this->fontTTF != NULL)
  {
    this->createFastTexture();
    return (this->getTexture() != 0);
  }
  else
  {
    PRINTF(1)("TTF_OpenFont: %s\n", TTF_GetError());
    return false;
  }

}

/**
 * loads a font From an XPM-array.
 * @param xpmArray the array of the XPM to load the font from.
 */
bool Font::loadFontFromSDL_Surface(SDL_Surface* surface)
{
  // loading to a texture.
  if(surface == NULL)
    return false;

  if (this->fontTTF != NULL)
  {
    TTF_CloseFont(this->fontTTF);
    this->fontTTF = NULL;
  }
  if (this->prepareSurface(surface))
  {
    this->setTexture(Texture::loadTexToGL(this->getStoredImage()));
  }

  // initializing the Glyphs.
  if (this->glyphArray == NULL)
  {
    float cx,cy;
    Glyph* tmpGlyph;
    this->glyphArray = new Glyph*[FONT_HIGHEST_KNOWN_CHAR];
    for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
    {
      tmpGlyph = this->glyphArray[i] = new Glyph;
      cx=(float)(i%16)/16.0f;                  // X Position Of Current Character
      cy=(float)(i/16)/16.0f;                  // Y Position Of Current Character

      tmpGlyph->texCoord[0] = cx;
      tmpGlyph->texCoord[1] = cx+0.0625f;
      tmpGlyph->texCoord[2] = cy+0.001f;
      tmpGlyph->texCoord[3] = cy+0.0625f;
      tmpGlyph->width = 1;
      tmpGlyph->advance = 1;
      tmpGlyph->bearingX = 1;
      tmpGlyph->bearingY = 1;
      tmpGlyph->height = 1;
    }
  }
  return true;
}


/**
 *  sets a specific renderStyle
 * @param renderStyle the Style to render: a string (char-array) containing:
 *   i: italic, b: bold, u, underline
 */
void Font::setStyle(const char* renderStyle)
{
  this->renderStyle = TTF_STYLE_NORMAL;

  for (int i = 0; i < strlen(renderStyle); i++)
    if (strncmp(renderStyle+i, "b", 1) == 0)
      this->renderStyle |= TTF_STYLE_BOLD;
  else if (strncmp(renderStyle+i, "i", 1) == 0)
    this->renderStyle |= TTF_STYLE_ITALIC;
  else if (strncmp(renderStyle+i, "u", 1) == 0)
    this->renderStyle |= TTF_STYLE_UNDERLINE;

  if (likely(this->fontTTF != NULL))
    TTF_SetFontStyle(this->fontTTF, this->renderStyle);
//  else
//    PRINTF(2)("Font was not initialized, please do so before setting the Font-Style.\n");
}

Font* Font::defaultFont = NULL;

/**
 * creates and exports an Image, that has all the characters
 * stored in a Array (as an image)
 * @param fileName the File to write the image into.
 */
void Font::createAsciiImage(const char* fileName)
{
  if (this->fontTTF == NULL)
    return;
  int height = this->getMaxHeight();

  //
  SDL_Color tmpColor = {0, 0, 0};
  // Surface definition.
  SDL_Rect tmpRect; // this represents a Rectangle for blitting.
  SDL_Surface* tmpSurf =  SDL_CreateRGBSurface(SDL_SWSURFACE,
                                               height*16, height*16,
                                               32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                                               0x000000FF,
                                               0x0000FF00,
                                               0x00FF0000,
                                               0xFF000000
#else
                                               0xFF000000,
                                               0x00FF0000,
                                               0x0000FF00,
                                               0x000000FF
#endif
                                              );
  tmpRect.x = 0; tmpRect.y = 0; tmpRect.w = tmpSurf->w; tmpRect.h = tmpSurf->h;
  SDL_SetClipRect(tmpSurf, &tmpRect);
  int maxLineHeight = 0;

  int posX, posY;
  // all the interessting Glyphs
  for (posY = 0; posY < 16; posY++)
  {
    for (posX = 0; posX < 16; posX++)
    {
      SDL_Surface* glyphSurf = NULL;
      if (likely(this->fontTTF != NULL))
      {
        SDL_Color white = {255, 255, 255};
        glyphSurf = TTF_RenderGlyph_Blended(this->fontTTF, posX+16*posY, white);
      }
      if( glyphSurf != NULL )
      {
        tmpRect.x = height*posX;
        tmpRect.y = height*posY;
        SDL_SetAlpha(glyphSurf, 0, 0);

        SDL_BlitSurface(glyphSurf, NULL, tmpSurf, &tmpRect);
        SDL_FreeSurface(glyphSurf);
      }
    }
  }
  SDL_SaveBMP(tmpSurf, fileName);
  SDL_FreeSurface(tmpSurf);
}

/**
 * initializes the default font
 */
void Font::initDefaultFont()
{
  if (Font::defaultFont == NULL)
    Font::defaultFont = new Font(font_xpm);
}

/**
 * deletes the default font
 */
void Font::removeDefaultFont()
{
  if (Font::defaultFont != NULL)
    delete Font::defaultFont;
  Font::defaultFont = NULL;
}


/**
 * @returns the maximum height of the Font, if the font was initialized, 0 otherwise
 */
int Font::getMaxHeight()
{
  if (likely (this->fontTTF != NULL))
    return TTF_FontHeight(this->fontTTF);
  else
    return 0;
}

/**
 * @returns the maximum ascent of the Font, if the font was initialized, 0 otherwise

   the ascent is the pixels of the font above the baseline
 */
int Font::getMaxAscent()
{
  if (likely(this->fontTTF != NULL))
    return TTF_FontAscent(this->fontTTF);
  else
    return 0;
}

/**
 * @returns the maximum descent of the Font, if the font was initialized, 0 otherwise

   the descent is the pixels of the font below the baseline
 */
int Font::getMaxDescent()
{
  if (likely(this->fontTTF != NULL))
    return TTF_FontDescent(this->fontTTF);
  else
    return 0;
}

/**
 * @param character The character to get info about.
 * @returns a Glyph struct of a character. This Glyph is a pointer,
   and MUST be deleted by the user..

   This only works for horizontal fonts. see
   http://freetype.sourceforge.net/freetype2/docs/tutorial/step2.html
   for more info about vertical Fonts
 */
Glyph* Font::getGlyphMetrics(Uint16 character)
{
  Glyph* rg = new Glyph;
  rg->character = character;
  if (likely (this->fontTTF!= NULL))
  {
    int miX, maX, miY, maY, adv;
    TTF_GlyphMetrics(this->fontTTF, rg->character,
                     &miX, &maX,
                     &miY, &maY,
                     &adv);
    rg->minX = (float)miX / (float)this->renderSize;
    rg->maxX = (float)maX / (float)this->renderSize;
    rg->minY = (float)miY / (float)this->renderSize;
    rg->maxY = (float)maY / (float)this->renderSize;
    rg->advance = (float)adv / (float)this->renderSize;
  }
  rg->height = rg->maxY - rg->minY;
  rg->width = rg->maxX - rg->minX;
  rg->bearingX = (rg->advance - rg->width) / 2;
  rg->bearingY = rg->maxY;
  return rg;
}

/**
 * creates a Fast-Texture of this Font
 */
bool Font::createFastTexture()
{
  /* interesting GLYPHS:
  *  32: space
  *  33-47: Special Characters.
  *  48-57: 0-9
  *  58-63: some more special chars (minor)
  *  65-90: A-Z
  *  97-122: a-z
  */
  int numberOfGlyphs = 91;

  this->initGlyphs(32, numberOfGlyphs);
//  this->glyphArray[32]->width = .5f; //!< @todo find out the real size of a Space

  int rectSize = this->findOptimalFastTextureSize();

  // setting default values. (maybe not needed afterwards)
  SDL_Color tmpColor;  tmpColor.r = tmpColor.g = tmpColor.b = 0;
  // Surface definition.
  SDL_Rect tmpRect; // this represents a Rectangle for blitting.
  SDL_Surface* tmpSurf =  SDL_CreateRGBSurface(SDL_SWSURFACE,
                                               rectSize, rectSize,
                                               32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                                               0x000000FF,
                                               0x0000FF00,
                                               0x00FF0000,
                                               0xFF000000
#else
                                               0xFF000000,
                                               0x00FF0000,
                                               0x0000FF00,
                                               0x000000FF
#endif
                                              );
  tmpRect.x = 0; tmpRect.y = 0; tmpRect.w = tmpSurf->w; tmpRect.h = tmpSurf->h;
  SDL_SetClipRect(tmpSurf, &tmpRect);
  int maxLineHeight = this->getMaxHeight();

  // all the interessting Glyphs
  for (int i = 0; i < 128; i++)
  {
    SDL_Surface* glyphSurf = NULL;
    Glyph* tmpGlyph;

    if (tmpGlyph = this->glyphArray[i])
    {
      if (tmpGlyph->height*this->renderSize > maxLineHeight)
        maxLineHeight = (int)(tmpGlyph->height*this->renderSize);

      if (tmpRect.x+tmpGlyph->advance*this->renderSize > tmpSurf->w)
      {
        tmpRect.x = 0;
        tmpRect.y = tmpRect.y + maxLineHeight;
      }
      if (tmpRect.y + maxLineHeight > tmpSurf->h)
      {
        PRINTF(1)("Protection, so font cannot write over the boundraries (!!this should not heappen!!)\n");
        break;
      }
          // reading in the new Glyph
      if (likely(this->fontTTF != NULL))
      {
        SDL_Color white = {255, 255, 255};
        glyphSurf = TTF_RenderGlyph_Blended(this->fontTTF, i, white);
      }
      if( glyphSurf != NULL )
      {
        SDL_SetAlpha(glyphSurf, 0, 0);
        int tmp = tmpRect.y;
        tmpRect.y += this->getMaxAscent()-(int)((float)tmpGlyph->bearingY*this->renderSize);
        SDL_BlitSurface(glyphSurf, NULL, tmpSurf, &tmpRect);
        tmpRect.y = tmp;

        tmpGlyph->texCoord[0] = (float)(tmpRect.x)/(float)tmpSurf->w;
        tmpGlyph->texCoord[1] = (float)(tmpRect.x + tmpGlyph->width*(float)this->renderSize)/(float)tmpSurf->w;
        tmpGlyph->texCoord[2] = (float)(tmpRect.y)/(float)tmpSurf->w;
        tmpGlyph->texCoord[3] = (float)(tmpRect.y+this->getMaxHeight())/(float)tmpSurf->w;
        SDL_FreeSurface(glyphSurf);
        tmpRect.x += (int)(tmpGlyph->advance * this->renderSize)+1;

        /*
        // Outputting Glyphs to BMP-files.
        char outname[1024];
        if (i < 10)
        sprintf( outname, "%s-glyph-00%d.bmp", this->getName(), i );
        else if (i <100)
        sprintf( outname, "%s-glyph-0%d.bmp", this->getName(), i );
        else
        sprintf( outname, "%s-glyph-%d.bmp", this->getName(), i );
        SDL_SaveBMP(tmpSurf, outname);*/
      }
    }
  }
  // outputting the GLYPH-table
  //   char outName[1024];
  //   sprintf( outName, "%s-glyphs.bmp", this->getName());
  //   SDL_SaveBMP(tmpSurf, outName);

  if (this->setSurface(tmpSurf))
    (this->setTexture(Texture::loadTexToGL(tmpSurf)));
}

/**
 *  stores Glyph Metrics in an Array.
 * @param from The Glyph to start from.
 * @param count The number of Glyphs to start From.
 */
void Font::initGlyphs(Uint16 from, Uint16 count)
{
  /* initialize the Array, and set all its entries to NULL
  *  only if the Glyph-array has not been initialized
  */
  if (!this->glyphArray)
  {
    this->glyphArray = new Glyph*[FONT_HIGHEST_KNOWN_CHAR];
    for (int i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
      this->glyphArray[i] = NULL;
  }

  Uint16 lastGlyph = from + count;

  for (int i = from; i <= lastGlyph; i++)
  {
      // setting up all the Glyphs we like.
    glyphArray[i] = getGlyphMetrics(i);
  }
  return;
}

/**
 * @returns the optimal size to use as the texture size

   @todo: this algorithm can be a lot faster, althought it does
   not really matter within the init-context, and 128 glyphs.

   This function searches for a 2^n sizes texture-size, this is for
   openGL-version < 1.2 compatibility ( and because it is realy easy like this :))
 */
int Font::findOptimalFastTextureSize()
{
  if (this->glyphArray == NULL)
    return 0;

  int i;
  int x,y; // the counters
  int maxLineHeight = this->getMaxHeight();
  unsigned int size = 32;  // starting Value, we have to start somewhere 32 seems reasonable. (take any small enough 2^i number)
  bool sizeOK = false;
  Glyph* tmpGlyph;

  while (!sizeOK)
  {
    x = 0; y = 0;
    for (i = 0; i < FONT_HIGHEST_KNOWN_CHAR; i++)
    {
      if((tmpGlyph = this->glyphArray[i]) != NULL)
      {
              // getting the height of the highest Glyph in the Line.
        if (tmpGlyph->height*this->renderSize > maxLineHeight)
          maxLineHeight = (int)(tmpGlyph->height*this->renderSize);

        if (x + tmpGlyph->advance*this->renderSize > size)
        {
          x = 0;
          y = y + maxLineHeight;
                  //maxLineHeight = 0;
        }
        if (y + maxLineHeight + 1 > size)
          break;
        x += (int)(tmpGlyph->advance*this->renderSize)+1;

      }
    }
    if (i >= FONT_HIGHEST_KNOWN_CHAR-1 || size > 8192)
      sizeOK = true;
    else
      size *= 2;
  }
  return size;
}


/**
 *  a simple function to get some interesting information about this class
 */
void Font::debug()
{
  // print the loaded font's style
  int style;
  if (likely(this->fontTTF != NULL))
    style = TTF_GetFontStyle(this->fontTTF);
  PRINTF(0)("The font style is:");
  if(style==TTF_STYLE_NORMAL)
    PRINTF(0)(" normal");
  else {
    if(style&TTF_STYLE_BOLD)
      PRINTF(0)(" bold");
    if(style&TTF_STYLE_ITALIC)
      PRINTF(0)(" italic");
    if(style&TTF_STYLE_UNDERLINE)
      PRINTF(0)(" underline");
  }
  PRINTF(0)("\n");
}
