/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_command.h"
#include "shell_command_class.h"

#include "list.h"
#include "debug.h"
#include "class_list.h"

#include "key_names.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

using namespace std;

/**
 * constructs and registers a new Command
 * @param commandName the name of the Command
 * @param className the name of the class to apply this command to
 * @param paramCount the count of parameters this command takes
 */
ShellCommand::ShellCommand(const char* commandName, const char* className, const Executor& executor)
{
  this->setClassID(CL_SHELL_COMMAND, "ShellCommand");
  this->setName(commandName);
  this->description = NULL;
  this->alias = NULL;
  this->executor = executor.clone();

//  this->classID = classID;
  this->shellClass = ShellCommandClass::getCommandClass(className); //ClassList::IDToString(classID);
  if (this->shellClass != NULL)
    this->shellClass->commandList.push_back(this);
}

/**
 * deconstructs a ShellCommand
 */
ShellCommand::~ShellCommand()
{
  if (this->alias != NULL && ShellCommandClass::aliasList != NULL)
  {
    ShellCommandClass::aliasList->remove(this->alias);
    delete this->alias;
  }
  delete this->executor;
}

/**
 * registers a new ShellCommand
 */
ShellCommand* ShellCommand::registerCommand(const char* commandName, const char* className, const Executor& executor)
{
  if (ShellCommand::isRegistered(commandName, className, executor))
    return NULL;
  else
    return new ShellCommand(commandName, className, executor);

}

/**
 * unregister an existing commandName
 * @param className the name of the Class the command belongs to.
 * @param commandName the name of the command itself
 */
void ShellCommand::unregisterCommand(const char* commandName, const char* className)
{
  /// FIXME
/*  if (ShellCommandClass::commandClassList == NULL)
    ShellCommandClass::initCommandClassList();

 const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);

 if (checkClass != NULL)
  {
    std::list<ShellCommand*>::iterator elem;
    for (elem = checkClass->commandList.begin(); elem != checkClass->commandList.end(); elem++)
    {
      if (!strcmp(commandName, (*elem)->getName()))
      {
        delete (*elem);
        checkClass->commandList.remove(*elem);
        break;
      }
    }

    if (checkClass->commandList->size() == 0)
    {
      ShellCommandClass::commandClassList->remove(checkClass);
      delete checkClass;
    }
  }*/
}

/**
 * checks if a command has already been registered.
 * @param commandName the name of the Command
 * @param className the name of the Class the command should apply to.
 * @param paramCount how many arguments the Command takes
 * @returns true, if the command is registered/false otherwise
 *
 * This is used internally, to see, if we have multiple command subscriptions.
 * This is checked in the registerCommand-function.
 */
bool ShellCommand::isRegistered(const char* commandName, const char* className, const Executor& executor)
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    ShellCommandClass::initCommandClassList();
    return false;
  }

  const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);
  if (checkClass != NULL)
  {
    std::list<ShellCommand*>::const_iterator elem;
    for (elem = checkClass->commandList.begin(); elem != checkClass->commandList.end(); elem++)
    {
      if (!strcmp(commandName, (*elem)->getName()))
      {
        PRINTF(2)("Command '%s::%s' already registered\n", className, commandName);
        return true;
      }
    }
   return false;
  }
  else
    return false;
}


/**
 * executes commands
 * @param executionString the string containing the following input
 * ClassName [ObjectName] functionName [parameter1[,parameter2[,...]]]
 * @return true on success, false otherwise.
 */
bool ShellCommand::execute(const char* executionString)
{
  if (ShellCommandClass::commandClassList == NULL)
    return false;

  long classID = CL_NULL;                      //< the classID retrieved from the Class.
  ShellCommandClass* commandClass = NULL;      //< the command class this command applies to.
  std::list<BaseObject*>* objectList = NULL;   //< the list of Objects stored in classID
  BaseObject* objectPointer = NULL;            //< a pointer to th Object to Execute the command on
  bool emptyComplete = false;                  //< if the completion input is empty string. e.g ""
  unsigned int fktPos = 1;                     //< the position of the function (needed for finding it)
//  long completeType = SHELLC_NONE;           //< the Type we'd like to complete.
  SubString inputSplits(executionString, " \t\n,");

  if (inputSplits.getCount() == 0)
    return false;
  if (inputSplits.getCount() >= 1)
  {
    // CHECK FOR ALIAS
    if (ShellCommandClass::aliasList != NULL)
    {
      list<ShellCommandAlias*>::iterator alias;
      for (alias = ShellCommandClass::aliasList->begin(); alias != ShellCommandClass::aliasList->end(); alias++ )
      {
        if ((*alias)->getName() != NULL && !strcmp((*alias)->getName(), inputSplits.getString(0)) && (*alias)->getCommand() != NULL &&
            (*alias)->getCommand()->shellClass != NULL )
        {
          objectList = ClassList::getList((*alias)->getCommand()->shellClass->getName());
          if (objectList != NULL)
          {
            if (inputSplits.getCount() > 1)
              (*alias)->getCommand()->executor->execute(objectList->front(), executionString+inputSplits.getOffset(1));
            else
              (*alias)->getCommand()->executor->execute(objectList->front(), "");
           return true;
          }
        }
      }
    }
    // looking for a Matching Class
    if (likely(ShellCommandClass::commandClassList != NULL))
    {
      list<ShellCommandClass*>::iterator commandClassIT;
      for (commandClassIT = ShellCommandClass::commandClassList->begin(); commandClassIT != ShellCommandClass::commandClassList->end(); commandClassIT++)
      {
        if ((*commandClassIT)->getName() && !strcasecmp(inputSplits.getString(0), (*commandClassIT)->getName()))
        {
          //elemCL->getName();
          classID = ClassList::StringToID((*commandClassIT)->getName());
          commandClass = (*commandClassIT);
          objectList = ClassList::getList((ClassID)classID);
          break;
        }
      }
    }

    if (commandClass != NULL && inputSplits.getCount() >= 2)
    {
      if (objectList != NULL)
      {
        // Checking for a Match in the Objects of classID (else take the first)
        list<BaseObject*>::const_iterator object;
        for (object = objectList->begin(); object != objectList->end(); object++)
        {
          if ((*object)->getName() != NULL && !strcasecmp((*object)->getName(), inputSplits.getString(1)))
          {
            objectPointer = (*object);
            fktPos = 2;
            break;
          }
         }

      //
        if (objectPointer == NULL)
          objectPointer = objectList->front();
      }
      // match a function.
      if (commandClass != NULL && (fktPos == 1 || (fktPos == 2 && inputSplits.getCount() >= 3)))
      {
        list<ShellCommand*>::iterator cmdIT;
        for (cmdIT = commandClass->commandList.begin(); cmdIT != commandClass->commandList.end(); cmdIT++)
        {
          if (!strcmp((*cmdIT)->getName(), inputSplits.getString(fktPos)))
          {
            if (objectPointer == NULL && (*cmdIT)->executor->getType() & Executor_Objective)
              return false;
            if (inputSplits.getCount() > fktPos+1)
              (*cmdIT)->executor->execute(objectPointer, executionString+inputSplits.getOffset(fktPos +1));
            else
              (*cmdIT)->executor->execute(objectPointer, "");
            return true;
          }
        }
      }
    }
  }
}

/**
 * lets a command be described
 * @param description the description of the Given command
 */
ShellCommand* ShellCommand::describe(const char* description)
{
  if (this == NULL)
    return NULL;
 else
 {
   this->description = description;
   return this;
 }
}

/**
 * adds an Alias to this Command
 * @param alias the name of the Alias to set
 * @returns itself
 */
ShellCommand* ShellCommand::setAlias(const char* alias)
{
  if (this == NULL)
    return NULL;

  if (this->alias != NULL)
  {
    PRINTF(2)("not more than one Alias allowed for functions (%s::%s)\n", this->getName(), this->shellClass->getName());
  }
  else
  {
    if (ShellCommandClass::aliasList == NULL)
      ShellCommandClass::aliasList = new std::list<ShellCommandAlias*>;

    ShellCommandAlias* aliasCMD = new ShellCommandAlias(alias, this);
    ShellCommandClass::aliasList->push_back(aliasCMD);
    this->alias = aliasCMD;
  }
  return this;
}

/**
 * sets default Values of the Commands
 * @param count how many default Values to set.
 * @param ... the default Values in order. They will be cast to the right type
 * @returns itself
 *
 * Be aware, that when you use this Function, you !!MUST!! match the input as
 * count, [EXACTLY THE SAME AS IF YOU WOULD CALL THE FUNCTION UP TO count ARGUMENTS]
 */
ShellCommand* ShellCommand::defaultValues(unsigned int count, ...)
{
  if (this == NULL)
    return NULL;

  va_list values;
  va_start(values, count);

  this->executor->defaultValues(count, values);

  return this;
}

/**
 * prints out nice information about the Shells Commands
 */
void ShellCommand::debug()
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    PRINT(0)("No Command registered.\n");
    return;
  }

  list<ShellCommandClass*>::iterator classIT;
  for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
  {
    PRINT(0)("Class:'%s' registered %d commands: \n", (*classIT)->className, (*classIT)->commandList.size());

    list<ShellCommand*>::iterator cmdIT;
    for (cmdIT = (*classIT)->commandList.begin(); cmdIT != (*classIT)->commandList.end(); cmdIT++)
    {
      PRINT(0)("  command:'%s' : params:%d: ", (*cmdIT)->getName(), (*cmdIT)->executor->getParamCount());
      /// FIXME
      /*      for (unsigned int i = 0; i< elem->paramCount; i++)
       printf("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
      if ((*cmdIT)->description != NULL)
       printf("- %s", (*cmdIT)->description);
      printf("\n");

    }
  }
}

/**
 * converts a Parameter to a String
 * @param parameter the Parameter we have.
 * @returns the Name of the Parameter at Hand
 */
const char* ShellCommand::paramToString(long parameter)
{
  return MultiType::MultiTypeToString((MT_Type)parameter);
// FIXME
  /*  switch (parameter)
  {
    case ParameterBool:
      return "BOOL";
      break;
    case ParameterChar:
      return "CHAR";
      break;
    case ParameterString:
      return "STRING";
      break;
    case ParameterInt:
      return "INT";
      break;
    case ParameterUInt:
      return "UINT";
      break;
    case ParameterFloat:
      return "FLOAT";
      break;
    case ParameterLong:
      return "LONG";
      break;
    default:
      return "NULL";
      break;
  }*/
}
