/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_completion.h"
#include "shell_command_class.h"

#include "shell_input.h"
#include "shell_command.h"

#include "substring.h"
#include "base_object.h"
#include "class_list.h"
#include "debug.h"

#include "stdlibincl.h"

using namespace std;

/**
 * standard constructor
 */
ShellCompletion::ShellCompletion(ShellInput* input)
{
  this->input = input;
}


/**
 * standard deconstructor
 */
ShellCompletion::~ShellCompletion ()
{
}



/**
 * autocompletes the Shell's inputLine
 * @returns true, if a result was found, false otherwise
 */
bool ShellCompletion::autoComplete(ShellInput* input)
{
  const char* completionLine;           //< the inputLine we complete.

  long classID;                         //< the classID retrieved from the Class.
  std::list<BaseObject*>* objectList;   //< the list of Objects stored in classID
  bool emptyComplete = false;           //< if the completion input is empty string. e.g ""
  long completeType = SHELLC_NONE;      //< the Type we'd like to complete.
  const char* completeString;           //< the string to complete.


  PRINTF(5)("AutoComplete on input\n");
  this->emptyCompletionList();

  if (input != NULL)
    this->input = input;
  if (this->input == NULL)
  {
    PRINTF(2)("No ShellInput supplied\n");
    return false;
  }

  // Check if we are in a input. eg. the supplied string "class " and now we complete either function or object
  if (this->input->getInput() != NULL &&
      strrchr(this->input->getInput(), ' ') >= this->input->getInput() + strlen(this->input->getInput())-1)
  {
    emptyComplete = true;
  }

  // CREATE INPUTS
  if (this->input->getInput() == NULL)
    completionLine = "";
  else
    completionLine = this->input->getInput() + strspn(this->input->getInput(), " \t\n");
  SubString inputSplits(completionLine, " \t\n,");

  // What String will be completed
  if (emptyComplete == true)
    completeString = "";
  else
    completeString = inputSplits.getString(inputSplits.getCount()-1);

  // CLASS COMPLETION
  if (inputSplits.getCount() == 0)
  {
    completeType |= SHELLC_CLASS;
    completeType |= SHELLC_ALIAS;
  }
  else if (inputSplits.getCount() == 1 && emptyComplete == false)
  {
    completeType |= SHELLC_CLASS;
    completeType |= SHELLC_ALIAS;
  }

  // OBJECT/FUNCTION COMPLETIONS
  else if ((inputSplits.getCount() == 1 && emptyComplete == true) ||
            (inputSplits.getCount() == 2 && emptyComplete == false))
  {
    classID = ClassList::StringToID(inputSplits.getString(0));
    objectList = ClassList::getList(classID);
    if (classID != CL_NULL)
      completeType |= SHELLC_OBJECT;
    //if (objectList != NULL && objectList->getSize() == 1)
      completeType |= SHELLC_FUNCTION;
  }
  else if ((inputSplits.getCount() == 2 && emptyComplete == true) ||
            (inputSplits.getCount() == 3 && emptyComplete == false))
  {
    classID = ClassList::StringToID(inputSplits.getString(0));
    if (classID == CL_NULL)
      return false;
    else
     completeType |= SHELLC_FUNCTION;
  }

  if (completeType & SHELLC_CLASS)
    this->objectComplete(completeString, CL_SHELL_COMMAND_CLASS);
  if (completeType & SHELLC_OBJECT)
    this->objectComplete(completeString, classID);
  if (completeType & SHELLC_FUNCTION)
    this->functionComplete(completeString, inputSplits.getString(0));
  if (completeType & SHELLC_ALIAS)
    this->aliasComplete(completeString);


  this->generalComplete(completeString);
  return true;
}

/**
 * autocompletes a className
 * @param classBegin the Beginning of a String to autoComplete
 * @return true on success, false otherwise
 */
bool ShellCompletion::classComplete(const char* classBegin)
{
  if (unlikely(classBegin == NULL))
    return false;
  const std::list<const char*>* clList = ClassList::getClassList();
  if (clList != NULL)
  {
    if (!this->addToCompleteList(clList, classBegin, SHELLC_CLASS))
      return false;
  }
  else
    return false;
  return true;
}

/**
 * autocompletes an ObjectName
 * @param objectBegin the beginning string of a Object
 * @param classID the ID of the Class to search for.
 * @return true on success, false otherwise
 */
bool ShellCompletion::objectComplete(const char* objectBegin, long classID)
{
  if (unlikely(objectBegin == NULL))
    return false;
  const std::list<BaseObject*>* boList = ClassList::getList(classID);
  if (boList != NULL)
  {
    SHELLC_TYPE type = SHELLC_OBJECT;
    if (classID == CL_SHELL_COMMAND_CLASS)
      type = SHELLC_CLASS;
    if (!this->addToCompleteList(boList, objectBegin, type))
      return false;
  }
  else
    return false;
  return true;
}

/**
 * completes a Function
 * @param functionBegin the beginning of the function String
 * @param classID the class' ID to complete the function of
 */
bool ShellCompletion::functionComplete(const char* functionBegin, const char* className)
{
  if (unlikely(functionBegin == NULL))
    return false;
  std::list<const char*> fktList;
  ShellCommandClass::getCommandListOfClass(className, &fktList);
  //printf("%s\n", boList->firstElement()->getName());
  if (!this->addToCompleteList(&fktList, functionBegin, SHELLC_FUNCTION))
    return false;
  return true;
}

/**
 * completes an Alias
 * @param aliasBegin the beginning of the Alias-String to complete
 * @returns true on succes, false if something went wrong
 */
bool ShellCompletion::aliasComplete(const char* aliasBegin)
{
  if (unlikely(aliasBegin == NULL))
    return false;
  std::list<const char*> aliasList;
  ShellCommandClass::getCommandListOfAlias(&aliasList);
  //printf("%s\n", boList->firstElement()->getName());
  if (!this->addToCompleteList(&aliasList, aliasBegin, SHELLC_ALIAS))
    return false;
  return true;
}


/**
 * completes the inputline on grounds of an inputList
 * @param begin the String to search in the inputList, and to extend with it.
 * @param displayAs how to display the found value to the user, printf-style, !!with only one %s!! ex.: "::%s::"
 * @param addBack what should be added at the end of the completion
 * @param addFront what should be added to the front of one finished completion
 * @return true if ok, false otherwise
 */
bool ShellCompletion::generalComplete(const char* begin, const char* displayAs, const char* addBack, const char* addFront)
{
  if (this->input == NULL )
    return false;
  if (completionList.size() == 0)
    return false;

  ShellC_Element addElem = completionList.front();
  const char* addString = addElem.name;
  unsigned int addLength = 0;
  unsigned int inputLenght = strlen(begin);

  // Determin the longest Match
  if (addString != NULL)
    addLength = strlen(addString);

  SHELLC_TYPE changeType = SHELLC_NONE;
  list<ShellC_Element>::iterator charIT;
  for (charIT = completionList.begin(); charIT != completionList.end(); charIT++)
  {
    if ((*charIT).type != changeType)
    {
      if (changeType != SHELLC_NONE)
        PRINT(0)("\n");
      PRINT(0)("%s: ", ShellCompletion::typeToString((*charIT).type));
      changeType = (*charIT).type;
    }
    PRINTF(0)("%s ", (*charIT).name);
    for (unsigned int i = inputLenght; i < addLength; i++)
      if (addString[i] != (*charIT).name[i])
      {
       addLength = i;
//       break;
      }
  }
  PRINT(0)("\n");

  if (addLength >= inputLenght)
  {
    char* adder = new char[addLength+1];
    strncpy(adder, addString, addLength);
    adder[addLength] = '\0';

    if (this->input)
    {
     this->input->removeCharacters(inputLenght);
     this->input->addCharacters(adder);

      if (completionList.size() == 1)
      {
        if ( addBack != NULL )
         this->input->addCharacters(addBack);
        this->input->addCharacter(' ');
      }
     delete[] adder;
    }
  }
  return true;
}

/**
 * @brief searches for classes, which beginn with completionBegin
 * @param inputList the List to parse through
 * @param completionBegin the beginning string
 * !! The strings MUST NOT be deleted !!
 */
bool ShellCompletion::addToCompleteList(const std::list<const char*>* inputList, const char* completionBegin, SHELLC_TYPE type)
{
  if (inputList == NULL || completionBegin == NULL)
    return false;
  unsigned int searchLength = strlen(completionBegin);

  list<const char*>::const_iterator string;
  for (string = inputList->begin(); string != inputList->end(); string++)
  {
    if (strlen(*string) >= searchLength &&
        !strncasecmp(*string, completionBegin, searchLength))
    {
      ShellC_Element newElem;
      newElem.name = *string;
      newElem.type = type;
      this->completionList.push_back(newElem);
    }
  }
  return true;
}

/**
 * searches for classes, which beginn with completionBegin
 * @param inputList the List to parse through
 * @param completionBegin the beginning string
 * !! The strings MUST NOT be deleted !!
 */
bool ShellCompletion::addToCompleteList(const std::list<BaseObject*>* inputList, const char* completionBegin, SHELLC_TYPE type)
{
  if (inputList == NULL || completionBegin == NULL)
    return false;
  unsigned int searchLength = strlen(completionBegin);

  list<BaseObject*>::const_iterator bo;
  for(bo = inputList->begin(); bo != inputList->end(); bo++)
  {
    if ((*bo)->getName() != NULL &&
        strlen((*bo)->getName()) >= searchLength &&
        !strncasecmp((*bo)->getName(), completionBegin, searchLength))
    {
      ShellC_Element newElem;
      newElem.name = (*bo)->getName();
      newElem.type = type;
      this->completionList.push_back(newElem);
    }
  }

  return true;
}

/**
 * deletes the Completion List.
 *
 * This is done at the beginning of each completion-run
 */
void ShellCompletion::emptyCompletionList()
{
  this->completionList.clear();
}

const char* ShellCompletion::typeToString(SHELLC_TYPE type)
{
  switch (type)
  {
    default:// SHELLC_NONE
      return "error";
    case  SHELLC_CLASS:
      return "class";
    case SHELLC_OBJECT:
      return "object";
    case SHELLC_FUNCTION:
      return "function";
    case SHELLC_ALIAS:
      return "alias";
  }
}
