/*!
 * @file class_list.h
  *  Definition of the Class List, that handles a Class-Specific-Control structure

 */

#ifndef _CLASS_LIST_H
#define _CLASS_LIST_H

#include "class_id.h"
#include <list>

// FORWARD DECLARATION
class BaseObject;

//! A class that handles Pointers to Objects of all type.
/**
 * here all the Pointers to all the Object of orxonox are stored, that implement BaseObject
 * for now.
 * You can get Any Object's Reference to BaseObject with dynamic_cast<T>(ClassList::getObject(name, CL_T_NAME));
 *  where: T: is the Class to cast to,
 *   name: the name of the Object (not className)
 *   CL_T_NAME: the class Identifier, (if CL_NULL or nothing it will take longer, because all BaseObject's are searched through)
 *
 * There is also the exists-function, that just checks, if a Reference is still in existence.
 *
 * @see ClassID, BaseObject, dynamic_cast
 */
class ClassList {

  public:
    ClassList(const long& classID, const char* className);
    virtual ~ClassList();

    // STATIC FUNCTIONS
    static void                          addToClassList(BaseObject* objectPointer, const long& classID, const char* className);
    static void                          removeFromClassList(BaseObject* objectPointer);

    static std::list<BaseObject*>*       getList(long classID = CL_NULL);
    static std::list<BaseObject*>*       getList(const char* className);
    static const std::list<const char*>* getClassList();
    static BaseObject*                   getObject(const char* name, long classID = CL_NULL);
    static bool                          exists(const BaseObject* object, long classID = CL_NULL);

    static void                          whatIs(const BaseObject* object);

    static const char*                   IDToString(long classID = CL_NULL);
    static long                          StringToID(const char* className);
    static void                          debug(unsigned int debugLevel = 0, long classID = CL_NULL);
    static void                          debugS(const char* className = 0x0, unsigned int debugLevel = 0);



  private:
    std::list<BaseObject*>          objectList;             //!< A list of Objects belonging to this Class

    long                            classID;                //!< ClassID stored in this ClassList \see ClassID
    const char*                     className;              //!< Name of the Class Stored here

    ClassList*                      next;                   //!< Pointer to the next class in the List

    // STATIC MEMBERS
    static ClassList*               first;                  //!< The first Class in the List
    static std::list<const char*>   classList;              //!< a List of all Names of all classes, that have registered so far.
    static unsigned int             classCount;             //!< The Count of classes that have been registered (should match the lower description)
};

#endif /* _CLASS_LIST_H */
