/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_command_class.h"

#include "shell_command.h"

#include "list.h"
#include "debug.h"
#include "class_list.h"

#include <stdio.h>
#include <string.h>

using namespace std;

std::list<ShellCommandClass*>* ShellCommandClass::commandClassList = NULL;
std::list<ShellCommandAlias*>* ShellCommandClass::aliasList = NULL;

/**
 * creates a new ShellCommandClass
 * @param className the Name of the command-class to create
 */
ShellCommandClass::ShellCommandClass(const char* className)
{
  this->setClassID(CL_SHELL_COMMAND_CLASS, "ShellCommandClass");
  this->setName(className);

  this->className = className;
  this->classID = CL_NULL;

  ShellCommandClass::commandClassList->push_back(this);
}

/**
 * destructs the shellCommandClass again
 */
ShellCommandClass::~ShellCommandClass()
{
  while(this->commandList.size() > 0)
  {
    delete this->commandList.front();
    this->commandList.pop_front();
  }
}

/**
 * collects the Commands registered to some class.
 * @param className the name of the Class to collect the Commands from.
 * @param stringList a List to paste the Commands into.
 * @returns true on success, false otherwise
 */
bool ShellCommandClass::getCommandListOfClass(const char* className, std::list<const char*>* stringList)
{
  if (stringList == NULL || className == NULL)
    return false;

  list<ShellCommandClass*>::iterator elem;
  for(elem = ShellCommandClass::commandClassList->begin(); elem != ShellCommandClass::commandClassList->end(); elem++)
  {
    if (!strcmp ((*elem)->getName(), className))
    {
      list<ShellCommand*>::iterator command;
      for(command = (*elem)->commandList.begin(); command != (*elem)->commandList.end(); command++)
        stringList->push_back((*command)->getName());
    }
  }
  return true;
}

/**
 * collects the Aliases registered to the ShellCommands
 * @param stringList a List to paste the Aliases into.
 * @returns true on success, false otherwise
 */
bool ShellCommandClass::getCommandListOfAlias(std::list<const char*>* stringList)
{
  if (stringList == NULL || ShellCommandClass::aliasList == NULL)
    return false;

  list<ShellCommandAlias*>::iterator alias;
  for (alias = ShellCommandClass::aliasList->begin(); alias != ShellCommandClass::aliasList->end(); alias++)
    stringList->push_back((*alias)->getName());
  return true;
}

/**
 * unregisters all Commands that exist
 */
void ShellCommandClass::unregisterAllCommands()
{
  /// FIXME

  /*  if (ShellCommandClass::commandClassList != NULL)
  {
    // unregister all commands
    tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
    ShellCommandClass* elem = iterator->firstElement();
    while(elem != NULL)
    {
      delete elem;

      elem = iterator->nextElement();
    }
    delete iterator;

    delete ShellCommandClass::commandClassList;
    ShellCommandClass::commandClassList = NULL;
  }

  // unregister all aliases (there should be nothing to do here :))
  if (ShellCommandClass::aliasList != NULL)
  {
    tIterator<ShellCommandAlias>* itAL = ShellCommandClass::aliasList->getIterator();
    ShellCommandAlias* elemAL = itAL->firstElement();
    while(elemAL != NULL)
    {
      delete elemAL;
      elemAL = itAL->nextElement();
    }
    delete itAL;
    delete ShellCommandClass::aliasList;
    ShellCommandClass::aliasList = NULL;
  }*/
}

/**
 * checks if a Class is already registered to the Commands' class-stack
 * @param className the Name of the Class to check for
 * @returns the CommandClass if found, NULL otherwise
 */
const ShellCommandClass* ShellCommandClass::isRegistered(const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    initCommandClassList();

  list<ShellCommandClass*>::const_iterator classIT;
  for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
  {
    if (!strcmp(className, (*classIT)->className))
    {
      if ((*classIT)->classID == CL_NULL)
        (*classIT)->classID = ClassList::StringToID(className);

      return (*classIT);
    }
  }
  return NULL;
}

/**
 * searches for a CommandClass
 * @param className the name of the CommandClass
 * @returns the CommandClass if found, or a new CommandClass if not
 */
ShellCommandClass* ShellCommandClass::getCommandClass(const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    initCommandClassList();

  list<ShellCommandClass*>::iterator classIT;
  for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
  {
    if (!strcmp(className, (*classIT)->className))
    {
      return (*classIT);
    }
  }
  return new ShellCommandClass(className);
}

/**
 * initializes the CommandList (if it is NULL)
 */
void ShellCommandClass::initCommandClassList()
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    ShellCommandClass::commandClassList = new std::list<ShellCommandClass*>;
    ShellCommand::registerCommand("debug", "ShellCommand", ExecutorStatic<ShellCommand>(ShellCommand::debug));
  }
}

/**
 * displays help about ShellCommandClass
 * @param className: the Class of Commands to show help about
 */
void ShellCommandClass::help(const char* className)
{
  if (className == NULL)
    return;
  if (likely(ShellCommandClass::commandClassList != NULL))
  {
    list<ShellCommandClass*>::iterator classIT;
    for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
    {
      if ((*classIT)->className && !strcasecmp(className, (*classIT)->className))
      {
        PRINT(0)("Class:'%s' registered %d commands: \n", (*classIT)->className, (*classIT)->commandList.size());
        list<ShellCommand*>::const_iterator cmdIT;
        for (cmdIT = (*classIT)->commandList.begin(); cmdIT != (*classIT)->commandList.end(); cmdIT++)
        {
          PRINT(0)("  command:'%s' : params:%d: ", (*cmdIT)->getName(), (*cmdIT)->executor->getParamCount());
          /// FIXME
          /*          for (unsigned int i = 0; i< elem->paramCount; i++)
            PRINT(0)("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
          if ((*cmdIT)->description != NULL)
            PRINT(0)("- %s", (*cmdIT)->description);
          PRINT(0)("\n");
        }
        return;
      }
    }
    PRINTF(3)("Class %s not found in Command's classes\n", className);
  }
  else
  {
    PRINTF(1)("List of commandClasses does not exist");
  }
}

