/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: ...
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_PHYSICS

#include "physics_engine.h"

#include "class_list.h"
#include "list.h"
#include "tinyxml.h"
#include "factory.h"
#include "load_param.h"

using namespace std;


/**
 *  standard constructor
*/
PhysicsEngine::PhysicsEngine()
{
  this->setClassID(CL_PHYSICS_ENGINE, "PhysicsEngine");
  this->setName("PhysicsEngine");
  this->interfaces = NULL;
}

/**
 *  the singleton reference to this class
*/
PhysicsEngine* PhysicsEngine::singletonRef = NULL;

/**
 *  standard deconstructor

*/
PhysicsEngine::~PhysicsEngine()
{
  // delete all PhysicsConnections that are still in existence
  while (this->connections.size() > 0)
  {
    PhysicsConnection* connection = this->connections.front();
    this->connections.pop_front();
    delete connection;
  }
//
//   // delete all PhysicsInterfaces, still in existence (this could be dangerous)
//   tIterator<PhysicsInterface>* itPI = this->interfaces->getIterator();
//   PhysicsInterface* enumPI = itPI->firstElement();
//   while (enumPI)
//   {
//     delete enumPI;
//
//     enumPI = itPI->nextElement();
//   }
//   delete itPI;
//
//   // delete all PhysicsFields, still in existence (this could be dangerous)
//   tIterator<Field>* itF = this->fields->getIterator();
//   Field* enumF = itF->firstElement();
//   while (enumF)
//   {
//     delete enumF;
//
//     enumF = itF->nextElement();
//   }
//   delete itF;

  PhysicsEngine::singletonRef = NULL;
}

/**
* @param root the XML-element to load settings from
 */
void PhysicsEngine::loadParams(const TiXmlElement* root)
{
  LoadParamXML(root, "Fields", this, PhysicsEngine, loadFields)
      .describe("loads a list of fields");

  LoadParamXML(root, "Connections", this, PhysicsEngine, loadConnections)
      .describe("loads a list of fields");
}

/**
 * @param root the XML-element to Load the PhysicsField from
 */
void PhysicsEngine::loadFields(const TiXmlElement* root)
{
  PRINTF(4)("Loading Physical Fields\n");

  const TiXmlElement* element = root->FirstChildElement();
  while (element != NULL)
  {
    Factory::getFirst()->fabricate(element);

    element = element->NextSiblingElement();
  }
}

/**
 * @param root the XML-element to load the PhysicsConnection from
 */
void PhysicsEngine::loadConnections(const TiXmlElement* root)
{
  PRINTF(4)("Loading Physical Connections\n");

  const TiXmlElement* element = root->FirstChildElement();
  while (element != NULL)
  {
    Factory::getFirst()->fabricate(element);

    element = element->NextSiblingElement();
  }
}

/**
* @param physicsInterfaceName the Name of the PhysicsInterface to search for
  @returns the PhysicsInterface if found, or NULL if not
 */
PhysicsInterface* PhysicsEngine::getPhysicsInterfaceByName(const char* physicsInterfaceName) const
{
  BaseObject* interface = ClassList::getObject(physicsInterfaceName, CL_PHYSICS_INTERFACE);
  return (interface != NULL)?  dynamic_cast<PhysicsInterface*>(interface) : NULL;
}

/**
 *  adds a Field to the list of handeled fields
 * @param field the field to add

   this is normally done in the constructor of any Field
*/
void PhysicsEngine::addField(Field* field)
{
  this->fields.push_back(field);
}

/**
 *  removes a Field from the list of handeled fields
 * @param field the field to remove

   this is normally done in the destructor of any Field
*/
void PhysicsEngine::removeField(Field* field)
{
  this->fields.remove(field);
}

/**
* @param FieldName the Name of the PhysicsInterface to search for
  @returns the Field if found, or NULL if not
 */
Field* PhysicsEngine::getFieldByName(const char* FieldName) const
{
  list<Field*>::const_iterator field;
  for (field = this->fields.begin(); field != this->fields.end(); field++)
    if (!strcmp(FieldName, (*field)->getName()))
      return (*field);
  return NULL;
}



/**
 *  adds A Physical Connection to the List of Connections
 * @param connection the Connection to add

   Usually this is done through the constructor of PhysicshConnections
*/
void PhysicsEngine::addConnection(PhysicsConnection* connection)
{
  this->connections.push_back(connection);
}

/**
 *  removes A Physical Connection from the List of Connections
 * @param connection the Connection to remove

   Usually this is done through the destructor of PhysicsConnections
*/
void PhysicsEngine::removeConnection(PhysicsConnection* connection)
{
  this->connections.remove(connection);
}

/**
* @param physicsConnectionName the Name of the PhysicsInterface to search for
  @returns the PhysicsConnection if found, or NULL if not
 */
PhysicsConnection* PhysicsEngine::getPhysicsConnectionByName(const char* physicsConnectionName) const
{
  list<PhysicsConnection*>::const_iterator pc;
  for (pc = this->connections.begin(); pc != this->connections.end(); pc++)
    if (!strcmp(physicsConnectionName, (*pc)->getName()))
      delete (*pc);
  return NULL;
}



/**
 *  Steps through all the Connections and Ticks them
 * @param dt The time Passed in Seconds

   This function brings a flow into the whole animation
*/
void PhysicsEngine::tick(float dt)
{
  /* go through all the PhysicsInterface(s) and tick them,
  meaning let the fields work */
  list<PhysicsConnection*>::iterator pc;
  for (pc = this->connections.begin(); pc != this->connections.end(); pc++)
    (*pc)->apply();

  /* actually tick all the PhysicsInterfaces. Move the objects around */
  if (this->interfaces != NULL || (this->interfaces = ClassList::getList(CL_PHYSICS_INTERFACE)) != NULL)
  {
    list<BaseObject*>::const_iterator tickPhys;
    for (tickPhys = this->interfaces->begin(); tickPhys != this->interfaces->end(); tickPhys++)
      dynamic_cast<PhysicsInterface*>(*tickPhys)->tickPhys(dt);
  }
}



/**
 *  print out interesting debug information of this class
*/
void PhysicsEngine::debug() const
{
  PRINT(0)("====================================\n");
  PRINT(0)("= Physics-Engine debug information =\n");
  PRINT(0)("====================================\n");
  PRINT(0)(" reference: %p\n", this);
  if (this->interfaces != NULL)
    PRINT(0)(" number of Interfaces: %d\n", this->interfaces->size());
  PRINT(0)(" number of Fields: %d\n", this->fields.size());
  PRINT(0)(" number of Connections: %d\n", this->connections.size());

  PRINT(0)("==============================PHYS==\n");
}
