/*!
 * @file animation.h
    A Subclass for all animations in orxonox
*/

#ifndef _ANIMATION_H
#define _ANIMATION_H

#include "list.h"
#include "base_object.h"
#include "confincl.h"  // must be here to determin the DEBUG-level
#include "debug.h"

// FORWARD DECLARATION

//! An enumerator of Functions to describe the flow of the Animation
/**
   @todo check with Patrick it of

   description in speed to the next keyframe:
   ANIM_CONSTANT: 0, infinity.
   ANIM_LINEAR: equal
   ANIM_SINE: fast, slow, fast
   ANIM_COSINE: slow, fast, slow
   ANIM_EXP: slow, fast
   ANIM_NEG_EXP: fast, slow
   ANIM_RANDOM: eratic

   ANIM_NULL: !!DO NOT USE THIS!! only for internal handling

   deprecated QUADRATIC
*/
typedef enum ANIM_FUNCTION
{
  ANIM_NULL         = 0,
  ANIM_CONSTANT     = 1,
  ANIM_LINEAR       = 2,
  ANIM_SINE         = 3,
  ANIM_COSINE       = 4,
  ANIM_EXP          = 5,
  ANIM_NEG_EXP      = 6,
  ANIM_QUADRATIC    = 7,
  ANIM_RANDOM       = 8,
};

#define ANIM_DEFAULT_FUNCTION ANIM_LINEAR      //!< A default function to choose from the above set

//! An enumerator describing what the animation should do after the last keyframe.
/**
   ANIM_INF_CONSTANT stays at the end of the animation
   ANIM_INF_REPLAY loops back to the beginning and replays the animation
   ANIM_INF_REWIND loops back to the beginning and then stops the animation
   ANIM_INF_DELETE deletes the animation. !! THIS IS DANGEROUS !! only do this with non-class variables
*/
typedef enum ANIM_INFINITY
{
  ANIM_INF_CONSTANT,
  ANIM_INF_REPLAY,
  ANIM_INF_REWIND,
  ANIM_INF_DELETE
}; //, ANIM_INF_LINEAR, ANIM_INF_PINGPONG};

//! A Superclass for describing an animation (all animations will be derived from this one)
/** implement in subclasses:
 *
 * De-/Constructor
 * Animation Functions
 * virtual tick
 * addKeyFrame
 * List of keyFrames
 * currentKeyFrame/nextKeyFrame
 * virtual rewind, to go to the first Keyframe. (other functions will call this one)
*/
class Animation : public BaseObject
{
 public:
  virtual ~Animation();

  void setInfinity(ANIM_INFINITY postInfinity = ANIM_INF_CONSTANT);

  void play(); // equals resume();
  void playNextKeyframes(int n = 1);
  void stop();
  void pause();
  void replay();
  //! A virtual function that should change to the first keyframe.
  virtual void rewind() = 0;

  /** A virtual function that ticks the animation @param dt the time passed */
  virtual void tick(float dt) = 0;

  /** @returns the BaseObject, this animation operates on */
  BaseObject* getBaseObject() const { return this->baseObject; };

  /** @returns if the Animation should be deleted */
  inline bool ifDelete() { return bDelete; };

 protected:
  Animation();

  void handleInfinity();

 protected:
  // variables
  float                 localTime;              //!< The Time passed since the beginning of the currentKeyFrame.
  ANIM_INFINITY         postInfinity;           //!< describing what the animation should do after the last keyframe.

  BaseObject*           baseObject;             //!< The same as object in the derived classes, but with reference to BaseObject
  unsigned int          keyFrameCount;          //!< The Count of KeyFrames.
  int                   keyFramesToPlay;        //!< How many more Keyframes to play. if negative it will be ignored if 0 stop.
  bool                  bRunning;               //!< If the animation is running
  bool                  bDelete;                //!< If true, the animation will be deleted through the AnimationPlayer.
};




/**********************TEST*******************************/
//! a simple testClass for the animation
class aTest
{
 public:
  inline aTest() { last = 0.0;}
  /** a little debug information to show the results of this class @param f new value */
  inline void littleDebug(float f) {  diff = f - last; printf("f=%f, diff=%f\n", f,diff); last = f;}
 private:
  float     diff;           //!< difference from the last value
  float     last;           //!< the last calculated value
};

//aTest::aTest() {}
//aTest::~aTest() {}

//void aTest::littleDebug(float f)

/**********************TEST*******************************/


#endif /* _ANIMATION_H */
