/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "text.h"
#include "font.h"

#include "graphics_engine.h"
#include "resource_manager.h"
#include "class_list.h"
#include "debug.h"
#include "p_node.h"

using namespace std;

/**
 *  creates a new Text Element
 * @param fontFile the Font to render this text in
 * @param type The renderType to display this font in
 */
Text::Text(const char* fontFile, unsigned int textSize)
{
  this->init();

  if (fontFile != NULL)
    this->setFont(fontFile, FONT_DEFAULT_RENDER_SIZE);
  this->setSizeY2D(textSize);
}

/**
 *  deletes a Text out of memory
 *
 * This also ereases the text from the textList of the TextEngine
 */
Text::~Text()
{
  if (this->font != NULL && this->font != Font::getDefaultFont())
    ResourceManager::getInstance()->unload(this->font);

  if (this->text)
    delete[] this->text;
}

/**
 * initializes Text
 */
void Text::init()
{
  this->setClassID(CL_TEXT, "Text");

  // initialize this Text
  this->font = NULL;
  this->text = NULL;
  this->externText = NULL;
  this->setAlignment(TEXT_DEFAULT_ALIGNMENT);
  this->blending = TEXT_DEFAULT_BLENDING;
  this->color = TEXT_DEFAULT_COLOR;
  this->setSize(TEXT_DEFAULT_SIZE);
  this->setText(NULL);
}

/**
 * sets the Font of this Text to font from fontFile
 * @param fontFile the File to load the Font from.
 * @param fontSize the Size of the Font
 */
void Text::setFont(const char* fontFile, unsigned int fontSize)
{
  Font* tmpFont;
  Text* newText;
  Vector tmpVec;

  // unloading the Font if we alrady have one loaded.
  if (this->font != NULL && this->font != Font::getDefaultFont())
    ResourceManager::getInstance()->unload(this->font);
  this->font = NULL;

  // load a new Font
  if (fontFile != NULL)
  {
    tmpFont = (Font*)ResourceManager::getInstance()->load(fontFile, TTF, RP_GAME, &fontSize);
    if (tmpFont != NULL)
      this->font = tmpFont;
    else
      PRINTF(2)("Font %s could not be loaded, probably file not found\n", fontFile);
  }
}

/**
 * Sets a new Text to the font
 * @param text the new text to set
 */
void Text::setText(const char* text, bool isExtern)
{
  if (isExtern)
  {
    this->externText = text;

    if (unlikely(this->text != NULL))
    {
      delete[] this->text;
      this->text = NULL;
    }
  }
  else
  {
    this->externText = NULL;
    if (this->text)
      delete[] this->text;
    if (text != NULL)
    {
      this->text = new char[strlen(text)+1];
      strcpy(this->text, text);
    }
    else
      this->text = NULL;
  }

  // setting up the Text-Width if DYNAMIC
//  if (this->type & TEXT_RENDER_DYNAMIC && this->getAlignment() != TEXT_ALIGN_LEFT && this->font != NULL)
  const Font* calcSizeFont = this->font;
  if (calcSizeFont != NULL || (calcSizeFont = Font::getDefaultFont()) != NULL)
  {
    Glyph** glyphArray = calcSizeFont->getGlyphArray();

    float width = 0;
    const char* tmpText = this->externText;
    if (this->externText == NULL)
      tmpText = this->text;
    if (tmpText != NULL)
    {
      while (*tmpText != '\0')
      {
        if(glyphArray[*tmpText] != NULL)
        {
          width += glyphArray[*tmpText]->advance;
        }
        tmpText++;
      }
      this->setSizeX2D(width *this->getSizeY2D());
    }
  }
}

/**
 *  draws the Text
 */
void Text::draw() const
{
  glPushMatrix();
  // transform for alignment.
  if (this->getAlignment() == TEXT_ALIGN_RIGHT)
    glTranslatef(-this->getSizeX2D(), 0, 0);
  else if (this->getAlignment() == TEXT_ALIGN_CENTER || this->getAlignment() == TEXT_ALIGN_SCREEN_CENTER)
    glTranslatef(-this->getSizeX2D()/2, 0, 0);

  // drawing this Text.
  // setting the Blending effects
  glColor4f(this->color.x, this->color.y, this->color.z, this->blending);
  glEnable(GL_BLEND);
  glEnable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, GL_MODULATE );

  Glyph** glyphArray;
  if (likely (this->font != NULL))
  {
    glyphArray = this->font->getGlyphArray();
    glBindTexture(GL_TEXTURE_2D, font->getTexture());
  }
  else
  {
    if (unlikely(Font::getDefaultFont() == NULL))
      Font::initDefaultFont();
    glyphArray = Font::getDefaultFont()->getGlyphArray();
    glBindTexture(GL_TEXTURE_2D, Font::getDefaultFont()->getTexture());
  }
  const char* tmpText = this->externText;
  if (this->externText == NULL)
    tmpText = this->text;
  if (likely(tmpText != NULL))
  {
    glTranslatef(getAbsCoor2D().x, getAbsCoor2D().y, 0);
    glRotatef(this->getAbsDir2D(), 0, 0, 1);
    Glyph* tmpGlyph;
    float posX = 0.0f;
    while (likely(*tmpText != '\0'))
    {
      if(likely((tmpGlyph = glyphArray[*tmpText]) != NULL))
      {
        glBegin(GL_QUADS);

        glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[2]);
        glVertex2d(posX, 0);

        glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[3]);
        glVertex2d(posX, this->getSizeY2D());

        glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[3]);
        glVertex2d(posX+tmpGlyph->width*this->getSizeY2D(), this->getSizeY2D());

        glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[2]);
        glVertex2d(posX+tmpGlyph->width*this->getSizeY2D(), 0);

        glEnd();
        glEndList();
        posX += tmpGlyph->advance * this->getSizeY2D();
      }
      ++tmpText;
    }
  }

  glPopMatrix();
}

/**
 *  prints out some nice debug information about this text
 */
void Text::debug() const
{
  if (this->externText == NULL)
    PRINT(0)("=== TEXT: %s ===\n", this->text);
  else
    PRINT(0)("=== TEXT: %s ===\n", this->externText);

  if (this->getBindNode())
    PRINT(0)("is bind to %s; ref=%p\n", this->getBindNode()->getName(), this->getBindNode());
  PRINT(0)("Color: %0.2f %0.2f %0.2f\n", this->color.x, this->color.y, this->color.z);
}


////////////
/// UTIL ///
////////////
/**
 * Loads a Font from an SDL_surface into a texture.
 * @param surface The surface to make the texture of
 * @param texCoord The texture coordinates of the 4 corners of the texture
 * @returns the ID of the texture
 */
GLuint Text::loadTexture(SDL_Surface *surface, TexCoord* texCoord)
{
  GLuint texture;
  int w, h;
  SDL_Surface *image;
  SDL_Rect area;
  Uint32 saved_flags;
  Uint8  saved_alpha;

  /* Use the surface width and height expanded to powers of 2 */
  w = powerOfTwo(surface->w);
  h = powerOfTwo(surface->h);
  if (texCoord != NULL)
  {
    texCoord->minU = 0.0f;
    texCoord->minV = 0.0f;
    texCoord->maxU = (GLfloat)surface->w / w;
    texCoord->maxV = (GLfloat)surface->h / h;
  }
  image = SDL_CreateRGBSurface(SDL_SWSURFACE,
                               w, h,
                               32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
                               0x000000FF,
                               0x0000FF00,
                               0x00FF0000,
                               0xFF000000
#else
                                   0xFF000000,
                               0x00FF0000,
                               0x0000FF00,
                               0x000000FF
#endif
                              );
  if ( image == NULL ) {
    return 0;
  }

  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, 0, 0);
  }

  /* Copy the surface into the GL texture image */
  area.x = 0;
  area.y = 0;
  area.w = surface->w;
  area.h = surface->h;
  SDL_BlitSurface(surface, &area, image, &area);

  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, saved_flags, saved_alpha);
  }

  /* Create an OpenGL texture for the image */
  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexImage2D(GL_TEXTURE_2D,
               0,
               GL_RGBA,
               w, h,
               0,
               GL_RGBA,
               GL_UNSIGNED_BYTE,
               image->pixels);
  SDL_FreeSurface(image); /* No longer needed the data */

  return texture;
}

/**
 *  Quick utility function for texture creation
 * @param input an integer
 * @returns the next bigger 2^n-integer than input
 */
int Text::powerOfTwo(int input)
{
  int value = 1;

  while ( value < input )
    value <<= 1;
  return value;
}
