/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_LOADING

#include "load_param.h"
#include "load_param_description.h"

#include "list.h"

#include <stdarg.h>

/**
 * Constructs a new LoadParameter
 * @param root the XML-element to load this Parameter from
 * @param paramName the Parameter to load
 * @param object the BaseObject, to load this parameter on to (will be cast to executor's Parameter)
 * @param executor the Executor, that executes the loading procedure.
 */
CLoadParam::CLoadParam(const TiXmlElement* root, const char* paramName, BaseObject* object, const Executor& executor, bool inLoadCycle)
{
  this->paramName = paramName;
  this->object = object;
  this->inLoadCycle = inLoadCycle;

  // determin the LoadString.
  if (likely(!inLoadCycle))
    this->loadString = grabParameter(root, paramName);
  else
  {
    if (!strcmp(root->Value(), paramName))
    {
      const TiXmlNode* val = root->FirstChild();
      if( val->ToText())
        this->loadString = val->Value();
    }
    else
      this->loadString = NULL;
  }

  // set the Executor.
  this->executor = executor.clone();
}

/**
 * This is a VERY SPECIAL deconsrtuctor.
 * It is made, so that it loads the Parameters on destruction.
 * meaning, if an Executor a valid Object exist, and all
 * Execution-Conditions are met, they are executed here.
 */
CLoadParam::~CLoadParam()
{
  if (likely(this->executor != NULL))
  {
    if (likely(this->object != NULL && this->executor != NULL) &&
        ( this->loadString != NULL ||
         ((this->executor->getType() & Executor_NoLoadString) == Executor_NoLoadString)))
    {
      PRINTF(4)("Loading '%s' with Parameters '%s' onto: '%s'(%s)\n", this->paramName, this->loadString, this->object->getName(), this->object->getClassName());
      this->executor->execute(this->object, this->loadString);
    }
    delete this->executor;
  }

}

/**
 * set the default values of the executor
 * @param count how many default values to set.
 * @param ... the default values !! must be at least count parameters!!
 */
CLoadParam& CLoadParam::defaultValues(unsigned int count, ...)
{
  va_list values;
  va_start(values, count);

  assert(executor != NULL);
  this->executor->defaultValues(count, values);

  return *this;
}



/**
 * @param descriptionText The text to set as a description for this Parameter
 * @returns a pointer to itself.
*/
CLoadParam& CLoadParam::describe(const char* descriptionText)
{
  if (LoadClassDescription::parametersDescription && this->paramDesc && !this->paramDesc->getDescription())
    {
      this->paramDesc->setDescription(descriptionText);
    }
  return *this;
}

// const LoadParamDescription* LoadParamDescription::getClass(const char* className)
// {
//   tIterator<LoadClassDescription>* iterator = LoadClassDescription::classList->getIterator();
//   LoadClassDescription* enumClassDesc = iterator->firstElement();
//   while (enumClassDesc)
//   {
//     if (!strcmp(enumClassDesc->className, classNameBegin, className))
//     {
//       delete iterator;
//       return enumClassDesc;
//     }
//     enumClassDesc = iterator->nextElement();
//   }
//   delete iterator;
//
//   return NULL;
// }




/*
 * @param object The object this Parameter is loaded too.
 * @param root: the XML-element to load this option from.
 * @param paramName: The name of the parameter loaded.
 * @param paramCount: how many parameters this loading-function takes
 * @param multi: if false LoadParam assumes only one occurence of this parameter in root, if true it assumes multiple occurences.
 * @param ...: the parameter information (1. Parameter, 2. Default Value for the Parameter, ...)
*/
/*LoadParam::LoadParam(const TiXmlElement* root, BaseObject* object, const char* paramName,
                             int paramCount, bool multi, const void* pointerToParam, ...)
{
  this->setClassID(CL_LOAD_PARAM, "LoadParam");
  this->executor = NULL;

  this->loadString = NULL;
  this->pointerToParam = pointerToParam;

  if (paramCount == 0 || this->pointerToParam != NULL)
    this->loadString = "none";
  else
{
      if (likely(!multi))
        this->loadString = grabParameter(root, paramName);
      else
{
          if (!strcmp(root->Value(), paramName))
{
              const TiXmlNode* val = root->FirstChild();
              if( val->ToText())
                this->loadString = val->Value();
}
}
}

  this->paramDesc = NULL;
  if (LoadClassDescription::parametersDescription)
{
    // locating the class
    this->classDesc = LoadClassDescription::addClass(object->getClassName());

    if ((this->paramDesc = this->classDesc->addParam(paramName)) != NULL)
{

      this->paramDesc->paramCount = paramCount;
      this->paramDesc->types = new int[paramCount];
      this->paramDesc->defaultValues = new char*[paramCount];

      va_list types;
      va_start (types, pointerToParam);
      char defaultVal[512];
      for(int i = 0; i < paramCount; i++)
{
        defaultVal[0] = '\0';
          // parameters parsed
        int tmpType = va_arg (types, int);
        this->paramDesc->types[i] = tmpType;
        switch (tmpType)
{
  case MT_INT:
            sprintf(defaultVal, "%d", va_arg(types, int));
            break;
//          case MT_LONG:
//            sprintf(defaultVal, "%0.3f", va_arg(types, l_LONG_TYPE));
//            break;
  case MT_FLOAT:
            sprintf(defaultVal, "%0.3f", va_arg(types, double));
            break;
  case MT_STRING:
            sprintf(defaultVal, "%s", va_arg(types, l_STRING_TYPE));
            break;
  case MT_EXT1:
            sprintf(defaultVal, "");
            break;
}
        this->paramDesc->defaultValues[i] = new char[strlen(defaultVal)+1];
        strcpy(this->paramDesc->defaultValues[i], defaultVal);
}
      va_end(types);

      int argCount = 0;
}
}
}*/










//////////////////////
// HELPER FUNCTIONS //
//////////////////////
/**
 * @param root: The XML-element to grab a parameter from
 * @param parameterName: the parameter to grab
 * @returns the Value of the parameter if found, NULL otherwise
*/
const char* grabParameter(const TiXmlElement* root, const char* parameterName)
{
  const TiXmlElement* element;
  const TiXmlNode* node;

  if (root == NULL || parameterName == NULL)
    return NULL;
  assert( parameterName != NULL);

  element = root->FirstChildElement( parameterName);
  if( element == NULL) return NULL;

  node = element->FirstChild();
  while( node != NULL)
    {
      if( node->ToText()) return node->Value();
      node = node->NextSibling();
    }
  return NULL;
}
