/*!
 * @file bv_tree.h
  *  Definition of a bounding volume tree

 */

#ifndef _OBB_TREE_NODE_H
#define _OBB_TREE_NODE_H

#include "bv_tree_node.h"



// FORWARD DECLARATION
class BoundingVolume;
class OBB;
class OBBTree;
class Plane;
class PNode;
//struct sVec3D;

//! A class that represents a bounding volume tree
class OBBTreeNode : public BVTreeNode {


  public:
    OBBTreeNode(const OBBTree* tree);
    OBBTreeNode(const OBBTree* tree, bool bRoot);
    virtual ~OBBTreeNode();

    /*  this function returns the bounding volume of this tree node @return: returns the BV */
    virtual inline const BoundingVolume* getBV() const { return (BoundingVolume*)this->bvElement; }
    
    virtual void spawnBVTree(const int depth, const sVec3D *verticesList, unsigned int length);
    virtual void spawnBVTree(const int depth, const modelInfo& modelInf, 
                             const int* triangleIndexes, unsigned int length);

    virtual void collideWith(BVTreeNode* treeNode, WorldEntity* nodeA, WorldEntity* nodeB);

    virtual void drawBV(int depth, int drawMode, const Vector& color = Vector(1,0,0), bool top = true) const;

    void debug() const;

  private:
    void calculateBoxCovariance(OBB* box, const sVec3D* verticesList, unsigned int length);
    void calculateBoxEigenvectors(OBB* box, const sVec3D* verticesList, unsigned int length);
    void calculateBoxAxis(OBB* box, const sVec3D* verticesList, unsigned int length);

    void calculateBoxCovariance(OBB* box, const modelInfo& modelInf, const int* triangleIndexes, unsigned int length);
    void calculateBoxEigenvectors(OBB* box, const modelInfo& modelInf, const int* triangleIndexes, unsigned int length);
    void calculateBoxAxis(OBB* box, const modelInfo& modelInf, const int* triangleIndexes, unsigned int length);


    void forkBox(OBB* box);

    bool overlapTest(OBB* boxA, OBB* boxB, WorldEntity* nodeA, WorldEntity* nodeB);

  protected:
    OBB*                bvElement;                  //!< the obb element
    OBBTreeNode*        nodeLeft;                   //!< ref to the left tree node
    OBBTreeNode*        nodeRight;                  //!< ref to the right tree node

  private:
    unsigned int        treeIndex;                  //!< Index number of the BV in the tree
    const sVec3D*       vertices;                   //!< pointer to the vertices data
    int                 numOfVertices;              //!< number of vertices in vertices data
    int                 depth;                      //!< the depth of the node in the tree
    const OBBTree*      obbTree;                    //!< reference to the obb tree
    Plane               separationPlane;            //!< the separation plane of the obb
    const sVec3D*       sepPlaneCenter;             //!< only needed to draw plane
    int                 longestAxisIndex;           //!< only needed to draw plane

    /* tmp saving place for obb variables */
    sVec3D*             tmpVert1;                   //!< pointer to the vert data of obbox1
    sVec3D*             tmpVert2;                   //!< pointer to the vert data of obbox1
    int                 tmpLen1;                    //!< len vert data obbox1
    int                 tmpLen2;                    //!< len vert data obbox2

    static float**      coMat;
    static float**      eigvMat;
    static float*       eigvlMat;
    static int*         rotCount;
};

#endif /* _OBB_TREE_NODE_H */
