/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: Benjamin Grauer

   2005-06-10: some naming conventions
*/


/**
 *  breaks a string into parts that were initially seperated by comma
 * @param string the string to break into substrings
*/

#include "substring.h"

#include "debug.h"
#include <string.h>
#include <assert.h>

SubString::SubString( const char* string, char splitter)
{
  this->splittersCount = 0;
  if (string == NULL)
  {
    this->strings = NULL;
    this->offsets = NULL;
    return;
  }

  for( int i = 0; i < strlen(string); i++)
    if( string[i] == splitter)
      this->splittersCount++;

  this->splittersCount += 1;

  this->strings = new char*[this->splittersCount];
  this->offsets = new unsigned int[this->splittersCount];
  assert (this->strings != NULL && this->offsets != NULL);

  int i = 0;
  int l = 0;

  if( this->splittersCount > 1)
  {
    const char* offset = string;
    const char* end = strchr( string, splitter);
    while( end != NULL)
    {
      assert( i < this->splittersCount);
      l = end - offset;
      this->strings[i] = new char[l + 1];
      assert( strings[i] != NULL);
      strncpy( strings[i], offset, l);
      strings[i][l] = '\0';
      this->offsets[i] = offset - string;
      i++;
      end++;
      offset = end;
      end = strchr( offset, splitter);
    }

    l = strlen( offset);
    strings[i] = new char[l + 1];
    strncpy( strings[i], offset, l);
    strings[i][l] = '\0';
    this->offsets[i] = offset - string;
  }
  else
  {
    this->strings[0] = new char[strlen(string)+1];
    strcpy(this->strings[0], string);
    this->offsets[0] = 0;
  }
}


/**
 * Splits a String into a Substring removing all whiteSpaces
 * @param string the String to Split
 * @param whiteSpaces MUST BE __TRUE__
 *
 */
SubString::SubString(const char* string, bool whiteSpaces)
{
  this->splittersCount = 0;
  if (string == NULL || whiteSpaces == false)
  {
    this->strings = NULL;
    this->offsets = NULL;
    return;
  }

  // chop the input to the beginning of something usefull
  if (strlen(string) > 0)
    string = string + strspn(string, " \t\n");

  // count the Splitters
  bool lastWasWhiteSpace = false;
  for(unsigned int i = 0; i < strlen(string); i++)
    if( string[i] == ' ' || string[i] == '\t' || string[i] == '\n' )
      lastWasWhiteSpace = true;
    else
    {
      if (lastWasWhiteSpace)
        this->splittersCount ++;
      lastWasWhiteSpace = false;
    }
  this->splittersCount += 1;

  // allocate memory
  this->strings = new char*[this->splittersCount];
  this->offsets = new unsigned int[this->splittersCount];
  assert (this->strings != NULL && this->offsets != NULL);


  // split the String into substrings
  int l = 0;
  unsigned int i = 0;
  if( this->splittersCount > 1)
  {
    const char* offset = string;
    const char* end = offset + strcspn(offset, " \t\n");
    for (i = 0; i < this->splittersCount; i++)
    {
      assert( i < this->splittersCount);
      l = end - offset;
      this->strings[i] = new char[l + 1];
      assert( strings[i] != NULL);
      strncpy( strings[i], offset, l);
      strings[i][l] = '\0';
      this->offsets[i] = offset - string;
      end += strspn(end, " \t\n");
      offset = end;
      end = offset + strcspn(offset, " \t\n");
    }
  }
  else
  {
    unsigned int length = strcspn(string, " \t\n");
    this->strings[0] = new char[length+1];
    strncpy(this->strings[0], string, length);
    this->strings[0][length] = '\0';
    offsets[0] = 0;
  }
}

SubString::SubString(const char* string, const char* splitters, char escapeChar)
{
  this->splittersCount = 0;
  if (string == NULL || splitters == NULL)
  {
    this->strings = NULL;
    this->offsets = NULL;
    return;
  }

  // chop the input to the beginning of something usefull
  if (strlen(string) > 0)
    string = string + strspn(string, splitters);

  // count the Splitters
  bool lastWasSplitter = false;
  for(unsigned int i = 0; i < strlen(string); i++)
  {

    if( strchr(splitters, string[i] ))
      lastWasSplitter = true;
    else
    {
      if (lastWasSplitter)
      {
        this->splittersCount ++;
        lastWasSplitter = false;
      }
    }
  }
  this->splittersCount += 1;

  // allocate memory
  this->strings = new char*[this->splittersCount];
  this->offsets = new unsigned int[this->splittersCount];
  assert (this->strings != NULL && this->offsets != NULL);


  // split the String into substrings
  int l = 0;
  unsigned int i = 0;
  if( this->splittersCount > 1)
  {
    const char* offset = string;
    const char* end = offset + strcspn(offset, splitters);
    for (i = 0; i < this->splittersCount; i++)
    {
      assert( i < this->splittersCount);
      l = end - offset;
      this->strings[i] = new char[l + 1];
      assert( strings[i] != NULL);
      strncpy( strings[i], offset, l);
      strings[i][l] = '\0';
      this->offsets[i] = offset - string;
      end += strspn(end, splitters);
      offset = end;
      end = offset + strcspn(offset, splitters);
    }
  }
  else
  {
    unsigned int length = strcspn(string, splitters);
    this->strings[0] = new char[length+1];
    strncpy(this->strings[0], string, length);
    this->strings[0][length] = '\0';
    offsets[0] = 0;
  }
}


/**
 *  removes the object from memory
*/
SubString::~SubString()
{
  if (this->strings)
  {
    for(unsigned int i = 0; i < this->splittersCount; i++)
      delete[] this->strings[i];
    delete[] this->strings;
  }
  delete[] this->offsets;
}

/**
 *  get a particular substring
 * @param i the ID of the substring to return
 * @returns the designated substring or NULL if an invalid ID was given
*/
const char* SubString::getString(unsigned int i)
{
  if( i < this->splittersCount && i >= 0)
    return this->strings[i];
  else
    return NULL;
}

/**
 * get a particular substring's offset
 * @param i the ID of the substring to get the offset from
 * @returns the offset or NULL if an invalid ID was given
 */
unsigned int SubString::getOffset(unsigned int i)
{
  if( i < this->splittersCount && i >= 0)
    return this->offsets[i];
  else
    return 0;
}

/**
 * Some nice debug information about this SubString
 */
void SubString::debug() const
{
  PRINT(0)("Substring-information::count=%d ::", this->splittersCount);
  if (this->strings != NULL)
    for (unsigned int i = 0; i < this->splittersCount; i++)
     PRINT(0)("s%d='%s'::", i, this->strings[i]);
  PRINT(0)("\n");
}
