/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_command.h"
#include "shell_command_class.h"

#include "list.h"
#include "debug.h"
#include "class_list.h"

#include "key_names.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

using namespace std;

/**
 * constructs and registers a new Command
 * @param commandName the name of the Command
 * @param className the name of the class to apply this command to
 * @param paramCount the count of parameters this command takes
 */
ShellCommand::ShellCommand(const char* commandName, const char* className, const Executor& executor)
{
  this->setClassID(CL_SHELL_COMMAND, "ShellCommand");
  this->setName(commandName);
  this->description = NULL;
  this->alias = NULL;
  this->executor = executor.clone();

//  this->classID = classID;
  this->shellClass = ShellCommandClass::getCommandClass(className); //ClassList::IDToString(classID);
  if (this->shellClass != NULL)
    this->shellClass->commandList->add(this);
}

/**
 * deconstructs a ShellCommand
 */
ShellCommand::~ShellCommand()
{
  if (this->alias != NULL && ShellCommandClass::aliasList != NULL)
  {
    ShellCommandClass::aliasList->remove(this->alias);
    delete this->alias;
  }
  delete this->executor;
}

/**
 * registers a new ShellCommand
 */
ShellCommand* ShellCommand::registerCommand(const char* commandName, const char* className, const Executor& executor)
{
  if (ShellCommand::isRegistered(commandName, className, executor))
    return NULL;
  else
    return new ShellCommand(commandName, className, executor);

}

/**
 * unregister an existing commandName
 * @param className the name of the Class the command belongs to.
 * @param commandName the name of the command itself
 */
void ShellCommand::unregisterCommand(const char* commandName, const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    ShellCommandClass::initCommandClassList();

 const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);

 if (checkClass != NULL)
  {
    tIterator<ShellCommand>* iterator = checkClass->commandList->getIterator();
    ShellCommand* elem = iterator->firstElement();
    while(elem != NULL)
    {
      if (!strcmp(commandName, elem->getName()))
      {
        checkClass->commandList->remove(elem);
        delete elem;
        break;
      }
      elem = iterator->nextElement();
    }
    delete iterator;

    if (checkClass->commandList->getSize() == 0)
    {
      ShellCommandClass::commandClassList->remove(checkClass);
      delete checkClass;
    }
  }
}

/**
 * checks if a command has already been registered.
 * @param commandName the name of the Command
 * @param className the name of the Class the command should apply to.
 * @param paramCount how many arguments the Command takes
 * @returns true, if the command is registered/false otherwise
 *
 * This is used internally, to see, if we have multiple command subscriptions.
 * This is checked in the registerCommand-function.
 */
bool ShellCommand::isRegistered(const char* commandName, const char* className, const Executor& executor)
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    ShellCommandClass::initCommandClassList();
    return false;
  }

  const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);
  if (checkClass != NULL)
  {
    tIterator<ShellCommand>* iterator = checkClass->commandList->getIterator();
    ShellCommand* elem = iterator->firstElement();
    while(elem != NULL)
   {
     if (!strcmp(commandName, elem->getName()))
     {
       PRINTF(2)("Command already registered\n");
       delete iterator;
       return true;
      }
     elem = iterator->nextElement();
   }
   delete iterator;
   return false;
  }
  else
    return false;
}


/**
 * executes commands
 * @param executionString the string containing the following input
 * ClassName [ObjectName] functionName [parameter1[,parameter2[,...]]]
 * @return true on success, false otherwise.
 */
bool ShellCommand::execute(const char* executionString)
{
  if (ShellCommandClass::commandClassList == NULL)
    return false;

  long classID = CL_NULL;                 //< the classID retrieved from the Class.
  ShellCommandClass* commandClass = NULL; //< the command class this command applies to.
  tList<BaseObject>* objectList = NULL;   //< the list of Objects stored in classID
  BaseObject* objectPointer = NULL;       //< a pointer to th Object to Execute the command on
  bool emptyComplete = false;             //< if the completion input is empty string. e.g ""
  unsigned int fktPos = 1;                //< the position of the function (needed for finding it)
//  long completeType = SHELLC_NONE;      //< the Type we'd like to complete.
  SubString inputSplits(executionString, true);

  if (inputSplits.getCount() == 0)
    return false;
  if (inputSplits.getCount() >= 1)
  {
    // CHECK FOR ALIAS
    if (ShellCommandClass::aliasList != NULL)
    {
      tIterator<ShellCommandAlias>* itAL = ShellCommandClass::aliasList->getIterator();
      ShellCommandAlias* elemAL = itAL->firstElement();
      while(elemAL != NULL)
      {
        if (elemAL->getName() != NULL && !strcmp(elemAL->getName(), inputSplits.getString(0)) && elemAL->getCommand() != NULL &&
            elemAL->getCommand()->shellClass != NULL )
        {
          objectList = ClassList::getList(elemAL->getCommand()->shellClass->getName());
          if (objectList != NULL)
          {
            if (inputSplits.getCount() > 1)
              elemAL->getCommand()->executor->execute(objectList->firstElement(), executionString+inputSplits.getOffset(1));
            else
              elemAL->getCommand()->executor->execute(objectList->firstElement(), "");
            delete itAL;
            return true;
          }
        }
        elemAL = itAL->nextElement();
      }
      delete itAL;
    }
    // looking for a Matching Class
    if (likely(ShellCommandClass::commandClassList != NULL))
    {
      tIterator<ShellCommandClass>* itCL = ShellCommandClass::commandClassList->getIterator();
      ShellCommandClass* elemCL = itCL->firstElement();
      while(elemCL != NULL)
      {
        if (elemCL->getName() && !strcasecmp(inputSplits.getString(0), elemCL->getName()))
        {
          //elemCL->getName();
          classID = ClassList::StringToID(elemCL->getName());
          commandClass = elemCL;
          objectList = ClassList::getList(classID);
          break;
        }
        elemCL = itCL->nextElement();
      }
      delete itCL;
    }

    if (commandClass != NULL && inputSplits.getCount() >= 2)
    {
      if (objectList != NULL)
      {
        // Checking for a Match in the Objects of classID (else take the first)
        tIterator<BaseObject>* itBO = objectList->getIterator();
        BaseObject* enumBO = itBO->firstElement();
        while(enumBO)
        {
          if (enumBO->getName() != NULL && !strcasecmp(enumBO->getName(), inputSplits.getString(1)))
          {
            objectPointer = enumBO;
            fktPos = 2;
            break;
          }
          enumBO = itBO->nextElement();
         }
         delete itBO;

      //
        if (objectPointer == NULL)
          objectPointer = objectList->firstElement();
      }
      // match a function.
      if (commandClass != NULL && (fktPos == 1 || (fktPos == 2 && inputSplits.getCount() >= 3)))
      {
        tIterator<ShellCommand>* itCMD = commandClass->commandList->getIterator();
        ShellCommand* enumCMD = itCMD->firstElement();
        while (enumCMD != NULL)
        {
          if (!strcmp(enumCMD->getName(), inputSplits.getString(fktPos)))
          {
            if (objectPointer == NULL && enumCMD->executor->getType() & Executor_Objective)
            {
              delete itCMD;
              return false;
            }
            if (inputSplits.getCount() > fktPos+1)
              enumCMD->executor->execute(objectPointer, executionString+inputSplits.getOffset(fktPos +1));
            else
              enumCMD->executor->execute(objectPointer, "");
            delete itCMD;
            return true;
          }

          enumCMD = itCMD->nextElement();
        }
        delete itCMD;
      }
    }
  }
}

/**
 * lets a command be described
 * @param description the description of the Given command
 */
ShellCommand* ShellCommand::describe(const char* description)
{
  if (this == NULL)
    return NULL;
 else
 {
   this->description = description;
   return this;
 }
}

/**
 * adds an Alias to this Command
 * @param alias the name of the Alias to set
 * @returns itself
 */
ShellCommand* ShellCommand::setAlias(const char* alias)
{
  if (this == NULL)
    return NULL;

  if (this->alias != NULL)
  {
    PRINTF(2)("not more than one Alias allowed for functions (%s::%s)\n", this->getName(), this->shellClass->getName());
  }
  else
  {
    if (ShellCommandClass::aliasList == NULL)
      ShellCommandClass::aliasList = new tList<ShellCommandAlias>;

    ShellCommandAlias* aliasCMD = new ShellCommandAlias(alias, this);
    ShellCommandClass::aliasList->add(aliasCMD);
    this->alias = aliasCMD;
  }
  return this;
}

/**
 * sets default Values of the Commands
 * @param count how many default Values to set.
 * @param ... the default Values in order. They will be cast to the right type
 * @returns itself
 *
 * Be aware, that when you use this Function, you !!MUST!! match the input as
 * count, [EXACTLY THE SAME AS IF YOU WOULD CALL THE FUNCTION UP TO count ARGUMENTS]
 */
ShellCommand* ShellCommand::defaultValues(unsigned int count, ...)
{
  if (this == NULL)
    return NULL;

  va_list values;
  va_start(values, count);

  this->executor->defaultValues(count, values);

  return this;
}

/**
 * prints out nice information about the Shells Commands
 */
void ShellCommand::debug()
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    PRINT(0)("No Command registered.\n");
    return;
  }

  tIterator<ShellCommandClass>* iteratorCL = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elemCL = iteratorCL->firstElement();
  while(elemCL != NULL)
  {
    PRINT(0)("Class:'%s' registered %d commands: \n", elemCL->className, elemCL->commandList->getSize());
    tIterator<ShellCommand>* iterator = elemCL->commandList->getIterator();
    const ShellCommand* elem = iterator->firstElement();
    while(elem != NULL)
    {
      PRINT(0)("  command:'%s' : params:%d: ", elem->getName(), elem->executor->getParamCount());
      /// FIXME
      /*      for (unsigned int i = 0; i< elem->paramCount; i++)
       printf("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
      if (elem->description != NULL)
       printf("- %s", elem->description);
      printf("\n");

      elem = iterator->nextElement();
    }
    delete iterator;
    elemCL = iteratorCL->nextElement();
  }
  delete iteratorCL;
}

/**
 * converts a Parameter to a String
 * @param parameter the Parameter we have.
 * @returns the Name of the Parameter at Hand
 */
const char* ShellCommand::paramToString(long parameter)
{
  return MultiType::MultiTypeToString((MT_Type)parameter);
// FIXME
  /*  switch (parameter)
  {
    case ParameterBool:
      return "BOOL";
      break;
    case ParameterChar:
      return "CHAR";
      break;
    case ParameterString:
      return "STRING";
      break;
    case ParameterInt:
      return "INT";
      break;
    case ParameterUInt:
      return "UINT";
      break;
    case ParameterFloat:
      return "FLOAT";
      break;
    case ParameterLong:
      return "LONG";
      break;
    default:
      return "NULL";
      break;
  }*/
}
