/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_command_class.h"

#include "shell_command.h"

#include "list.h"
#include "debug.h"
#include "class_list.h"

#include <stdio.h>
#include <string.h>

using namespace std;

tList<ShellCommandClass>* ShellCommandClass::commandClassList = NULL;
tList<ShellCommandAlias>* ShellCommandClass::aliasList = NULL;

/**
 * creates a new ShellCommandClass
 * @param className the Name of the command-class to create
 */
ShellCommandClass::ShellCommandClass(const char* className)
{
  this->setClassID(CL_SHELL_COMMAND_CLASS, "ShellCommandClass");
  this->setName(className);

  this->className = className;
  this->classID = CL_NULL;
  this->commandList = new tList<ShellCommand>;

  ShellCommandClass::commandClassList->add(this);
}

/**
 * destructs the shellCommandClass again
 */
ShellCommandClass::~ShellCommandClass()
{
  tIterator<ShellCommand>* iterator = this->commandList->getIterator();
  ShellCommand* elem = iterator->firstElement();
  while(elem != NULL)
  {
    delete elem;
    elem = iterator->nextElement();
  }
  delete iterator;
  delete this->commandList;
}

/**
 * collects the Commands registered to some class.
 * @param className the name of the Class to collect the Commands from.
 * @param stringList a List to paste the Commands into.
 * @returns true on success, false otherwise
 */
bool ShellCommandClass::getCommandListOfClass(const char* className, tList<const char>* stringList)
{
  if (stringList == NULL || className == NULL)
    return false;

  tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elem = iterator->firstElement();
  while(elem != NULL)
  {
    if (!strcmp (elem->getName(), className))
    {
      tIterator<ShellCommand>* itFkt = elem->commandList->getIterator();
      ShellCommand* command = itFkt->firstElement();
      while (command != NULL)
      {
        stringList->add(command->getName());
        command = itFkt->nextElement();
      }
      delete itFkt;
    }

    elem = iterator->nextElement();
  }
  delete iterator;
  return true;
}

/**
 * collects the Aliases registered to the ShellCommands
 * @param stringList a List to paste the Aliases into.
 * @returns true on success, false otherwise
 */
bool ShellCommandClass::getCommandListOfAlias(tList<const char>* stringList)
{
  if (stringList == NULL || ShellCommandClass::aliasList == NULL)
    return false;

  tIterator<ShellCommandAlias>* iterator = ShellCommandClass::aliasList->getIterator();
   ShellCommandAlias* elem = iterator->firstElement();
   while(elem != NULL)
   {
     stringList->add(elem->getName());
     elem = iterator->nextElement();
   }
   delete iterator;
   return true;
}

/**
 * unregisters all Commands that exist
 */
void ShellCommandClass::unregisterAllCommands()
{
  if (ShellCommandClass::commandClassList != NULL)
  {
    // unregister all commands
    tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
    ShellCommandClass* elem = iterator->firstElement();
    while(elem != NULL)
    {
      delete elem;

      elem = iterator->nextElement();
    }
    delete iterator;

    delete ShellCommandClass::commandClassList;
    ShellCommandClass::commandClassList = NULL;
  }

  // unregister all aliases (there should be nothing to do here :))
  if (ShellCommandClass::aliasList != NULL)
  {
    tIterator<ShellCommandAlias>* itAL = ShellCommandClass::aliasList->getIterator();
    ShellCommandAlias* elemAL = itAL->firstElement();
    while(elemAL != NULL)
    {
      delete elemAL;
      elemAL = itAL->nextElement();
    }
    delete itAL;
    delete ShellCommandClass::aliasList;
    ShellCommandClass::aliasList = NULL;
  }
}

/**
 * checks if a Class is already registered to the Commands' class-stack
 * @param className the Name of the Class to check for
 * @returns the CommandClass if found, NULL otherwise
 */
const ShellCommandClass* ShellCommandClass::isRegistered(const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    initCommandClassList();

  tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elem = iterator->firstElement();
  while(elem != NULL)
  {
    if (!strcmp(className, elem->className))
    {
      if (elem->classID == CL_NULL)
        elem->classID = ClassList::StringToID(className);

      delete iterator;
      return elem;
    }
    elem = iterator->nextElement();
  }
  delete iterator;
  return NULL;
}

/**
 * searches for a CommandClass
 * @param className the name of the CommandClass
 * @returns the CommandClass if found, or a new CommandClass if not
 */
ShellCommandClass* ShellCommandClass::getCommandClass(const char* className)
{
  if (ShellCommandClass::commandClassList == NULL)
    initCommandClassList();

  tIterator<ShellCommandClass>* iterator = ShellCommandClass::commandClassList->getIterator();
  ShellCommandClass* elem = iterator->firstElement();
  while(elem != NULL)
  {
    if (!strcmp(className, elem->className))
    {
      delete iterator;
      return elem;
    }
    elem = iterator->nextElement();
  }
  delete iterator;
  return new ShellCommandClass(className);
}

/**
 * initializes the CommandList (if it is NULL)
 */
void ShellCommandClass::initCommandClassList()
{
  if (ShellCommandClass::commandClassList == NULL)
  {
    ShellCommandClass::commandClassList = new tList<ShellCommandClass>;
    ShellCommand::registerCommand("debug", "ShellCommand", ExecutorStatic<ShellCommand>(ShellCommand::debug));
  }
}

/**
 * displays help about ShellCommandClass
 * @param className: the Class of Commands to show help about
 */
void ShellCommandClass::help(const char* className)
{
  if (className == NULL)
    return;
  if (likely(ShellCommandClass::commandClassList != NULL))
  {
    tIterator<ShellCommandClass>* itCL = ShellCommandClass::commandClassList->getIterator();
    ShellCommandClass* elemCL = itCL->firstElement();
    while(elemCL != NULL)
    {
      if (elemCL->className && !strcasecmp(className, elemCL->className))
      {
        PRINT(0)("Class:'%s' registered %d commands: \n", elemCL->className, elemCL->commandList->getSize());
        tIterator<ShellCommand>* iterator = elemCL->commandList->getIterator();
        const ShellCommand* elem = iterator->firstElement();
        while(elem != NULL)
        {
          PRINT(0)("  command:'%s' : params:%d: ", elem->getName(), elem->executor->getParamCount());
          /// FIXME
          /*          for (unsigned int i = 0; i< elem->paramCount; i++)
            PRINT(0)("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
          if (elem->description != NULL)
            PRINT(0)("- %s", elem->description);
          PRINT(0)("\n");
          elem = iterator->nextElement();
        }
        delete iterator;

        delete itCL;
        return;
      }
      elemCL = itCL->nextElement();
    }
    delete itCL;
    PRINTF(3)("Class %s not found in Command's classes\n", className);
  }
  else
  {
    PRINTF(1)("List of commandClasses does not exist");
  }
}

