/*!
 * @file shell_command.h
 * Definition of a on-screen-shell
 */

#ifndef _SHELL_COMMAND_H
#define _SHELL_COMMAND_H

#include "base_object.h"

#include "helper_functions.h"
#include "multi_type.h"
#include "substring.h"
#include "functor_list.h"
#include "executor/executor.h"
#include <stdarg.h>

#define     SHELL_COMMAND_MAX_SIZE      //!< The maximum size of a Shell Command



// FORWARD DECLARATION
template<class T> class tList;

/**
 * an easy to use Macro to create a Command
 * @param command the name of the command (without "" around the string)
 * @param class the name of the class to apply this command to (without the "" around the string)
 * @param function the function to call
 *
 * MEANING:
 *  ShellCommand* someUniqueVarName =
 *       ShellCommand<ClassName>::registerCommand("commandNameInShell", "ClassName", &ClassName::FunctionToCall);
 *
 * In the Shell you would call this Command using:
 * $ ClassName [ObjectName] commandNameInShell [parameters]
 */
//#define SHELL_COMMAND(command, class, function) \
//        ShellCommand* shell_command_##class##_##command = ShellCommand<class>::registerCommand(#command, #class, &class::function)
#define SHELL_COMMAND(command, class, function) \
           ShellCommand* shell_command_##class##_##command = ShellCommand::registerCommand(#command, #class, new ExecutorObjective<class>(&class::function))

/**
 * an easy to use Macro to create a Command
 * @param command the name of the command (without "" around the string)
 * @param class the name of the class to apply this command to (without the "" around the string)
 * @param function the function to call
 *
 * MEANING:
 *  ShellCommand* someUniqueVarName =
 *       ShellCommand<ClassName>::registerCommand("commandNameInShell", "ClassName", &ClassName::FunctionToCall);
 *
 * In the Shell you would call this Command using:
 * $ ClassName [ObjectName] commandNameInShell [parameters]
 */
#define SHELL_COMMAND_STATIC(command, class, function) \
           ShellCommand* shell_command_##class##_##command = ShellCommand::registerCommand(#command, #class, new ExecutorStatic<class>(function))


//! an enumerator for the definition of the Type.
typedef enum {
  ShellCommand_Objective = 1,
  ShellCommand_Static    = 2,
} ShellCommand_Type;

////////////////
// BASE CLASS //
////////////////
class ShellCommand;
class ShellCommandAlias;

//! A class to hold all Classes that have (once) registered Commands.
class ShellCommandClass : public BaseObject
{
  friend class ShellCommand;

  public:
    /** @returns the CommandClassList */
    static const tList<ShellCommandClass>* getCommandClassList() { return ShellCommandClass::commandClassList; };
    static bool getCommandListOfClass(const char* className, tList<const char>* stringList);
    static bool getCommandListOfAlias(tList<const char>* aliasList);

    static ShellCommandClass* getCommandClass(const char* className);
    static void unregisterAllCommands();

    static void help (const char* className);

  private:
    ShellCommandClass(const char* className);
    ~ShellCommandClass();

    static const ShellCommandClass* isRegistered(const char* className);
    static void initCommandClassList();

  private:
    const char*                      className;                 //!< The Name of the Class. This should match the ClassName of the Commands Class.
    long                             classID;                   //!< The classID of this Class
    tList<ShellCommand>*         commandList;               //!< A list of Commands from this Class
    static tList<ShellCommandClass>* commandClassList;          //!< A list of Classes
    static tList<ShellCommandAlias>* aliasList;                 //!< An Alias to A Command. (only for classes with one Instance)
};


//! a baseClass for all possible ShellCommands
class ShellCommand : public BaseObject
{
  friend class ShellCommandClass;
  public:
    static bool execute (const char* executionString);

    ShellCommand* describe(const char* description);
    ShellCommand* setAlias(const char* alias);
    ShellCommand* defaultValues(unsigned int count, ...);

    static ShellCommand* registerCommand(const char* commandName, const char* className, Executor* executor);

    static void unregisterCommand(const char* commandName, const char* className);

    static void debug();

  protected:
    ShellCommand(const char* commandName, const char* className, Executor* executor);
    ~ShellCommand();

    /** @returns the Type of this Function (either static or objective) */
    inline ShellCommand_Type getType() { return this->functorType; };

    static bool isRegistered(const char* commandName, const char* className, Executor* executor);
    static const char* paramToString(long parameter);

  protected:
    ShellCommand_Type                functorType;                          //!< The type of Function we've got (either static or objective).
    void*                            functionPointer;                      //!< The pointeer to the function of the Class (or static pointer if ClassID == CL_NULL )
    unsigned int                     paramCount;                           //!< the count of parameters.
    unsigned int*                    parameters;                           //!< Parameters the function of this Command takes.
    MultiType*                       defaultValue;                         //!< Default Values.

  private:
    ShellCommandClass*               shellClass;                           //!< A Pointer to the Shell-Class this Command belongs to.
    ShellCommandAlias*               alias;                                //!< An Alias for the Class.

    const char*                      description;                          //!< A description for this commnand. (initially NULL). Assigned with (create)->describe("blablabla");
    Executor*                        executor;

};

//! A Class, that handles aliases.
class ShellCommandAlias
{
  friend class ShellCommand;
  public:
    /** @returns the Name of the Alias. */
    const char* getName() const { return this->aliasName; };
    /** @returns the Command, this Alias is asociated with */
    ShellCommand* getCommand() const { return this->command; };

  private:
    /** @param aliasName the name of the Alias @param command the Command, to associate this alias with */
    ShellCommandAlias(const char* aliasName, ShellCommand* command) { this->aliasName = aliasName; this->command = command; };

  private:
    const char*         aliasName;       //!< the name of the Alias
    ShellCommand*   command;         //!< a pointer to the command, this alias executes.
};

#endif /* _SHELL_COMMAND_H */
