/*!
 * @file executor.h
 * Definition of a on-screen-shell
 */

#ifndef _EXECUTOR_H
#define _EXECUTOR_H

#include "base_object.h"

#include "helper_functions.h"
#include "multi_type.h"
#include "substring.h"
#include "functor_list.h"

#include <stdarg.h>

// FORWARD DECLARATION
template<class T> class tList;

//! an enumerator for the definition of the Type.
typedef enum {
  Executor_Objective   = 1,
  Executor_Static    = 2,
} Executor_Type;

////////////////
// BASE CLASS //
////////////////
//! a baseClass for all possible Executors
class Executor: public BaseObject
{
  public:
    static bool execute (const char* executionString);

    Executor* describe(const char* description);
    Executor* defaultValues(unsigned int count, ...);

    /** @returns the CommandList of the Shell */
    static void unregisterCommand(const char* commandName, const char* className);

    static void debug();

  protected:
    Executor(const char* commandName, const char* className, unsigned int paramCount, ...);
    ~Executor();

    /** @returns the Type of this Function (either static or objective) */
    inline Executor_Type getType() { return this->functorType; };

    static bool isRegistered(const char* commandName, const char* className, unsigned int paramCount, ...);
    static const char* paramToString(long parameter);

  private:
    /** executes a Command @param object the object to apply this to @param parameters the parameters the command takes */
    virtual void executeCommand (BaseObject* object, const char* parameters) = 0;

  protected:
    Executor_Type                    functorType;                          //!< The type of Function we've got (either static or objective).
    void*                            functionPointer;                      //!< The pointeer to the function of the Class (or static pointer if ClassID == CL_NULL )
    unsigned int                     paramCount;                           //!< the count of parameters.
    unsigned int*                    parameters;                           //!< Parameters the function of this Command takes.
    MultiType*                       defaultValue;                         //!< Default Values.

  private:
    const char*                      description;                          //!< A description for this commnand. (initially NULL). Assigned with (create)->describe("blablabla");
};

///////////////////////////////////////////////////
///////////////////////////////////////////////////

/////////////////////////////////
// MACRO DEFINITION EXTENSIONS //
/////////////////////////////////
//! where to chek for default BOOL values
#define   l_BOOL_DEFGRAB(i)         this->defaultValue[i].getBool()
//! where to chek for default INT values
#define   l_INT_DEFGRAB(i)          this->defaultValue[i].getInt()
//! where to chek for default UINT values
#define   l_UINT_DEFGRAB(i)         (unsigned int)this->defaultValue[i].getInt()
//! where to chek for default LONG values
#define   l_LONG_DEFGRAB(i)         (long)this->defaultValue[i].getInt()
//! where to chek for default FLOAT values
#define   l_FLOAT_DEFGRAB(i)        this->defaultValue[i].getFloat()
//! where to chek for default STRING values
#define   l_STRING_DEFGRAB(i)       this->defaultValue[i].getString()

//////////////////////////
// COMMAND REGISTRATION //
//////////////////////////
// EXECUTOR can be redefined as Executor or ExecutorStatic
// EXECUTOREXECUTER can be redefined too.
// EXECUTORINCLASS
// EXECUTORTYPE
//! registers a command without any parameters
#define ExecutorRegister0() \
  static EXECUTOR<T>* registerCommand(const char* commandName, const char* className, void EXECUTORINCLASS(*function)()) \
  { \
    if (isRegistered(commandName, className, 0)== true) \
      return NULL; \
    return new EXECUTOR<T>(commandName, className, function); \
  }

//! registers a command with 1 parameter
#define ExecutorRegister1(t1) \
  static EXECUTOR<T>* registerCommand(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE)) \
  { \
    if (isRegistered(commandName, className, 1, t1##_PARAM)== true) \
      return NULL; \
    return new EXECUTOR<T>(commandName, className, function); \
  }

//! registers a command with 2 parameters
#define ExecutorRegister2(t1,t2) \
  static EXECUTOR<T>* registerCommand(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE)) \
  { \
    if (isRegistered(commandName, className, 2, t1##_PARAM, t2##_PARAM)== true) \
      return NULL; \
    return new EXECUTOR<T>(commandName, className, function); \
  }

//! registers a command with 3 parameters
#define ExecutorRegister3(t1,t2,t3) \
  static EXECUTOR<T>* registerCommand(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE)) \
  { \
    if (isRegistered(commandName, className, 3, t1##_PARAM, t2##_PARAM, t3##_PARAM)== true) \
      return NULL; \
    return new EXECUTOR<T>(commandName, className, function); \
  }

//! registers a command with 4 parameters
#define ExecutorRegister4(t1,t2,t3,t4) \
  static EXECUTOR<T>* registerCommand(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE)) \
  { \
    if (isRegistered(commandName, className, 4, t1##_PARAM, t2##_PARAM, t3##_PARAM, t4##_PARAM)== true) \
      return NULL; \
    return new EXECUTOR<T>(commandName, className, function); \
  }

//! registers a command with 5 parameters
#define ExecutorRegister5(t1,t2,t3,t4,t5) \
  static EXECUTOR<T>* registerCommand(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE, t5##_TYPE)) \
  { \
    if (isRegistered(commandName, className, 5, t1##_PARAM, t2##_PARAM, t3##_PARAM, t4##_PARAM, t5##_PARAM)== true) \
      return NULL; \
    return new EXECUTOR<T>(commandName, className, function); \
  }

///////////////////////
// FUNCTION POINTERS //
///////////////////////
#define ExecutorFunctionPoiter0() \
  void EXECUTORINCLASS(*functionPointer_0)();

#define ExecutorFunctionPoiter1(t1) \
  void EXECUTORINCLASS(*functionPointer_1_##t1)(t1##_TYPE);

#define ExecutorFunctionPoiter2(t1, t2) \
  void EXECUTORINCLASS(*functionPointer_2_##t1##_##t2)(t1##_TYPE, t2##_TYPE);


#define ExecutorFunctionPoiter3(t1, t2, t3) \
  void EXECUTORINCLASS(*functionPointer_3_##t1##_##t2##_##t3)(t1##_TYPE, t2##_TYPE, t3##_TYPE);

#define ExecutorFunctionPoiter4(t1, t2, t3, t4) \
  void EXECUTORINCLASS(*functionPointer_4_##t1##_##t2##_##t3##_##t4)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE);


#define ExecutorFunctionPoiter5(t1, t2, t3, t4, t5) \
  void EXECUTORINCLASS(*functionPointer_5_##t1##_##t2##_##t3##_##t4##_##t5)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE, t5##_TYPE); \


//////////////////
// CONSTRUCTORS //
/////////////////
//! creates a command that takes no parameters
#define ExecutorConstructor0() \
  EXECUTOR(const char* commandName, const char* className, void EXECUTORINCLASS(*function)()) \
  : Executor(commandName, className, 0) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_0 = function; \
  }

//! creates a command that takes one parameter
#define ExecutorConstructor1(t1) \
  EXECUTOR(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE)) \
  : Executor(commandName, className, 1, t1##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_1_##t1 = function; \
  }

//! creates a command that takes two parameters
#define ExecutorConstructor2(t1,t2) \
  EXECUTOR(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE)) \
  : Executor(commandName, className, 2, t1##_PARAM, t2##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_2_##t1##_##t2 = function; \
  }

//! creates a command that takes three parameter
#define ExecutorConstructor3(t1,t2,t3) \
  EXECUTOR(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE)) \
  : Executor(commandName, className, 3, t1##_PARAM, t2##_PARAM, t3##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_3_##t1##_##t2##_##t3 = function; \
  }

//! creates a command that takes four parameter
#define ExecutorConstructor4(t1,t2,t3,t4) \
  EXECUTOR(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE)) \
  : Executor(commandName, className, 4, t1##_PARAM, t2##_PARAM, t3##_PARAM, t4##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_4_##t1##_##t2##_##t3##_##t4 = function; \
  }

//! creates a command that takes five parameter
#define ExecutorConstructor5(t1,t2,t3,t4,t5) \
  EXECUTOR(const char* commandName, const char* className, void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE, t5##_TYPE)) \
  : Executor(commandName, className, 5, t1##_PARAM, t2##_PARAM, t3##_PARAM, t4##_PARAM, t5##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_5_##t1##_##t2##_##t3##_##t4##_##t5 = function; \
  }

///////////////
// EXECUTION //
///////////////
//! execute-macro for functions with no parameters
#define ExecutorExecute0() \
  if (this->paramCount == 0) \
    EXECUTOREXECUTER(_0)()

//! execute-macro for functions with one parameter
#define ExecutorExecute1(t1) \
   else if (this->paramCount == 1 && this->parameters[0] == t1##_PARAM) \
    EXECUTOREXECUTER(_1_##t1)(t1##_FUNC(parameters, t1##_DEFGRAB(0)))

//! execute-macro for functions with two parameters
#define ExecutorExecute2(t1,t2) \
   else if (this->paramCount == 2 && this->parameters[0] == t1##_PARAM && this->parameters[1] == t2##_PARAM) \
    EXECUTOREXECUTER(_2_##t1##_##t2)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)))

//! execute-macro for functions with three parameters
#define ExecutorExecute3(t1,t2,t3) \
   else if (this->paramCount == 3 && this->parameters[0] == t1##_PARAM && this->parameters[1] == t2##_PARAM && this->parameters[2] == t3##_PARAM) \
    EXECUTOREXECUTER(_3_##t1##_##t2##_##t3)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)), t3##_FUNC(sub.getString(2), t3##_DEFGRAB(2)))

//! execute-macro for functions with four parameters
#define ExecutorExecute4(t1,t2,t3,t4) \
   else if (this->paramCount == 4 && this->parameters[0] == t1##_PARAM && this->parameters[1] == t2##_PARAM && this->parameters[2] == t3##_PARAM && this->parameters[3] == t4##_PARAM) \
    EXECUTOREXECUTER(_4_##t1##_##t2##_##t3##_##t4)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)), t3##_FUNC(sub.getString(2), t3##_DEFGRAB(2)), t4##_FUNC(sub.getString(3), t4##_DEFGRAB(3)))

//! execute-macro for functions with five parameters
#define ExecutorExecute5(t1,t2,t3,t4,t5) \
   else if (this->paramCount == 5 && this->parameters[0] == t1##_PARAM && this->parameters[1] == t2##_PARAM && this->parameters[2] == t3##_PARAM && this->parameters[3] == t4##_PARAM && this->parameters[4] == t5##_PARAM) \
    EXECUTOREXECUTER(_5_##t1##_##t2##_##t3##_##t4##_##t5)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)), t3##_FUNC(sub.getString(2), t3##_DEFGRAB(2)), t4##_FUNC(sub.getString(3), t4##_DEFGRAB(3)), t5##_FUNC(sub.getString(4), t5##_DEFGRAB(4)))





//////////\//////////
// DYNAMIC FUNCTOR //
///////////\/////////
//! keeps information about a Executor
template<class T> class ExecutorObjective : public Executor
{
  public:
#ifdef FUNCTOR_LIST
#undef FUNCTOR_LIST
#endif
#ifdef EXECUTOR
#undef EXECUTOR
#endif
#define EXECUTOR                       ExecutorObjective
#ifdef EXECUTOREXECUTER
#undef EXECUTOREXECUTER
#endif
#define EXECUTOREXECUTER(nameExt)      (dynamic_cast<T*>(object)->*(fp.functionPointer##nameExt))
#ifdef EXECUTORINCLASS
#undef EXECUTORINCLASS
#endif
#define EXECUTORINCLASS(FUNCTION)      (T::FUNCTION)
#ifdef EXECUTORTYPE
#undef EXECUTORTYPE
#endif
#define EXECUTORTYPE                   Executor_Objective
//! FUNCTOR_LIST is the List of command-registerers
#define FUNCTOR_LIST(x) ExecutorRegister ## x
#include "functor_list.h"
#undef FUNCTOR_LIST


  private:
//! FUNCTOR_LIST is the List of FunctionPointers
    union FunctionPointers {
#define FUNCTOR_LIST(x) ExecutorFunctionPoiter ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    } fp;

//! FUNCTOR_LIST is the List of CommandConstructors
#define FUNCTOR_LIST(x) ExecutorConstructor ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

    virtual void executeCommand (BaseObject* object, const char* parameters)
    {
      SubString sub(parameters, true);
//! FUNCTOR_LIST is the List of Executive Functions
#define FUNCTOR_LIST(x) ExecutorExecute ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    }
};



////////////////////
// STATIC FUNCTOR //
////////////////////
//! keeps information about a Executor, that points to a Static Function
template<class T> class ExecutorStatic : public Executor
{
  public:
#ifdef FUNCTOR_LIST
#undef FUNCTOR_LIST
#endif
#ifdef EXECUTOR
#undef EXECUTOR
#endif
#define EXECUTOR                      ExecutorStatic
#ifdef EXECUTOREXECUTER
#undef EXECUTOREXECUTER
#endif
#define EXECUTOREXECUTER(nameExt)     fp.functionPointer##nameExt
#ifdef EXECUTORINCLASS
#undef EXECUTORINCLASS
#endif
#define EXECUTORINCLASS(FUNCTION)     (FUNCTION)
#ifdef EXECUTORTYPE
#undef EXECUTORTYPE
#endif
#define EXECUTORTYPE                   Executor_Static

//! FUNCTOR_LIST is the List of command-registerers
#define FUNCTOR_LIST(x) ExecutorRegister ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

  private:
//! FUNCTOR_LIST is the List of FunctionPointers
    union FunctionPointers {
#define FUNCTOR_LIST(x) ExecutorFunctionPoiter ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    } fp;

//! FUNCTOR_LIST is the List of CommandConstructors
#define FUNCTOR_LIST(x) ExecutorConstructor ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

    virtual void executeCommand (BaseObject* object, const char* parameters)
    {
  SubString sub(parameters, true);
//! FUNCTOR_LIST is the List of Executive Functions
#define FUNCTOR_LIST(x) ExecutorExecute ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    }
};

//! A Class, that handles aliases.
class ExecutorAlias
{
  friend class ExecutorBase;
  public:
    /** @returns the Name of the Alias. */
    const char* getName() const { return this->aliasName; };
    /** @returns the Command, this Alias is asociated with */
    ExecutorBase* getCommand() const { return this->command; };

  private:
    /** @param aliasName the name of the Alias @param command the Command, to associate this alias with */
    ExecutorAlias(const char* aliasName, ExecutorBase* command) { this->aliasName = aliasName; this->command = command; };

  private:
    const char*         aliasName;       //!< the name of the Alias
    ExecutorBase*       command;         //!< a pointer to the command, this alias executes.
};

#endif /* _EXECUTOR_H */
