/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

/*!
 * @file load_param.h
 * A Class and macro-functions, that makes our lives easy to load-in parameters
 */

#ifndef _LOAD_PARAM_H
#define _LOAD_PARAM_H

#include "functor_list.h"

#include "debug.h"

#include "factory.h"
#include "substring.h"
#include "tinyxml.h"

#include "helper_functions.h"

// Forward Declaration //
template<class T> class tList;
class LoadClassDescription;
class LoadParamDescription;

/**************************
**** REAL DECLARATIONS ****
**************************/
//! abstract Base class for a Loadable parameter
class LoadParamBase : public BaseObject
{
 public:
  LoadParamBase* describe(const char* descriptionText);

 protected:
  LoadParamBase(const TiXmlElement* root, BaseObject* object, const char* paramName, int paramCount, bool multi, const void* pointerToParam, ...);

 protected:
  LoadClassDescription*    classDesc;            //!< The LoadClassDescription of this LoadParameter
  LoadParamDescription*    paramDesc;            //!< The LoadParameterDescription of this LoadParameter
  const char*              loadString;           //!< The string loaded by this LoadParam
  const void*              pointerToParam;       //!< A Pointer to a Parameter.
};


//! macro that makes it even more easy to load a Parameter
/**
 * @param className the name of the class to load
 * @param parameterName the name of the parameter to load as written in the XML-file
 * @param function the function to call
 */
#define LOAD_PARAM(className, parameterName, paramFunction) \
        LoadParam<className>(root, #parameterName, this, &className::paramFunction)

/**
 * this Starts a Cycle in the Loading Process
 * be aware, that in the cycle the first parameter of load_param should because
 * called element, and that you must say true at the Fith parameter, or it will fail
 * also you will have to close the Cycle again with LOAD_PARAM_END_CYCLE
 */
#define LOAD_PARAM_START_CYCLE   const TiXmlElement* element; \
                                 element = root->FirstChildElement(); \
                                 while( element != NULL) \
{
/**
   * closes a LoadParam Loop
   * @see LOAD_PARAM_START_CYCLE
 */
#define LOAD_PARAM_END_CYCLE        element = element->NextSiblingElement(); \
}


/*****************************************
**** MACROS DEFINITIONS OF LOADABLES *****
*****************************************/
// 0. TYPES
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with no argument
 */
#define LoadParam0() \
LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(), bool multi = false) \
  : LoadParamBase(root, pt2Object, paramName, 0, multi, NULL, "") \
{ \
  if (loadString != NULL && root != NULL) \
    (*pt2Object.*function)(); \
  else \
    PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName());\
}

// 1. TYPE
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with 1 argument
 * @param type1 The type of the first functionParameter
 */
#define LoadParam1(type1) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT) \
  : LoadParamBase(root, pt2Object, paramName, 1, multi, NULL, type1##_PARAM, default1) \
{ \
      if (loadString != NULL && root != NULL) \
        (*pt2Object.*function)(type1##_FUNC(loadString, default1)); \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}

// 2. TYPES
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with 2 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 *
 * @TODO DEFAULT VALUES HACK
 */
#define LoadParam2(type1, type2) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE, type2##_TYPE), \
           bool multi = false,  type1##_TYPE default1 = type1##_DEFAULT,  type2##_TYPE default2 = type2##_DEFAULT) \
  : LoadParamBase(root, pt2Object, paramName, 2, multi, NULL, type1##_PARAM, default1, type2##_PARAM, default2) \
{ \
      if (loadString != NULL && root != NULL) \
{ \
          SubString subLoads(loadString); \
          if (subLoads.getCount() >= 1) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 2, subLoads.getCount()); \
} \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}


// 3. TYPES
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with 3 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 * @param type3 The type of the third functionParameter
 */
#define LoadParam3(type1, type2, type3) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE, type2##_TYPE, type3##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT, type2##_TYPE default2 = type2##_DEFAULT, type3##_TYPE default3 = type3##_DEFAULT)\
  : LoadParamBase(root, pt2Object, paramName, 3, multi, NULL, type1##_PARAM, default1, type2##_PARAM, default2, type3##_PARAM, default3) \
{ \
      if (loadString != NULL && root != NULL) \
{ \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 3) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2), type3##_FUNC(subLoads.getString(2), default3)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 3, subLoads.getCount()); \
} \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}


// 4. TYPES
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with 4 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 * @param type3 The type of the third functionParameter
 * @param type4 The type of the forth functionParameter
 */
#define LoadParam4(type1, type2, type3, type4) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE, type2##_TYPE, type3##_TYPE, type4##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT, type2##_TYPE default2 = type2##_DEFAULT, type3##_TYPE default3 = type3##_DEFAULT, \
           type4##_TYPE default4 = type4##_DEFAULT) \
  : LoadParamBase(root, pt2Object, paramName, 4, multi, NULL, type1##_PARAM, default1, type2##_PARAM, default2, type3##_PARAM, default3, \
                  type4##_PARAM, default4) \
{ \
      if (loadString != NULL && root != NULL) \
{ \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 4) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2), type3##_FUNC(subLoads.getString(2), default3), type4##_FUNC(subLoads.getString(3), default4)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 4, subLoads.getCount()); \
} \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}


// 5. TYPES
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with 5 arguments
 * @param type1 The type of the first functionParameter
 * @param type2 The type of the second functionParameter
 * @param type3 The type of the third functionParameter
 * @param type4 The type of the forth functionParameter
 * @param type5 The type of the fifth functionParameter
 */
#define LoadParam5(type1, type2, type3, type4, type5) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, \
           void(T::*function)(type1##_TYPE, type2##_TYPE, type3##_TYPE, type4##_TYPE, type5##_TYPE), \
           bool multi = false, type1##_TYPE default1 = type1##_DEFAULT, type2##_TYPE default2 = type2##_DEFAULT, type3##_TYPE default3 = type3##_DEFAULT, \
           type4##_TYPE default4 = type4##_DEFAULT, type5##_TYPE default5 = type5##_DEFAULT ) \
  : LoadParamBase(root, pt2Object, paramName, 5, multi, NULL, type1##_PARAM, default1, type2##_PARAM, default2, type3##_PARAM, default3, \
                  type4##_PARAM, default4, type5##_PARAM, default5) \
{ \
      if (loadString != NULL && root != NULL) \
{ \
          SubString subLoads(loadString); \
          if (subLoads.getCount() == 5) \
            (*pt2Object.*function)(type1##_FUNC(subLoads.getString(0), default1), type2##_FUNC(subLoads.getString(1), default2), type3##_FUNC(subLoads.getString(2), default3), type4##_FUNC(subLoads.getString(3), default4), type5##_FUNC(subLoads.getString(4), default5)); \
          else \
            PRINTF(2)("Not loaded Parameter %s of %s, because wrong count of arguments.\n -> Should have %d but have %d\n", \
                      paramName, pt2Object->getClassName(), 5, subLoads.getCount()); \
} \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}

// Pointer TYPE
/**
 * a Macro to easily implement many different Constructors for the LoadParam-Class with one Pointer argument
 * @param type1 The type of the Pointer
 */
#define LoadParamPT(type1) \
 LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(type1##_TYPE), type1##_TYPE pointerToParam, bool multi = false) \
  : LoadParamBase(root, pt2Object, paramName, 1, multi, pointerToParam, type1##_PARAM) \
{ \
      if (pointerToParam != NULL && root != NULL) \
        (*pt2Object.*function)((type1##_TYPE) pointerToParam); \
      else \
        PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName()); \
}

//! derived template class, so all the Classes can load something.
template<class T> class LoadParam : public LoadParamBase
{
  public:

#define FUNCTOR_LIST(x)    LoadParam ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

  //! makes functions with one Vector loadable
  //LoadParam1(l_VECTOR);

  // loads a Ti-XML-element
  LoadParam(const TiXmlElement* root, const char* paramName, T* pt2Object, void(T::*function)(const TiXmlElement*), bool multi = false)
  : LoadParamBase(root, pt2Object, paramName, 1, multi, NULL, "XML")
  {
    if (root != NULL)
    {
      const TiXmlElement* elem = root->FirstChildElement(paramName);
      if (elem != NULL)
        (*pt2Object.*function)(elem);
      else
        PRINTF(4)("%s of %s is empty\n", paramName, pt2Object->getClassName());
    }
    else
      PRINTF(4)("Not loaded parameter %s of %s\n", paramName, pt2Object->getClassName());
  }

  //LoadParamPT(l_XML_ELEM);
};

// helper function

const char* grabParameter(const TiXmlElement* root, const char* parameterName);

#endif /* _LOAD_PARAM_H */
