/*!
 * @file ini_parser.h
 * A small ini file parser
 *
 * Can be used to find a defined [Section] in an ini file and get the VarName = Value entries
 */

#ifndef _INI_PARSER_H
#define _INI_PARSER_H

#define PARSELINELENGHT     512       //!< how many chars to read at once
#ifndef NULL
#define NULL 0x0                      //!< NULL
#endif

// FORWARD DECLARATION //
template<class T> class tList;

//! ini-file parser
/**
 * This class can be used to load an initializer file and parse it's contents for variablename=value pairs.
 */
class IniParser
{
  private:
    ////////////////////////////////////
    //! a struct for Entries in the Parser's File's Sections
    struct IniEntry
    {
      char*              name;     //!< name of a given Entry
      char*              value;    //!< value of a given Entry
    };
    //! a struct for Sections in the Parser's file
    struct IniSection
    {
      char*               name;    //!< name of a given section
      tList<IniEntry>*    entries; //!< a list of entries for this section
    };
    ////////////////////////////////////

  public:
    IniParser (const char* filename = NULL);
    ~IniParser ();

    bool readFile(const char* fileName);
    bool writeFile(const char* fileName);

    bool addSection(const char* sectionName);
    bool getSection(const char* sectionName);

    void getFirstSection();
    const char* nextSection();

    /** @returns true if the file is opened, false otherwise*/
    bool isOpen() const { return (sections != NULL)?true:false; };
    /** @returns the fileName we have opened. */
    const char* getFileName() const { return this->fileName; };

    bool addVar(const char* entryName, const char* value, const char* sectionName = NULL);
    const char* getVar(const char* entryName, const char* sectionName, const char* defaultValue = "") const;

    void getFirstVar();
    bool nextVar();

    /** @returns the name of the Current selected Section */
    const char* getCurrentSection() const { return (this->currentSection!=NULL)?this->currentSection->name:NULL; };
    /** @returns the current entries Name, or NULL if we havn't selected a Entry */
    const char* getCurrentName() const { return (this->currentEntry!=NULL)?this->currentEntry->name:NULL; };
    /** @returns the current entries Value, or NULL if we havn't selected a Entry */
    const char* getCurrentValue() const { return (this->currentEntry!=NULL)?this->currentEntry->value:NULL; };

    void debug() const;

  private:
    void deleteSections();
    void setFileName(const char* fileName);

  private:
    char*                 fileName;        //!< The name of the File that was parsed.
    tList<IniSection>*    sections;        //!< a list of all stored Sections of the Parser
    IniSection*           currentSection;  //!< the current selected Section
    IniEntry*             currentEntry;    //!< the current selected entry (in currentSection)
};

#endif /* _INI_PARSER_H */
