/*!
 * @file light.h
  *  Handles Lights.

    A Light is one of the more important things in a 3D-environment,
    without it one sees nothing :)
    It is here for diffuse-, specular- and Bump-Mappings.
*/

#ifndef _LIGHT_H
#define _LIGHT_H

#include "p_node.h"
#include "glincl.h"

//! The maximum number of Lights this OpenGL-implementation supports.
#define NUMBEROFLIGHTS GL_MAX_LIGHTS


// FORWARD DECLARATIONS //
class Vector;
class TiXmlElement;

//! A class that handles Lights. The LightManager operates on this.
class Light : public PNode
{
 public:
  Light(const TiXmlElement* root = NULL);
  virtual ~Light();

  void loadParams(const TiXmlElement* root);

  void setDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
  void setSpecularColor(GLfloat r, GLfloat g, GLfloat b);
  void setAttenuation(float constantAttenuation, float linearAttenuation, float quadraticAttenuation);
  void setSpotDirection(const Vector& direction);
  void setSpotDirection(float x, float y, float z) { setSpotDirection(Vector(x,y,z)); };
  void setSpotCutoff(GLfloat cutoff);

  /** @returns the lightNumber*/
  int getLightNumber() const {return this->lightNumber;}

  virtual void draw() const;

  void debug() const;

  // attributes
 private:
  int              lightNumber;               //!< The number of this Light.
  GLfloat          diffuseColor[4];           //!< The Diffuse Color this Light emmits.
  GLfloat          specularColor[4];          //!< The specular Color of this Light.
  float            constantAttenuation;       //!< The Factor of the the Constant Attenuation.
  float            linearAttenuation;         //!< The Factor of the the Linear Attenuation.
  float            quadraticAttenuation;      //!< The Factor of the the Quadratic Attenuation.
  GLfloat          spotDirection[4];          //!< The direction of the Spot Light.
  GLfloat          spotCutoff;                //!< The cutoff Angle of the Light Source
};



//! A class that handles Lights
/**
   A Light is a source that emits light rays (photons)

   <b>Usage:</b>\n
   First you have to get the Light Manager up and running by using LightManager::getInstance().
   This automatically initiates the GL_LIGHTING, and sets some default stuff about the light.\n
   Then you will create a new light using:
   \li new Light();

   if you want to operate on this Light just apply the following functions onto it.
       (You can also optain the Light-pointer with LightManager::getInstance()->getLight(lightNumber))

   now you can operate on the light as follows:
   \li void setDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
   \li void setSpecularColor(GLfloat r, GLfloat g, GLfloat b);
   \li void setAttenuation(float constantAttenuation, float linearAttenuation, float quadraticAttenuation);
   \li void setSpotDirection(Vector direction);
   \li void setSpotCutoff(GLfloat cutoff);
   \li all PNode stuff also works

   To redraw the light use
   \li void draw() const; (this is automatically done by the LightManager)

   for some nice output just use:
   \li void debug() const; (either on LightManager for a resume or on any Light for single information.)
*/
class LightManager : public BaseObject
{
  friend class Light;

 public:
  virtual ~LightManager();
  /** @returns a Pointer to the only object of this Class */
  inline static LightManager* getInstance() { if (!singletonRef) singletonRef = new LightManager();  return singletonRef; };

  void loadParams(const TiXmlElement* root);
  void loadLights(const TiXmlElement* root);

  void setAmbientColor(GLfloat r, GLfloat g, GLfloat b);

  Light* getLight(int lightNumber) const;
  inline Light* getLight() const { return this->currentLight; };

  void draw() const;

  void debug() const;

 private:
  LightManager();

  int  registerLight(Light* light);
  void unregisterLight(Light* light);

 private:
  static LightManager*    singletonRef;       //!< This is the LightHandlers Reference.
  GLfloat                 ambientColor[4];    //!< The ambient Color of the scene.

  Light**                 lights;             //!< An array of Lenght NUMBEROFLIGHTS, that holds pointers to all LightValues.
  Light*                  currentLight;       //!< The current Light, we are working with.

};

#endif /* _LIGHT_H */
