/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "render_2d.h"

#include "graphics_engine.h"
#include "class_list.h"
#include "list.h"
#include "element_2d.h"

#include <math.h>

using namespace std;

/**
 * standard constructor
 */
Render2D::Render2D ()
{
   this->setClassID(CL_RENDER_2D, "Render2D");
   this->setName("Render2D");

   for (int i = 0; i < E2D_LAYER_COUNT; i++)
     this->element2DList[i] = new tList<Element2D>;
}

/**
 *  the singleton reference to this class
 */
Render2D* Render2D::singletonRef = NULL;

/**
 * standard deconstructor
 */
Render2D::~Render2D ()
{
  delete NullElement2D::getInstance();

  for (int i = 0; i < E2D_LAYER_COUNT; i++)
    delete this->element2DList[i];

  Render2D::singletonRef = NULL;
}


/**
 * registers a 2D-element to the 2D-Renderer
 * @param element2D the element to registers
 *
 * do not use this function by yourself, because this is used by Element2D's constructor
 */
void Render2D::registerElement2D(Element2D* element2D)
{
  if (likely(element2D != NULL) && element2D->getLayer() < E2D_LAYER_COUNT && element2D->getLayer() != E2D_LAYER_EXTERN)
    this->element2DList[element2D->getLayer()]->add(element2D);
  // DEBUG printf("::::%p, %d %d \n", element2D, element2D->getLayer(), this->element2DList[element2D->getLayer()]->getSize());
}


/**
 * unregisters a 2D-element from the 2D-Renderer
 * @param element2D The element to unregister
 *
 * do not use this function by yourself, because this is used by Element2D's destructor
 */
void Render2D::unregisterElement2D(Element2D* element2D)
{
  if (likely(element2D != NULL) && element2D->getLayer() < E2D_LAYER_COUNT && element2D->getLayer() != E2D_LAYER_EXTERN)
    this->element2DList[element2D->getLayer()]->remove(element2D);
    // DEBUG  printf(":::%s layer: %d, %d\n", element2D->getClassName(), element2D->getLayer(), this->element2DList[element2D->getLayer()]->getSize());
}


/**
 * moves an 2D-Element to another Layer
 * @param element2D the Element to move
 * @param to which layer to move it to.
 */
void Render2D::moveToLayer(Element2D* element2D, E2D_LAYER to)
{
  if (element2D == NULL)
    return;

  if (E2D_LAYER_COUNT  < to)
    to = E2D_DEFAULT_LAYER;
  if (likely(element2D->getLayer() != to))
  {
    if (element2D->getLayer() != E2D_LAYER_EXTERN)
      this->element2DList[element2D->getLayer()]->removeFromLast(element2D);
    if (to != E2D_LAYER_EXTERN)
      this->element2DList[to]->add(element2D);
  }
}


/**
 * ticks all the 2d-elements
 * @param dt the timestep since last dt
 */
void Render2D::tick(float dt)
{
  for (int i = 0; i < E2D_LAYER_COUNT; i++)
  {
    tIterator<Element2D>* iterator = this->element2DList[i]->getIterator();
    Element2D* elem = iterator->firstElement();
    while (elem != NULL)
    {
      if (elem->isActive())
        elem->tick(dt);
      elem = iterator->nextElement();
    }
    delete iterator;
  }
}

/**
 * renders all the Elements of the Render2D-engine's layer
 * @param layer the Layer to draw (if E2D_LAYER_ALL then all layers will be drawn)
 */
void Render2D::draw(E2D_LAYER layer) const
{
  GraphicsEngine::enter2DMode();

  if (layer != E2D_LAYER_ALL)
  {
    if (likely(layer != E2D_LAYER_EXTERN && this->element2DList[layer]->getSize() > 0))
    {
      tIterator<Element2D>* iterator = this->element2DList[layer]->getIterator();
      Element2D* elem = iterator->firstElement();
      while (elem != NULL)
      {
        if (elem->isVisible())
          elem->draw();
        elem = iterator->nextElement();
      }
      delete iterator;
    }
  }
  else  // if (layer != E2D_LAYER_ALL)
  {
    for (int i = 0; i < E2D_LAYER_COUNT; i++)
    {
      if (this->element2DList[i]->getSize() > 0)
      {
        tIterator<Element2D>* iterator = this->element2DList[i]->getIterator();
        Element2D* elem = iterator->firstElement();
        while (elem != NULL)
        {
          if (elem->isVisible())
            elem->draw();
          elem = iterator->nextElement();
        }
        delete iterator;
      }
    }
  }
  GraphicsEngine::leave2DMode();
}


