/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "element_2d.h"
#include "render_2d.h"

#include "graphics_engine.h"
#include "p_node.h"
#include "load_param.h"
#include "tinyxml.h"
#include "class_list.h"
#include "list.h"
#include "color.h"

using namespace std;

/**
 * standard constructor
 */
Element2D::Element2D()
{
  this->init();
  Render2D::getInstance()->registerElement2D(this);
  this->setParent2D(NullElement2D::getInstance());
}

/**
 * standard constructor
 * @param parent the parent to set for this Element2D
 *
 * NullElement2D needs this constructor with parameter NULL to initialize
 * itself. Otherwise it would result in an endless Loop.
 */
Element2D::Element2D (Element2D* parent)
{
  this->init();
  Render2D::getInstance()->registerElement2D(this);

  // check Parenting, and if ok parent the stuff
  if (this->parent != NULL)
    this->setParent2D(parent);
  else if (NullElement2D::isInstanciated())
    this->setParent2D(NullElement2D::getInstance());
}

/**
 * standard deconstructor
 *
 * There are two general ways to delete an Element2D
 * 1. delete instance;
 *   -> result
 *    delete this Node and all its children and children's children...
 *    (danger if you still want the instance!!)
 *
 * 2. instance->remove2D(); delete instance;
 *   -> result:
 *    moves its children to the NullParent
 *    then deletes the Element.
 */
Element2D::~Element2D ()
{
  // delete what has to be deleted here
  Render2D::getInstance()->unregisterElement2D(this);

  // remove the Node, delete it's children.
  tIterator<Element2D>* iterator = this->children->getIterator();
  Element2D* child = iterator->firstElement();

  while( child != NULL)
  {
    delete child;
    child = iterator->nextElement();
  }
  delete iterator;

  if (this->parent != NULL)
  {
    this->parent->children->remove(this);
    this->parent = NULL;
  }
  delete this->children;

  // remove all other allocated memory.
  if (this->toCoordinate != NULL)
    delete this->toCoordinate;
  if (this->toDirection != NULL)
    delete this->toDirection;
}


/**
 * initializes a Element2D
 */
void Element2D::init()
{
  this->setClassID(CL_ELEMENT_2D, "Element2D");

  this->setVisibility(true);
  this->setActiveness(true);
  this->setAlignment(E2D_ALIGN_NONE);
  this->layer = E2D_TOP;
  this->bindNode = NULL;

  this->setParentMode2D(E2D_PARENT_ALL);
  this->parent = NULL;
  this->children = new tList<Element2D>;
  this->absDirection = 0.0;
  this->relDirection = 0.0;
  this->bRelCoorChanged = true;
  this->bRelDirChanged = true;
  this->toCoordinate = NULL;
  this->toDirection = NULL;
  this->setSize2D(1,1);
}

/**
 * Loads the Parameters of an Element2D from...
 * @param root The XML-element to load from
 */
void Element2D::loadParams(const TiXmlElement* root)
{
  LoadParam<Element2D>(root, "alignment", this, &Element2D::setAlignment)
      .describe("loads the alignment: (either: center, left, right or screen-center)");

  LoadParam<Element2D>(root, "layer", this, &Element2D::setLayer)
      .describe("loads the layer onto which to project: (either: top, medium, bottom, below-all)");

  LoadParam<Element2D>(root, "bind-node", this, &Element2D::setBindNode)
      .describe("sets a node, this 2D-Element should be shown upon (name of the node)");

  LoadParam<Element2D>(root, "visibility", this, &Element2D::setVisibility)
      .describe("if the Element is visible or not");


  // PNode-style:
  LoadParam<Element2D>(root, "rel-coor", this, &Element2D::setRelCoor2D)
      .describe("Sets The relative position of the Node to its parent.");

  LoadParam<Element2D>(root, "abs-coor", this, &Element2D::setAbsCoor2D)
      .describe("Sets The absolute Position of the Node.");

  LoadParam<Element2D>(root, "rel-dir", this, &Element2D::setRelDir2D)
      .describe("Sets The relative rotation of the Node to its parent.");

  LoadParam<Element2D>(root, "abs-dir", this, &Element2D::setAbsDir2D)
      .describe("Sets The absolute rotation of the Node.");

  LoadParam<Element2D>(root, "parent", this, &Element2D::setParent2D)
      .describe("the Name of the Parent of this Element2D");

  LoadParam<Element2D>(root, "parent-mode", this, &Element2D::setParentMode2D)
      .describe("the mode to connect this node to its parent ()");

  // cycling properties
  if (root != NULL)
  {
    const TiXmlElement* element = root->FirstChildElement();
    while (element != NULL)
    {
      LoadParam<Element2D>(root, "parent", this, &Element2D::addChild2D, true)
          .describe("adds a new Child to the current Node.");

      element = element->NextSiblingElement();
    }
  }
}

/**
 * sets the alignment of the 2D-element in form of a String
 * @param alignment the alignment @see loadParams
*/
void Element2D::setAlignment(const char* alignment)
{
  if (!strcmp(alignment, "center"))
    this->setAlignment(E2D_ALIGN_CENTER);
  else if (!strcmp(alignment, "left"))
    this->setAlignment(E2D_ALIGN_LEFT);
  else if (!strcmp(alignment, "right"))
    this->setAlignment(E2D_ALIGN_RIGHT);
  else if (!strcmp(alignment, "screen-center"))
    this->setAlignment(E2D_ALIGN_SCREEN_CENTER);
}


/**
 * moves a Element to another layer
 * @param layer the Layer this is drawn on
 */
void Element2D::setLayer(E2D_LAYER layer)
{
  Render2D::getInstance()->moveToLayer(this, layer);
  this->layer = layer;
}

/**
 * sets the layer onto which this 2D-element is projected to.
 * @param layer the layer @see loadParams
 */
void Element2D::setLayer(const char* layer)
{
  if (!strcmp(layer, "top"))
    this->setLayer(E2D_TOP);
  else if (!strcmp(layer, "medium"))
    this->setLayer(E2D_MEDIUM);
  else if (!strcmp(layer, "bottom"))
    this->setLayer(E2D_BOTTOM);
  else if (!strcmp(layer, "below-all"))
    this->setLayer(E2D_BELOW_ALL);
}


/**
 * sets a node, this 2D-Element should be shown upon
 * @param bindNode the name of the Node (should be existing)
 */
void Element2D::setBindNode(const char* bindNode)
{
  const PNode* tmpBindNode = dynamic_cast<const PNode*>(ClassList::getObject(bindNode, CL_PARENT_NODE));
  if (tmpBindNode != NULL)
    this->bindNode = tmpBindNode;
}

/**
 * sets the relative coordinate of the Element2D to its parent
 * @param relCoord the relative coordinate to the parent
 */
void Element2D::setRelCoor2D (const Vector& relCoord)
{
  if (this->toCoordinate!= NULL)
  {
    delete this->toCoordinate;
    this->toCoordinate = NULL;
  }
  this->relCoordinate = relCoord;
  this->bRelCoorChanged = true;
}


/**
 * sets the relative coordinate of the Element2D to its Parent
 * @param x the x coordinate
 * @param y the y coordinate
 * @param z the z coordinate
 */
void Element2D::setRelCoor2D (float x, float y, float z)
{
  this->setRelCoor2D(Vector(x,y,z));
}

/**
 * sets the Relative coordinate to the parent in Pixels
 * @param x the relCoord X
 * @param y the relCoord Y
 */
void Element2D::setRelCoor2Dpx (int x, int y)
{
  this->setRelCoor2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                     (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                     0
                           ));
}

/**
 * sets a new relative position smoothely
 * @param relCoordSoft the new Position to iterate to
 * @param bias how fast to iterate to this position
 */
void Element2D::setRelCoorSoft2D(const Vector& relCoordSoft, float bias)
{
  if (likely(this->toCoordinate == NULL))
    this->toCoordinate = new Vector();

  *this->toCoordinate = relCoordSoft;
  this->bias = bias;
}

/**
 * sets a new relative position smoothely
 * @param x the new x-coordinate in Pixels of the Position to iterate to
 * @param y the new y-coordinate in Pixels of the Position to iterate to
 * @param bias how fast to iterate to this position
 */
void Element2D::setRelCoorSoft2Dpx (int x, int y, float bias)
{
  this->setRelCoorSoft2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                         (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                         0),
                         bias);
}

/**
 * set relative coordinates smoothely
 * @param x x-relative coordinates to its parent
 * @param y y-relative coordinates to its parent
 * @param z z-relative coordinates to its parent
 * @see  void PNode::setRelCoorSoft (const Vector&, float)
 */
void Element2D::setRelCoorSoft2D(float x, float y, float depth, float bias)
{
  this->setRelCoorSoft2D(Vector(x, y, depth), bias);
}

/**
 * @param absCoord set absolute coordinate
 */
void Element2D::setAbsCoor2D (const Vector& absCoord)
{
  if (this->toCoordinate!= NULL)
  {
    delete this->toCoordinate;
    this->toCoordinate = NULL;
  }

  if( likely(this->parentMode & E2D_PARENT_MOVEMENT))
  {
    /* if you have set the absolute coordinates this overrides all other changes */
    if (likely(this->parent != NULL))
      this->relCoordinate = absCoord - parent->getAbsCoor2D ();
    else
      this->relCoordinate = absCoord;
  }
  if( this->parentMode & E2D_PARENT_ROTATE_MOVEMENT)
  {
    if (likely(this->parent != NULL))
      this->relCoordinate = absCoord - parent->getAbsCoor2D ();
    else
      this->relCoordinate = absCoord;
  }

  this->bRelCoorChanged = true;
}

/**
 * @param x x-coordinate.
 * @param y y-coordinate.
 * @param z z-coordinate.
 * @see void PNode::setAbsCoor (const Vector& absCoord)
 */
void Element2D::setAbsCoor2D (float x, float y, float depth)
{
  this->setAbsCoor2D(Vector(x, y, depth));
}

/**
 * @param x x-coordinate in Pixels
 * @param y y-coordinate in Pixels
 * @see void PNode::setAbsCoor (const Vector& absCoord)
 */
void Element2D::setAbsCoor2Dpx (int x, int y)
{
  this->setAbsCoor2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                     (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                     0
                           ));
}

/**
 *  shift coordinate ralative
 * @param shift shift vector
 *
 * This simply adds the shift-Vector to the relative Coordinate
 */
void Element2D::shiftCoor2D (const Vector& shift)
{
  this->relCoordinate += shift;
  this->bRelCoorChanged = true;

}

/**
 * shifts in PixelSpace
 * @param x the pixels to shift in X
 * @param y the pixels to shift in Y
 */
void Element2D::shiftCoor2Dpx (int x, int y)
{
  this->shiftCoor2D(Vector((float)x/(float)GraphicsEngine::getInstance()->getResolutionX(),
                    (float)y/(float)GraphicsEngine::getInstance()->getResolutionY(),
                     0));
}

/**
 *  set relative direction
 * @param relDir to its parent
 */
void Element2D::setRelDir2D (float relDir)
{
  if (this->toDirection!= NULL)
  {
    delete this->toDirection;
    this->toDirection = NULL;
  }

  this->relDirection = relDir;
  this->bRelDirChanged = true;
}

/**
 * sets the Relative Direction of this node to its parent in a Smoothed way
 * @param relDirSoft the direction to iterate to smoothely.
 * @param bias how fast to iterate to the new Direction
 */
void Element2D::setRelDirSoft2D(float relDirSoft, float bias)
{
  if (likely(this->toDirection == NULL))
    this->toDirection = new float;

  *this->toDirection = relDirSoft;
  this->bias = bias;
}

/**
 *  sets the absolute direction
 * @param absDir absolute coordinates
 */
void Element2D::setAbsDir2D (float absDir)
{
  if (this->toDirection!= NULL)
  {
    delete this->toDirection;
    this->toDirection = NULL;
  }

  if (likely(this->parent != NULL))
    this->relDirection = absDir - this->parent->getAbsDir2D();
  else
    this->relDirection = absDir;

  this->bRelDirChanged = true;
}

/**
 * shift Direction
 * @param shift the direction around which to shift.
 */
void Element2D::shiftDir2D (float shiftDir)
{
  this->relDirection = this->relDirection + shiftDir;
  this->bRelDirChanged = true;
}

/**
 *  adds a child and makes this node to a parent
 * @param child child reference
 * @param parentMode on which changes the child should also change ist state
 *
 * use this to add a child to this node.
 */
void Element2D::addChild2D (Element2D* child, int parentingMode)
{
  if( likely(child->parent != NULL))
  {
    PRINTF(5)("Element2D::addChild() - reparenting node: removing it and adding it again\n");
    child->parent->children->remove(child);
  }
  if (parentingMode != E2D_PARENT_NONE)
    child->parentMode = parentingMode;
  child->parent = this;
  if (likely(this != NULL))
    this->children->add(child);
  child->parentCoorChanged();
}

/**
 * @see Element2D::addChild(Element2D* child);
 * @param childName the name of the child to add to this PNode
 */
void Element2D::addChild2D (const char* childName)
{
  Element2D* childNode = dynamic_cast<Element2D*>(ClassList::getObject(childName, CL_ELEMENT_2D));
  if (childNode != NULL)
    this->addChild2D(childNode);
}

/**
 * removes a child from the node
 * @param child the child to remove from this Node..
 *
 * Children from nodes will not be lost, they are referenced to NullPointer
 */
void Element2D::removeChild2D (Element2D* child)
{
  if (child != NULL)
  {
    child->remove2D();
//    this->children->remove(child);
//    child->parent = NULL;
  }
}

/**
 * remove this Element from the tree and adds all children to NullElement2D
 *
 * afterwards this Node is free, and can be reattached, or deleted freely.
 */
void Element2D::remove2D()
{
  tIterator<Element2D>* iterator = this->children->getIterator();
  Element2D* pn = iterator->firstElement();

  while( pn != NULL)
  {
    NullElement2D::getInstance()->addChild2D(pn, pn->getParentMode2D());
    pn = iterator->nextElement();
  }
  delete iterator;

  delete this->children;
  this->children = new tList<Element2D>;

  if (this->parent != NULL)
  {
    this->parent->children->remove(this);
    this->parent = NULL;
  }
}

/**
 * sets the parent of this Element2D
 * @param parent the Parent to set
 */
void Element2D::setParent2D (Element2D* parent)
{
  parent->addChild2D(this);
}

/**
 * @see Element2D::setParent(Element2D* parent);
 * @param parentName the name of the Parent to set to this Element2D
 */
void Element2D::setParent2D (const char* parentName)
{
  Element2D* parentNode = dynamic_cast<Element2D*>(ClassList::getObject(parentName, CL_ELEMENT_2D));
  if (parentNode != NULL)
    parentNode->addChild2D(this);

}

/**
 * does the reparenting in a very smooth way
 * @param parentNode the new Node to connect this node to.
 * @param bias the speed to iterate to this new Positions
 */
void Element2D::setParentSoft2D(Element2D* parentNode, float bias)
{
  if (this->parent == parentNode)
    return;

  if (likely(this->toCoordinate == NULL))
  {
    this->toCoordinate = new Vector();
    *this->toCoordinate = this->getRelCoor2D();
  }
  if (likely(this->toDirection == NULL))
  {
    this->toDirection = new float;
    *this->toDirection = this->getRelDir2D();
  }
  this->bias = bias;


  Vector tmpV = this->getAbsCoor2D();
  float tmpQ = this->getAbsDir2D();

  parentNode->addChild2D(this);

  if (this->parentMode & PNODE_ROTATE_MOVEMENT) //! @todo implement this.
    ;//this->setRelCoor(this->parent->getAbsDir().inverse().apply(tmpV - this->parent->getAbsCoor()));
  else
    this->relCoordinate = (tmpV - parentNode->getAbsCoor2D());
  this->bRelCoorChanged = true;

  this->relDirection = (tmpQ - parentNode->getAbsDir2D());
  this->bRelDirChanged = true;
}

/**
 * does the reparenting in a very smooth way
 * @param parentName the name of the Parent to reconnect to
 * @param bias the speed to iterate to this new Positions
 */
void Element2D::setParentSoft2D(const char* parentName, float bias)
{
  Element2D* parentNode = dynamic_cast<Element2D*>(ClassList::getObject(parentName, CL_ELEMENT_2D));
  if (parentNode != NULL)
    this->setParentSoft2D(parentNode, bias);
}

/**
 *  sets the mode of this parent manually
 * @param parentMode a String representing this parentingMode
 */
void Element2D::setParentMode2D (const char* parentingMode)
{
  this->setParentMode2D(Element2D::charToParentingMode2D(parentingMode));
}

/**
 *  updates the absCoordinate/absDirection
 * @param dt The time passed since the last update

   this is used to go through the parent-tree to update all the absolute coordinates
   and directions. this update should be done by the engine, so you don't have to
   worry, normaly...
 */
void Element2D::update2D (float dt)
{
  // setting the Position of this 2D-Element.
  if( likely(this->parent != NULL))
  {
      // movement for nodes with smoothMove enabled
    if (unlikely(this->toCoordinate != NULL))
    {
      Vector moveVect = (*this->toCoordinate - this->relCoordinate) *fabsf(dt)*bias;

      if (likely(moveVect.len() >= .001))//PNODE_ITERATION_DELTA))
      {
        this->shiftCoor2D(moveVect);
      }
      else
      {
        Vector tmp = *this->toCoordinate;
        this->setRelCoor2D(tmp);
        PRINTF(5)("SmoothMove of %s finished\n", this->getName());
      }
    }
    if (unlikely(this->toDirection != NULL))
    {
      float rotFlot = (*this->toDirection - this->relDirection) *fabsf(dt)*bias;
      if (likely(fabsf(rotFlot) >= .001))//PNODE_ITERATION_DELTA))
      {
        this->shiftDir2D(rotFlot);
      }
      else
      {
        float tmp = *this->toDirection;
        this->setRelDir2D(tmp);
        PRINTF(5)("SmoothRotate of %s finished\n", this->getName());
      }
    }

    // MAIN UPDATE /////////////////////////////////////
    this->lastAbsCoordinate = this->absCoordinate;

    PRINTF(5)("Element2D::update - %s - (%f, %f, %f)\n", this->getName(), this->absCoordinate.x, this->absCoordinate.y, this->absCoordinate.z);


    if( this->parentMode & E2D_PARENT_LOCAL_ROTATE && this->bRelDirChanged)
    {
      /* update the current absDirection - remember * means rotation around sth.*/
      this->prevRelDirection = this->relDirection;
      this->absDirection = this->relDirection + parent->getAbsDir2D();;
    }


    if (unlikely(this->alignment & E2D_ALIGN_SCREEN_CENTER && this->bRelCoorChanged))
    {
      this->prevRelCoordinate = this->relCoordinate;
      this->absCoordinate.x = .5 + this->relCoordinate.x;
      this->absCoordinate.y = .5 + this->relCoordinate.y;
      this->absCoordinate.z = 0.0;
    }
    else if (unlikely(this->bindNode != NULL))
    {
      GLdouble projectPos[3];
      gluProject(this->bindNode->getAbsCoor().x,
                 this->bindNode->getAbsCoor().y,
                 this->bindNode->getAbsCoor().z,
                 GraphicsEngine::modMat,
                 GraphicsEngine::projMat,
                 GraphicsEngine::viewPort,
                 projectPos,
                 projectPos+1,
                 projectPos+2);
      this->prevRelCoordinate.x = this->absCoordinate.x = projectPos[0] /* /(float)GraphicsEngine::getInstance()->getResolutionX() */ + this->relCoordinate.x;
      this->prevRelCoordinate.y = this->absCoordinate.y = (float)GraphicsEngine::getInstance()->getResolutionY() -  projectPos[1] + this->relCoordinate.y;
      this->prevRelCoordinate.z = this->absCoordinate.z = projectPos[2] + this->relCoordinate.z;
      this->bRelCoorChanged = true;
    }
    else
    {
      if(likely(this->parentMode & PNODE_MOVEMENT && this->bRelCoorChanged))
      {
        /* update the current absCoordinate */
        this->prevRelCoordinate = this->relCoordinate;
        this->absCoordinate = this->parent->getAbsCoor2D() + this->relCoordinate;
      }
      else if( this->parentMode & PNODE_ROTATE_MOVEMENT && this->bRelCoorChanged)
      {
        /* update the current absCoordinate */
        this->prevRelCoordinate = this->relCoordinate;
        float sine = sin(this->parent->getAbsDir2D());
        float cose = cos(this->parent->getAbsDir2D());
//        this->absCoordinate.x = this->relCoordinate.x*cose - this->relCoordinate.y*sine + this->parent->getRelCoor2D().x*(1-cose) +this->parent->getRelCoor2D().y*sine;
//        this->absCoordinate.y = this->relCoordinate.x*sine + this->relCoordinate.y*cose + this->parent->getRelCoor2D().y*(1-cose) +this->parent->getRelCoor2D().x*sine;

        this->absCoordinate.x = this->parent->getAbsCoor2D().x + (this->relCoordinate.x*cos(this->parent->getAbsDir2D()) - this->relCoordinate.y * sin(this->parent->getAbsDir2D()));
        this->absCoordinate.y = this->parent->getAbsCoor2D().y + (this->relCoordinate.x*sin(this->parent->getAbsDir2D()) + this->relCoordinate.y * cos(this->parent->getAbsDir2D()));

      }
    }
    /////////////////////////////////////////////////
  }
  else
  {
    PRINTF(5)("Element2D::update - (%f, %f, %f)\n", this->absCoordinate.x, this->absCoordinate.y, this->absCoordinate.z);
    if (this->bRelCoorChanged)
    {
      this->prevRelCoordinate = this->relCoordinate;
      this->absCoordinate = this->relCoordinate;
    }
    if (this->bRelDirChanged)
    {
      this->prevRelDirection = this->relDirection;
      this->absDirection = this->getAbsDir2D() + this->relDirection;
    }
  }


  // UPDATE CHILDREN
  if(this->children->getSize() > 0)
  {
    tIterator<Element2D>* iterator = this->children->getIterator();
    Element2D* pn = iterator->firstElement();
    while( pn != NULL)
    {
      /* if this node has changed, make sure, that all children are updated also */
      if( likely(this->bRelCoorChanged))
        pn->parentCoorChanged ();
      if( likely(this->bRelDirChanged))
        pn->parentDirChanged ();

      pn->update2D(dt);
      pn = iterator->nextElement();
    }
    delete iterator;
  }

  // FINISHING PROCESS
  this->velocity = (this->absCoordinate - this->lastAbsCoordinate) / dt;
  this->bRelCoorChanged = false;
  this->bRelDirChanged = false;
}

/**
 *  displays some information about this pNode
 * @param depth The deph into which to debug the children of this Element2D to.
 * (0: all children will be debugged, 1: only this Element2D, 2: this and direct children...)
 * @param level The n-th level of the Node we draw (this is internal and only for nice output)
 */
void Element2D::debug (unsigned int depth, unsigned int level) const
{
  for (unsigned int i = 0; i < level; i++)
    PRINT(0)(" |");
  if (this->children->getSize() > 0)
    PRINT(0)(" +");
  else
    PRINT(0)(" -");
  PRINT(0)("Element2D(%s::%s) - absCoord: (%0.2f, %0.2f), relCoord(%0.2f, %0.2f), direction(%0.2f) - %s\n",
  this->getClassName(),
  this->getName(),
  this->absCoordinate.x,
  this->absCoordinate.y,
  this->relCoordinate.x,
  this->relCoordinate.y,
  this->getAbsDir2D(),
  Element2D::parentingModeToChar2D(parentMode));
  if (depth >= 2 || depth == 0)
  {
    tIterator<Element2D>* iterator = this->children->getIterator();
    Element2D* pn = iterator->firstElement();
    while( pn != NULL)
    {
      if (depth == 0)
        pn->debug(0, level + 1);
      else
        pn->debug(depth - 1, level +1);
      pn = iterator->nextElement();
    }
    delete iterator;
  }
}

/**
 * ticks the 2d-Element
 * @param dt the time elapsed since the last tick
 */
void Element2D::tick(float dt)
{

}

/**
 * displays the Element2D at its position with its rotation as a Plane.
 */
void Element2D::debugDraw2D(unsigned int depth, float size, Vector color) const
{

}


// helper functions //
/**
 * converts a parentingMode into a string that is the name of it
 * @param parentingMode the ParentingMode to convert
 * @return the converted string
 */
const char* Element2D::parentingModeToChar2D(int parentingMode)
{
  if (parentingMode == E2D_PARENT_LOCAL_ROTATE)
    return "local-rotate";
  else if (parentingMode == E2D_PARENT_ROTATE_MOVEMENT)
    return "rotate-movement";
  else if (parentingMode == E2D_PARENT_MOVEMENT)
    return "movement";
  else if (parentingMode == E2D_PARENT_ALL)
    return "all";
  else if (parentingMode == E2D_PARENT_ROTATE_AND_MOVE)
    return "rotate-and-move";
}

/**
 * converts a parenting-mode-string into a int
 * @param parentingMode the string naming the parentingMode
 * @return the int corresponding to the named parentingMode
 */
E2D_PARENT_MODE Element2D::charToParentingMode2D(const char* parentingMode)
{
  if (!strcmp(parentingMode, "local-rotate"))
    return (E2D_PARENT_LOCAL_ROTATE);
  else  if (!strcmp(parentingMode, "rotate-movement"))
    return (E2D_PARENT_ROTATE_MOVEMENT);
  else  if (!strcmp(parentingMode, "movement"))
    return (E2D_PARENT_MOVEMENT);
  else  if (!strcmp(parentingMode, "all"))
    return (E2D_PARENT_ALL);
  else  if (!strcmp(parentingMode, "rotate-and-move"))
    return (E2D_PARENT_ROTATE_AND_MOVE);
}




///////////////////
// NullElement2D //
///////////////////
NullElement2D* NullElement2D::singletonRef = 0;

/**
 *  creates the one and only NullElement2D
 * @param absCoordinate the cordinate of the Parent (normally Vector(0,0,0))
 */
NullElement2D::NullElement2D () : Element2D(NULL)
{
  this->setClassID(CL_NULL_ELEMENT_2D, "NullElement2D");
  this->setName("NullElement2D");

  this->setParentMode2D(E2D_PARENT_ALL);
  NullElement2D::singletonRef = this;
}


/**
 *  standard deconstructor
 */
NullElement2D::~NullElement2D ()
{
  NullElement2D::singletonRef = NULL;
}
