/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "player.h"

#include "track_manager.h"
#include "objModel.h"
#include "resource_manager.h"
#include "factory.h"

#include "weapon_manager.h"
#include "test_gun.h"
#include "turret.h"

#include "list.h"

#include "event_handler.h"

#include "event.h"


using namespace std;

CREATE_FACTORY(Player);

/**
 * creates a new Player
 * @param isFree if the player is free
*/
Player::Player()
{
  this->init();

  //weapons:
  Weapon* wpRight = new TestGun(this->weaponMan, 0);
  Weapon* wpLeft = new TestGun(this->weaponMan, 1);

  this->weaponMan->addWeapon(wpRight);
//  this->weaponMan->addWeapon(wpLeft, WM_CONFIG1, WM_SLOT1);
//  this->weaponMan->addWeapon(wpRight, WM_CONFIG2);
//  this->weaponMan->addWeapon(wpLeft, WM_CONFIG2);
}

/**
 * loads a Players information from a specified file.
 * @param fileName the name of the File to load the player from (absolute path)
 */
Player::Player(const char* fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for player.\n", fileName);
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 *  creates a new Player from Xml Data
 * @param root the xml element containing player data

   @todo add more parameters to load
*/
Player::Player(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(this->weaponMan, 0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(this->weaponMan, 1);
  wpLeft->setName("testGun Left");

  Weapon* turret1 = new Turret(this->weaponMan);
  turret1->setName("Turret1");
  turret1->setStateDuration(WS_SHOOTING, .2);
  Weapon* turret2 = new Turret(this->weaponMan);
  turret2->setName("Turret2");
  turret2->setStateDuration(WS_SHOOTING, .3);
  Weapon* turret3 = new Turret(this->weaponMan);
  turret3->setName("Turret3");
  turret3->setStateDuration(WS_SHOOTING, .17);

  Weapon* turret4 = new Turret(this->weaponMan);
  turret4->setName("Turret4");
  turret4->setStateDuration(WS_SHOOTING, .3);


  this->weaponMan->addWeapon(wpLeft, 1, 0);
  this->weaponMan->addWeapon(wpRight,1 ,1);
  this->weaponMan->addWeapon(turret1, 2, 2);
  this->weaponMan->addWeapon(turret2, 2, 3);
  this->weaponMan->addWeapon(turret3, 2, 4);
  this->weaponMan->addWeapon(turret4, 2, 5);

  //this->weaponMan->addWeapon(turret, 3, 0);

  this->weaponMan->changeWeaponConfig(0);
}

/**
 *  destructs the player, deletes alocated memory
 */
Player::~Player ()
{
  /* do not delete the weapons, they are contained in the pnode tree
  and will be deleted there.
  this only frees the memory allocated to save the list.
  */
  delete this->weaponMan;
}

#include "glgui_pushbutton.h"

/**
 * initializes a Player
 */
void Player::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_PLAYER, "Player");

  PRINTF(4)("PLAYER INIT\n");
  travelSpeed = 15.0;
  bUp = bDown = bLeft = bRight = bAscend = bDescend = false;
  bFire = false;
  acceleration = 10.0;

  GLGuiButton* button = new GLGuiPushButton();
  button->show();
  button->setLabel("test");
//  button->setBindNode(this);
  button->setRelCoor2D(500, 20);

  this->weaponMan = new WeaponManager(this);
  this->weaponMan->setSlotCount(6);

  this->weaponMan->setSlotPosition(0, Vector(-2.6, .1, -3.0));

  this->weaponMan->setSlotPosition(1, Vector(-2.6, .1, 3.0));

  this->weaponMan->setSlotPosition(2, Vector(-1.5, .5, -.5));
  this->weaponMan->setSlotDirection(2, Quaternion(-M_PI_4*.5, Vector(1,0,0)));

  this->weaponMan->setSlotPosition(3, Vector(-1.5, .5, .5));
  this->weaponMan->setSlotDirection(3, Quaternion(M_PI_4*.5, Vector(1,0,0)));

  this->weaponMan->setSlotPosition(4, Vector(-1.5, -.5, .5));
  this->weaponMan->setSlotDirection(4, Quaternion(-M_PI_4*.5+M_PI, Vector(1,0,0)));

  this->weaponMan->setSlotPosition(5, Vector(-1.5, -.5, -.5));
  this->weaponMan->setSlotDirection(5, Quaternion(+M_PI_4*.5-M_PI, Vector(1,0,0)));

}


/**
 * loads the Settings of a Player from an XML-element.
 * @param root the XML-element to load the Player's properties from
 */
void Player::loadParams(const TiXmlElement* root)
{
  static_cast<WorldEntity*>(this)->loadParams(root);



}

/**
 * adds a weapon to the weapon list of player
 * @param weapon to add
*/
void Player::addWeapon(Weapon* weapon)
{
  this->weaponMan->addWeapon(weapon);
}


/**
 *  removes a weapon from the player
 * @param weapon to remove
*/
void Player::removeWeapon(Weapon* weapon)
{
  this->weaponMan->removeWeapon(weapon);
}


/**
 *  effect that occurs after the player is spawned
*/
void Player::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}


/**
 *  the action occuring if the player left the game
*/
void Player::leftWorld ()
{}



/**
 *  if the player is hit, call this function
 * @param weapon hit by this weapon
 * @param loc ??
*/
void Player::hit (WorldEntity* weapon, Vector* loc)
{
}


/**
  *  Collision with another Entity has this effect
  * @param other the other colider
  * @param ownhitflags ??
  * @param otherhitflags ??
*/
void Player::collide (WorldEntity* other, Uint32 ownhitflags, Uint32 otherhitflags)
{
}


/**
 *  draws the player after transforming him.
*/
void Player::draw ()
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  /* rotate */
  Vector tmpRot = this->getAbsDir().getSpacialAxis();
  glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->model->draw();
  glPopMatrix();

  this->weaponMan->draw();

  //this->debug(0);
}


/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void Player::tick (float time)
{
  // player controlled movement
  this->move(time);

  this->weaponMan->tick(time);
  // weapon system manipulation
  this->weaponAction();
}


/**
 *  action if player moves
 * @param time the timeslice since the last frame
*/
void Player::move (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */
  Vector direction (1.0, 0.0, 0.0);
  //direction = this->absDirection.apply (direction);
  Vector orthDirection (0.0, 0.0, 1.0);
  //orthDirection = orthDirection.cross (direction);

  if( this->bUp && this->getRelCoor().x < 20)
    accel = accel+(direction*acceleration);
  if( this->bDown && this->getRelCoor().x > -5)
    accel = accel -(direction*acceleration);
  if( this->bLeft &&  TrackManager::getInstance()->getWidth() > -this->getRelCoor().z*2)
    accel = accel - (orthDirection*acceleration);
  if( this->bRight &&  TrackManager::getInstance()->getWidth() > this->getRelCoor().z*2)
    accel = accel + (orthDirection*acceleration);
  if( this->bAscend ) { /* FIXME */ }
  if( this->bDescend) {/* FIXME */} /* @todo up and down player movement */

  Vector move = accel * time;
  this->shiftCoor (move);
}


/**
 * weapon manipulation by the player
*/
void Player::weaponAction()
{
  if( this->bFire)
    {
      this->weaponMan->fire();
    }
}

/**
 * @todo switch statement ??
 */
void Player::process(const Event &event)
{
  if( event.type == KeyMapper::PEV_UP)
      this->bUp = event.bPressed;
  else if( event.type == KeyMapper::PEV_DOWN)
      this->bDown = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
      this->bRight= event.bPressed;
  else if( event.type == KeyMapper::PEV_LEFT)
      this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_FIRE1)
      this->bFire = event.bPressed;
  else if( event.type == KeyMapper::PEV_NEXT_WEAPON && event.bPressed)
    this->weaponMan->nextWeaponConfig();//if( !event.bPressed) this->bWeaponChange = !this->bWeaponChange;
  else if ( event.type == KeyMapper::PEV_PREVIOUS_WEAPON && event.bPressed)
    this->weaponMan->previousWeaponConfig();
}
