/*!
 * @file element_2d.h
 * Definition of the 2D elements rendered on top through the GraphicsEngine
 *
 * @todo reimplement it, so it looks just like PNode.
*/

#ifndef _ELEMENT_2D_H
#define _ELEMENT_2D_H

#include "base_object.h"

#include "vector.h"

// FORWARD DECLARATION
class PNode;
class TiXmlElement;
template<class T> class tList;

//!< An enumerator defining the Depth of a 2D-element.
typedef enum
{
  E2D_BELOW_ALL                 =     1,        //!< Will be rendered below the 3D-scene. @todo make this work.
  E2D_BOTTOM                    =     2,        //!< Will be rendered on the bottom Layer
  E2D_MEDIUM                    =     4,        //!< Will be rendered on the medium Layer.
  E2D_TOP                       =     8,        //!< Will be rendered on top of everything else

  E2D_LAYER_COUNT               =     4         //!< The count of Layers.
} E2D_LAYER;
#define E2D_DEFAULT_LAYER       E2D_TOP
#define E2D_ALL_LAYERS          E2D_TOP | E2D_MEDIUM | E2D_BOTTOM | E2D_BELOW_ALL

typedef enum
{
  E2D_ALIGN_NONE                =     0,
  E2D_ALIGN_LEFT                =     1,
  E2D_ALIGN_RIGHT               =     2,
  E2D_ALIGN_CENTER              =     4,
  E2D_ALIGN_SCREEN_CENTER       =     8
} E2D_ALIGNMENT;

typedef enum
{
  E2D_PARENT_NONE               =   0,
  E2D_PARENT_LOCAL_ROTATE       =   1,    //!< Rotates all the children around their centers.
  E2D_PARENT_ROTATE_MOVEMENT    =   2,    //!< Moves all the children around the center of their parent, without the rotation around their own centers.

  E2D_PARENT_MOVEMENT           =   4,    //!< Moves all children along with the parent.
// special linkage modes
  E2D_PARENT_ALL                =   3,    //!< Moves all children around the center of their parent, and also rotates their centers
  E2D_PARENT_ROTATE_AND_MOVE    =   5     //!< Rotates all children around their axis, and moves them as the Parent Moves, but does not rotate around the center of their parent.
} E2D_PARENT_MODE;
#define     E2D_DEFAULT_PARENTING_MODE  E2D_PARENT_ALL

//! A Struct defining the Position of an Element in 2D-space
struct Position2D
{
  float       x;                 //!< The x-coordinate.
  float       y;                 //!< The y-coordinate.
  float       depth;             //!< The distance from the viewing plane.
};

//! A class for ...
class Element2D : virtual public BaseObject {

  public:
    Element2D();
    Element2D(Element2D* parent);
    virtual ~Element2D();

    void init();
    void loadParams(const TiXmlElement* root);

    /** @param alignment the new Alignment of the 2D-Element */
    inline void setAlignment(E2D_ALIGNMENT alignment) { this->alignment = alignment; };
    void setAlignment(const char* alignment);
    inline E2D_ALIGNMENT getAlignment() const { return this->alignment; };

    void setLayer(E2D_LAYER layer);
    void setLayer(const char* layer);
    /** @returns the Layer this Element is drawn to */
    inline E2D_LAYER getLayer() { return this->layer; };

    /** @param visible true if the Element should be visible false otherwise (will not be rendered) */
    inline void setVisibility(bool visible) { this->visible = visible; };
    /** @param active true if the Element should be active, false otherwise (will not be ticked) */
    inline void setActiveness(bool active) { this->active = active; };


    /** @param bindNode the Node this 2D-element should follow. if NULL the Element will not follow anything */
    inline void setBindNode(const PNode* bindNode) { this->bindNode = bindNode; };
    void setBindNode(const char* bindNode);
    inline const PNode* getBindNode() const { return this->bindNode; };

    /** @returns the visibility state */
    inline bool isVisible() { return this->visible; };
    /** @returns the active-State */
    inline bool isActive() { return this->active; };


    inline void setSize2D(float x, float y) { this->sizeX = x, this->sizeY = y; };
    inline void setSizeX2D(float x) { this->sizeX = x; };
    inline void setSizeY2D(float y) { this->sizeY = y; };
    inline float getSizeX2D() const { return this->sizeX; };
    inline float getSizeY2D() const { return this->sizeY; };

    // LIKE PNODE
  public:
    void setRelCoor2D (const Vector& relCoord);
    void setRelCoor2D (float x, float y, float dontCare = 1.0);
    void setRelCoor2Dpx (int x, int y);
    void setRelCoorSoft2D (const Vector& relCoordSoft, float bias = 1.0);
    void setRelCoorSoft2D (float x, float y, float dontCare = 1.0, float bias = 1.0);
    void setRelCoorSoft2Dpx (int x, int y, float bias = 1.0);
    /** @returns the relative position */
    inline const Vector& getRelCoor2D () const { return this->prevRelCoordinate; };
    /** @returns the Relative Coordinate Destination */
    inline const Vector& getRelCoorSoft2D() const { return (this->toCoordinate)?*this->toCoordinate:this->relCoordinate; };
    const Vector& getRelCoor2Dpx() const;
    void setAbsCoor2D (const Vector& absCoord);
    void setAbsCoor2D (float x, float y, float depth = 1.0);
    void setAbsCoor2Dpx (int x, int y);
    /** @returns the absolute position */
    inline const Vector& getAbsCoor2D () const { return this->absCoordinate; };
    const Vector& getAbsCoor2Dpx () const;

    void shiftCoor2D (const Vector& shift);
    void shiftCoor2Dpx (int x, int y);

    void setRelDir2D (float relDir);
    void setRelDirSoft2D(float relDirSoft, float bias = 1.0);
    /** @returns the relative Direction */
    inline float getRelDir2D () const { return this->prevRelDirection; };
    /** @returns the Relative Directional Destination */
    inline float getRelDirSoft2D() const { return (this->toDirection)?*this->toDirection:this->relDirection; };
    void setAbsDir2D (float absDir);
    /** @returns the absolute Direction */
    inline float getAbsDir2D () const { return this->absDirection; };
    void shiftDir2D (float shiftDir);

    /** @returns the Speed of the Node */
    inline float getSpeed() const { return 0; };
    /** @returns the Velocity of the Node */
    inline const Vector& getVelocity() const { return this->velocity; };


    void addChild2D (Element2D* child, int parentingMode = E2D_PARENT_NONE);
    void addChild2D (const char* childName);
    void removeChild2D (Element2D* child);
    void remove2D();

    void setParent2D (Element2D* parent);
    void setParent2D (const char* parentName);
    /** @returns the parent of this Element2D */
    inline Element2D* getParent () const { return this->parent; };
    /** @returns the List of Children of this Element2D */
    inline const tList<Element2D>* getChildren2D() const { return this->children; };

    void setParentSoft2D(Element2D* parentNode, float bias = 1.0);
    void setParentSoft2D(const char* parentName, float bias = 1.0);

    /** @param parentMode sets the parentingMode of this Node */
    void setParentMode2D (E2D_PARENT_MODE parentMode) { this->parentMode = parentMode; };
    void setParentMode2D (const char* parentingMode);
    /** @returns the Parenting mode of this node */
    int getParentMode2D() const { return this->parentMode; };

    void update2D (float dt);

    void debug (unsigned int depth = 1, unsigned int level = 0) const;
    void debugDraw2D(unsigned int depth = 1, float size = 1.0, Vector color = Vector(1,1,1)) const;

    // helper functions //
    static const char* parentingModeToChar2D(int parentingMode);
    static E2D_PARENT_MODE charToParentingMode2D(const char* parentingMode);

  private:
    /** tells the child that the parent's Coordinate has changed */
    inline void parentCoorChanged () { this->bRelCoorChanged = true; }
    /** tells the child that the parent's Direction has changed */
    inline void parentDirChanged () { this->bRelDirChanged = true; }
    /** @returns the last calculated coordinate */
    inline Vector getLastAbsCoor() { return this->lastAbsCoordinate; }

  public:
    virtual void tick(float dt);
    virtual void draw() const = 0;

  private:
    const PNode*            bindNode;           //!< a node over which to display this 2D-element
    float                   sizeX;              //!< The size of the rendered item in x-direction
    float                   sizeY;              //!< The size of the rendered Item in y-direction

    E2D_ALIGNMENT           alignment;          //!< How the Element is aligned around its Position

    bool                    visible;            //!< If the given Element2D is visible.
    bool                    active;             //!< If the given Element2D is active.
    E2D_LAYER               layer;              //!< What layer this Element2D is on.

    bool                    bRelCoorChanged;    //!< If Relative Coordinate has changed since last time we checked
    bool                    bRelDirChanged;     //!< If Relative Direction has changed since last time we checked

    Vector                  relCoordinate;      //!< coordinates relative to the parent
    Vector                  absCoordinate;      //!< absolute coordinates in the world ( from (0,0,0) )
    float                   relDirection;       //!< direction relative to the parent
    float                   absDirection;       //!< absolute diretion in the world ( from (0,0,1) )

    Vector                  prevRelCoordinate;  //!< The last Relative Coordinate from the last update-Cycle.
    Vector                  lastAbsCoordinate;  //!< this is used for speedcalculation, it stores the last coordinate
    float                   prevRelDirection;   //!< The last Relative Direciton from the last update-Cycle.

    Vector                  velocity;           //!< Saves the velocity.

    Vector*                 toCoordinate;       //!< a position to which to iterate. (This is used in conjunction with setParentSoft.and set*CoorSoft)
    float*                  toDirection;        //!< a direction to which to iterate. (This is used in conjunction with setParentSoft and set*DirSoft)
    float                   bias;               //!< how fast to iterate to the given position (default is 1)

    Element2D*              parent;             //!< a pointer to the parent node
    tList<Element2D>*       children;           //!< list of the children of this Element2D

    unsigned int            parentMode;         //!< the mode of the binding
};

//! The top joint of all Element2D's every Element2D is somehow connected to this one.
class NullElement2D : public Element2D {

  public:
    /** @returns a Pointer to the only object of this Class */
    inline static NullElement2D* getInstance() { if (!NullElement2D::singletonRef) NullElement2D::singletonRef = new NullElement2D();  return NullElement2D::singletonRef; };
    inline static bool isInstanciated() { return (NullElement2D::singletonRef != NULL)?true:false; };
    virtual ~NullElement2D ();

  private:
    NullElement2D ();
    virtual void draw() const {};

  private:
    static NullElement2D* singletonRef;        //!< A reference to the NullElement2D

};

#endif /* _ELEMENT_2D_H */
